﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_Vector.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Arithmetic.h>
#include <nn/nn_Log.h>
#include "Sound3D.h"

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

namespace
{
    void CalcListenerMatrix(nn::util::Matrix4x3fType* mtx)
    {
        // リスナーの位置
        nn::util::Vector3fType pos;
        nn::util::VectorSet( &pos, 0.0f, 0.0f, -3.0f );

        // リスナーの Up 方向のベクトル
        nn::util::Vector3fType upVec;
        nn::util::VectorSet( &upVec, 0.0f, 1.0f, 0.0f );

        // リスナーの方向ベクトル
        nn::util::Vector3fType direction;
        float directionSin;
        float directionCos;
        nn::util::SinCosEst( &directionSin, &directionCos, 0.0f );
        nn::util::VectorSet( &direction, -directionSin, 0.0f, -directionCos );

        // リスナー行列生成
        nn::util::Vector3fType target;
        nn::util::VectorAdd( &target, pos, direction );
        nn::util::MatrixLookAtRightHanded( mtx, pos, target, upVec );
    }
    void PlaySound(nn::atk::Sound3DActor& actor, CommonObject& obj, nn::atk::SoundArchive::ItemId soundId, const char* label)
    {
        nn::atk::SoundHandle& soundHandle = obj.GetSoundHandle();
        soundHandle.Stop( 6 );

        bool result = actor.StartSound( &soundHandle, soundId ).IsSuccess();
        NN_LOG("PlaySound(%s) ... (%d)\n", label, result);
    }
    const char* GetFilterTypeName(int type)
    {
        const char* filterTypeString[]=
        {
            "NONE",
            "LPF"
        };
        if ( type < 0 || type >= sizeof(filterTypeString) / sizeof(filterTypeString[0]) )
        {
            return "INVALID";
        }
        return filterTypeString[type];
    }
    int ChangeFilterType(int currentType)
    {
        if( currentType == nn::atk::BiquadFilterType_None )
        {
            return nn::atk::BiquadFilterType_LowPassFilter;
        }
        else
        {
            return nn::atk::BiquadFilterType_None;
        }
    }

    FlagList g_LocalFlagList(nullptr, 0);
}

void Sound3DCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    m_CommonObject.Initialize();
    m_FilterType = nn::atk::BiquadFilterType_None;

    nn::atk::SoundArchive& soundArchive = m_CommonObject.GetSoundArchive();
    nn::atk::SoundArchivePlayer& soundArchivePlayer = m_CommonObject.GetSoundArchivePlayer();

    //  Sound3DManager の初期化
    {
        const std::size_t memSize = m_Sound3DManager.GetRequiredMemSize( &soundArchive );
        m_pMemoryForSound3DManager = nns::atk::Allocate( memSize );
        NN_ABORT_UNLESS_NOT_NULL( m_pMemoryForSound3DManager );

        const bool isSuccess = m_Sound3DManager.Initialize( &soundArchive, m_pMemoryForSound3DManager, memSize );
        NN_ABORT_UNLESS( isSuccess, "failed to initialize Sound3DManager.\n" );
        m_Sound3DManager.SetBiquadFilterType( m_FilterType );
        m_Sound3DManager.SetMaxPriorityReduction( 32 );
        m_Sound3DManager.SetSonicVelocity( 340.0f / 60.0f );
    }

    //  Sound3DListener の初期化
    {
        nn::util::Matrix4x3fType listenerMatrix;
        nn::util::Vector3fType zeroVelocity;
        CalcListenerMatrix( &listenerMatrix );
        nn::util::VectorSet( &zeroVelocity, 0.0f, 0.0f, 0.0f );

        m_Sound3DListener.SetMatrix( listenerMatrix );
        m_Sound3DListener.SetVelocity( zeroVelocity );
        m_Sound3DListener.SetMaxVolumeDistance( 5.0f );
        m_Sound3DListener.SetUnitDistance( 5.0f );
        m_Sound3DListener.SetInteriorSize( 5.0f );
        m_Sound3DManager.AddListener( &m_Sound3DListener );
    }

    // Sound3DActor の初期化
    {
        nn::util::Vector3fType zeroVelocity;
        nn::util::VectorSet( &zeroVelocity, 0.0f, 0.0f, 0.0f );

        m_Sound3DActor.Initialize( &soundArchivePlayer, &m_Sound3DManager );
        nn::util::VectorSet( &m_ActorPos, 0.0f, 0.0f, 0.0f );
        m_Sound3DActor.SetPosition( m_ActorPos );
        m_Sound3DActor.SetVelocity( zeroVelocity );
    }
}

void Sound3DCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    //  Sound3DManager は SoundArchivePlayer よりも後に終了処理されるべきですが、
    //    m_CommonObject.Finalize() の複雑さと今後の拡張のことを考えると、
    //    Sound3DCheckModule で独自に終了処理を行うのはメンテナンスコストが高いと考えられます。
    //  そこで終了処理の前に鳴っているサウンドを Stop することで
    //    AtkSandbox では Sound3DManager を先に終了処理するようにします。
    {
        auto& soundHandle = m_CommonObject.GetSoundHandle();
        soundHandle.Stop( 0 );
    }

    m_Sound3DActor.Finalize();
    m_Sound3DManager.Finalize();
    nns::atk::Free( m_pMemoryForSound3DManager );

    m_CommonObject.Finalize();
}

void Sound3DCheckModule::OnLoadData() NN_NOEXCEPT
{
    bool isSuccess;
    nn::atk::SoundDataManager& soundDataManager = m_CommonObject.GetSoundDataManager();
    nn::atk::SoundHeap& soundHeap = m_CommonObject.GetSoundHeap();

    isSuccess = soundDataManager.LoadData( SE_IDLE, &soundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_IDLE) failed." );
    isSuccess = soundDataManager.LoadData( SE_IDLE_DOPPLER, &soundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_IDLE_DOPPLER) failed." );
    isSuccess = soundDataManager.LoadData( SE_IDLE_FILTER, &soundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_IDLE_FILTER) failed." );
    isSuccess = soundDataManager.LoadData( SE_IDLE_DOPPLER_FILTER, &soundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_IDLE_DOPPLER_FILTER) failed." );
}

void Sound3DCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    NN_LOG("[A]              Play Sound (Doppler ON , Filter ON )\n");
    NN_LOG("[B]              Play Sound (Doppler ON , Filter OFF)\n");
    NN_LOG("[X]              Play Sound (Doppler OFF, Filter ON )\n");
    NN_LOG("[Y]              Play Sound (Doppler OFF, Filter OFF)\n");
    NN_LOG("[R+A]            Change Filter Type\n");
    NN_LOG("[R+B]            Stop Sound\n");
    NN_LOG("[RIGHT/LEFT]     Move Actor Position.x\n");
    NN_LOG("[UP/DOWN]        Move Actor Position.z\n");
}

void Sound3DCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::R >() )
    {
        // FilterType の変更
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
        {
            m_FilterType = ChangeFilterType( m_FilterType );
            m_Sound3DManager.SetBiquadFilterType( m_FilterType );
            PrintStatus();
        }

        // StopSound
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
        {
            nn::atk::SoundHandle& soundHandle = m_CommonObject.GetSoundHandle();
            soundHandle.Stop( 6 );
        }
    }
    else
    {
        // PlaySound
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
        {
            PlaySound( m_Sound3DActor, m_CommonObject, SE_IDLE_DOPPLER_FILTER, "SE_IDLE_DOPPLER_FILTER");
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
        {
            PlaySound( m_Sound3DActor, m_CommonObject, SE_IDLE_DOPPLER, "SE_IDLE_DOPPLER");
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
        {
            PlaySound( m_Sound3DActor, m_CommonObject, SE_IDLE_FILTER, "SE_IDLE_FILTER");
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >() )
        {
            PlaySound( m_Sound3DActor, m_CommonObject, SE_IDLE, "SE_IDLE");
        }
    }

    // Sound3DActor の座標変更
    static const float ControlPadStep = 0.2f;
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Left >() )
    {
        MoveActor( -ControlPadStep, 0.0f, 0.0f );
    }
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Right >() )
    {
        MoveActor(  ControlPadStep, 0.0f, 0.0f );
    }
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Up >() )
    {
        MoveActor( 0.0f, 0.0f, -ControlPadStep );
    }
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Down >() )
    {
        MoveActor( 0.0f, 0.0f,  ControlPadStep );
    }
}

void Sound3DCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    m_Sound3DActor.SetPosition( m_ActorPos );
    m_CommonObject.Update();
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void Sound3DCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
}
#endif

FlagList& Sound3DCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}

void Sound3DCheckModule::PrintStatus() NN_NOEXCEPT
{
    nn::util::Float3 actorPos;
    nn::util::VectorStore(&actorPos, m_ActorPos);

    NN_LOG("ActorPos( %4.1f, %4.1f, %4.1f ), FilterType= %s\n", actorPos.v[0], actorPos.v[1], actorPos.v[2], GetFilterTypeName(m_FilterType));

    const nn::atk::SoundParam* pParam = m_CommonObject.GetSoundHandle().GetAmbientParam();
    if( pParam != nullptr )
    {
        const auto& tvParam = pParam->GetTvParam();
        NN_LOG( "Pan= %.2f  SPan= %.2f  Volume= %.2f  Pitch= %.2f\n", tvParam.GetPan(), tvParam.GetSurroundPan(), tvParam.GetVolume(), pParam->GetPitch() );
    }
}

void Sound3DCheckModule::MoveActor(float x, float y, float z) NN_NOEXCEPT
{
    nn::util::Vector3fType movement;
    nn::util::VectorSet( &movement, x, y, z );
    nn::util::VectorAdd( &m_ActorPos, m_ActorPos, movement );
    PrintStatus();
}
