﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Filter.h"

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

namespace
{
    enum FilterType
    {
        FilterType_None,
        FilterType_HandleLowPassFilter,
        FilterType_HandleBiquadLowPassFilter,
        FilterType_HandleBiquadHighPassFilter,
        FilterType_HandleBiquadBandPassFilter512,
        FilterType_HandleBiquadBandPassFilter1024,
        FilterType_HandleBiquadBandPassFilter2048,
        FilterType_HandleBiquadLowPassFilterNw4fCompatible48k,
        FilterType_HandleBiquadHighPassFilterNw4fCompatible48k,
        FilterType_HandleBiquadBandPassFilter512Nw4fCompatible48k,
        FilterType_HandleBiquadBandPassFilter1024Nw4fCompatible48k,
        FilterType_HandleBiquadBandPassFilter2048Nw4fCompatible48k,
        FilterType_PlayerLowPassFilter,
        FilterType_PlayerBiquadLowPassFilter,
        FilterType_PlayerBiquadHighPassFilter,
        FilterType_PlayerBiquadBandPassFilter512,
        FilterType_PlayerBiquadBandPassFilter1024,
        FilterType_PlayerBiquadBandPassFilter2048,
        FilterType_PlayerBiquadLowPassFilterNw4fCompatible48k,
        FilterType_PlayerBiquadHighPassFilterNw4fCompatible48k,
        FilterType_PlayerBiquadBandPassFilter512Nw4fCompatible48k,
        FilterType_PlayerBiquadBandPassFilter1024Nw4fCompatible48k,
        FilterType_PlayerBiquadBandPassFilter2048Nw4fCompatible48k,
        FilterType_Max = FilterType_PlayerBiquadBandPassFilter2048Nw4fCompatible48k
    };

    const char* FilterType_String[] =
    {
        "NONE",
        "HANDLE_LPF",
        "HANDLE_BIQUAD_LPF",
        "HANDLE_BIQUAD_HPF",
        "HANDLE_BIQUAD_BPF_512",
        "HANDLE_BIQUAD_BPF_1024",
        "HANDLE_BIQUAD_BPF_2048",
        "HANDLE_BIQUAD_LPF_COMPAT48K",
        "HANDLE_BIQUAD_HPF_COMPAT48K",
        "HANDLE_BIQUAD_BPF_512_COMPAT48K",
        "HANDLE_BIQUAD_BPF_1024_COMPAT48K",
        "HANDLE_BIQUAD_BPF_2048_COMPAT48K",
        "PLAYER_LPF",
        "PLAYER_BIQUAD_LPF",
        "PLAYER_BIQUAD_HPF",
        "PLAYER_BIQUAD_BPF_512",
        "PLAYER_BIQUAD_BPF_1024",
        "PLAYER_BIQUAD_BPF_2048",
        "PLAYER_BIQUAD_LPF_48KCOMPAT",
        "PLAYER_BIQUAD_HPF_48KCOMPAT",
        "PLAYER_BIQUAD_BPF_512_48KCOMPAT",
        "PLAYER_BIQUAD_BPF_1024_48KCOMPAT",
        "PLAYER_BIQUAD_BPF_2048_48KCOMPAT",
    };

    const char* GetFilterTypeString(int type)
    {
        if (type > FilterType_Max || type < 0)
        {
            return "INVALID";
        }
        return FilterType_String[type];
    }

    FlagList g_LocalFlagList(nullptr, 0);
}

void FilterCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    CommonObject::InitializeParam param;
    param.GetSoundSystemParam().rendererSampleRate = ConvertSampleRateTypeToInt(static_cast<SampleRateType>(m_SampleRateType));
    m_CommonObject.Initialize(param);
    NN_LOG("rendererSampleRate: %d\n", param.GetSoundSystemParam().rendererSampleRate);

    m_FilterType = FilterType_None;
    m_LowPassFilterValue = DefaultLowPassFilterValue;
    m_BiquadFilterValue = DefaultBiquadFilterValue;
    SetupFilter();
    DumpFilterParam();
}

void FilterCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    m_CommonObject.Finalize();
}

void FilterCheckModule::OnLoadData() NN_NOEXCEPT
{
    m_CommonObject.LoadData();
}

void FilterCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    m_CommonObject.PrintUsage();

    NN_LOG("[Left/Right]   Select Filter Type\n");
    NN_LOG("[Up/Down]      Change Filter Value\n");
    NN_LOG("[R + Up/Down]  Change Sample Rate\n");
}

void FilterCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    int processCount = m_CommonObject.UpdateInput();
    bool isFilterParamChanged = false;

    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
    {
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
        {
            SetSampleRateType( static_cast<SampleRateType>(m_SampleRateType + 1) );
            ReinitializeAtk();
            processCount++;
        }
        else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
        {
            SetSampleRateType( static_cast<SampleRateType>(m_SampleRateType - 1) );
            ReinitializeAtk();
            processCount++;
        }
    }
    else
    {
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
        {
            SetFilterType( m_FilterType + 1 );
            isFilterParamChanged = true;
            processCount++;
        }
        else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
        {
            SetFilterType( m_FilterType - 1 );
            isFilterParamChanged = true;
            processCount++;
        }
        else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
        {
            if ( m_FilterType == FilterType_HandleLowPassFilter ||
                 m_FilterType == FilterType_PlayerLowPassFilter )
            {
                SetLowPassFilterValue(m_LowPassFilterValue - ChangeFilterValue);
            }
            else
            {
                SetBiquadFilterValue(m_BiquadFilterValue - ChangeFilterValue);
            }
            isFilterParamChanged = true;
            processCount++;
        }
        else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
        {
            if( m_FilterType == FilterType_HandleLowPassFilter ||
                m_FilterType == FilterType_PlayerLowPassFilter )
            {
                SetLowPassFilterValue(m_LowPassFilterValue + ChangeFilterValue);
            }
            else
            {
                SetBiquadFilterValue(m_BiquadFilterValue + ChangeFilterValue);
            }
            isFilterParamChanged = true;
            processCount++;
        }
    }

    if (processCount > 0)
    {
        SetupFilter();
    }
    if (isFilterParamChanged)
    {
        DumpFilterParam();
    }
}

void FilterCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    m_CommonObject.Update();
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void FilterCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
}
#endif

FlagList& FilterCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}

void FilterCheckModule::SetSampleRateType( SampleRateType sampleRateType ) NN_NOEXCEPT
{
    if ( sampleRateType > SampleRateType_Max )
    {
        m_SampleRateType = static_cast<SampleRateType>(0);
    }
    else if ( sampleRateType < 0 )
    {
        m_SampleRateType = SampleRateType_Max;
    }
    else
    {
        m_SampleRateType = sampleRateType;
    }
}

void FilterCheckModule::ReinitializeAtk() NN_NOEXCEPT
{
    OnFinalizeAtk();
    OnInitializeAtk();
    OnLoadData();
}

int FilterCheckModule::ConvertSampleRateTypeToInt( SampleRateType sampleRateType ) NN_NOEXCEPT
{
    switch (sampleRateType)
    {
    case SampleRateType_32000:
        return 32000;
    case SampleRateType_48000:
        return 48000;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void FilterCheckModule::SetFilterType( int filterType ) NN_NOEXCEPT
{
    if ( filterType > FilterType_Max )
    {
        m_FilterType = 0;
    }
    else if ( filterType < 0 )
    {
        m_FilterType = FilterType_Max;
    }
    else
    {
        m_FilterType = filterType;
    }
}

void FilterCheckModule::SetLowPassFilterValue(float lowPassFilterValue) NN_NOEXCEPT
{
    if (lowPassFilterValue > LowPassFilterValueMax - RoundFilterValue)
    {
        m_LowPassFilterValue = LowPassFilterValueMax;
    }
    else if (lowPassFilterValue < LowPassFilterValueMin + RoundFilterValue)
    {
        m_LowPassFilterValue = LowPassFilterValueMin;
    }
    else
    {
        m_LowPassFilterValue = lowPassFilterValue;
    }
}

void FilterCheckModule::SetBiquadFilterValue(float biquadFilterValue) NN_NOEXCEPT
{
    if (biquadFilterValue > BiquadFilterValueMax - RoundFilterValue)
    {
        m_BiquadFilterValue = BiquadFilterValueMax;
    }
    else if (biquadFilterValue < BiquadFilterValueMin + RoundFilterValue)
    {
        m_BiquadFilterValue = BiquadFilterValueMin;
    }
    else
    {
        m_BiquadFilterValue = biquadFilterValue;
    }
}

void FilterCheckModule::SetupFilter() NN_NOEXCEPT
{
    switch ( m_FilterType )
    {
    case FilterType_None:
        // ハンドルもプレイヤーもデフォルト値
        SetFilterParam(
            LowPassFilterValueMax,
            LowPassFilterValueMax,
            nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin,
            nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin);
        break;
    case FilterType_HandleLowPassFilter:
        // ハンドルの Low Pass Filter のみ設定、他はデフォルト値
        SetFilterParam(
            m_LowPassFilterValue,
            LowPassFilterValueMax,
            nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin,
            nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin);
        break;
    case FilterType_HandleBiquadLowPassFilter:
    case FilterType_HandleBiquadHighPassFilter:
    case FilterType_HandleBiquadBandPassFilter512:
    case FilterType_HandleBiquadBandPassFilter1024:
    case FilterType_HandleBiquadBandPassFilter2048:
    case FilterType_HandleBiquadLowPassFilterNw4fCompatible48k:
    case FilterType_HandleBiquadHighPassFilterNw4fCompatible48k:
    case FilterType_HandleBiquadBandPassFilter512Nw4fCompatible48k:
    case FilterType_HandleBiquadBandPassFilter1024Nw4fCompatible48k:
    case FilterType_HandleBiquadBandPassFilter2048Nw4fCompatible48k:
        // ハンドルの Biquad Filter のみ設定、他はデフォルト値
        SetFilterParam(
            LowPassFilterValueMax,
            LowPassFilterValueMax,
            m_FilterType - FilterType_HandleLowPassFilter, m_BiquadFilterValue,
            nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin);
        break;
    case FilterType_PlayerLowPassFilter:
        // プレイヤーの Low Pass Filter のみ設定、他はデフォルト値
        SetFilterParam(
            LowPassFilterValueMax,
            m_LowPassFilterValue,
            nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin,
            nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin);
        break;
    case FilterType_PlayerBiquadLowPassFilter:
    case FilterType_PlayerBiquadHighPassFilter:
    case FilterType_PlayerBiquadBandPassFilter512:
    case FilterType_PlayerBiquadBandPassFilter1024:
    case FilterType_PlayerBiquadBandPassFilter2048:
    case FilterType_PlayerBiquadLowPassFilterNw4fCompatible48k:
    case FilterType_PlayerBiquadHighPassFilterNw4fCompatible48k:
    case FilterType_PlayerBiquadBandPassFilter512Nw4fCompatible48k:
    case FilterType_PlayerBiquadBandPassFilter1024Nw4fCompatible48k:
    case FilterType_PlayerBiquadBandPassFilter2048Nw4fCompatible48k:
        // プレイヤーの Biquad Filter のみ設定、他はデフォルト値
        SetFilterParam(
            LowPassFilterValueMax,
            LowPassFilterValueMax,
            nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin,
            m_FilterType - FilterType_PlayerLowPassFilter, m_BiquadFilterValue);
        break;
    default:
        NN_LOG("Invalid filter type.\n");
        break;
    }
}

void FilterCheckModule::DumpFilterParam() NN_NOEXCEPT
{
    NN_LOG("filter( %-22s )   ", GetFilterTypeString( m_FilterType ));
    if ( m_FilterType == FilterType_None )
    {
        NN_LOG("\n");
    }
    else if ( m_FilterType == FilterType_HandleLowPassFilter ||
              m_FilterType == FilterType_PlayerLowPassFilter )
    {
        NN_LOG("value( %.1f )\n", m_LowPassFilterValue );
    }
    else
    {
        NN_LOG("value( %.1f )\n", m_BiquadFilterValue );
    }
}

void FilterCheckModule::SetFilterParam(float handleLpfValue, float playerLpfValue, int handleBiquadType, float handleBiquadValue, int playerBiquadType, float playerBiquadValue) NN_NOEXCEPT
{
    nn::atk::SoundPlayer& player = m_CommonObject.GetSoundArchivePlayer().GetSoundPlayer(DEFAULT_PLAYER);
    nn::atk::SoundHandle& handle = m_CommonObject.GetSoundHandle();

    // LPF
    handle.SetLowPassFilterFrequency(handleLpfValue);
    player.SetLowPassFilterFrequency(playerLpfValue);

    // Biquad Filter
    handle.SetBiquadFilter(handleBiquadType, handleBiquadValue);
    player.SetBiquadFilter(playerBiquadType, playerBiquadValue);
}

