﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DebugApi.h"

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

namespace
{
    FlagList g_LocalFlagList(nullptr, 0);
}

void FsAccessLogger::BeginRead(void* arg) NN_NOEXCEPT
{
    if (m_FileStreamAddress != nullptr)
    {
        NN_ASSERT_EQUAL(arg, m_FileStreamAddress);
    }
    NN_LOG("FsAccess Begin (%p)\n", arg);
    m_BeginTick =  nn::os::GetSystemTick();
}
void FsAccessLogger::EndRead(void* arg) NN_NOEXCEPT
{
    if (m_FileStreamAddress != nullptr)
    {
        NN_ASSERT_EQUAL(arg, m_FileStreamAddress);
    }
    nn::os::Tick interval = nn::os::GetSystemTick() - m_BeginTick;
    NN_LOG("FsAccess End   (%p) (%04dus)\n", arg, interval.ToTimeSpan().GetMicroSeconds());
}
void FsAccessLogger::SetFileStreamAddress(void* address) NN_NOEXCEPT
{
    m_FileStreamAddress = address;
}

void DebugApiCheckModule::CommonObjectForDebugApi::PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT
{
    bool result = GetSoundArchivePlayer().PrepareSound( &GetSoundHandle(), soundId ).IsSuccess();
    NN_LOG("StartSound(%s)%s ... (%d)\n", debugLabelName, CheckCacheState(soundId) ? " with edit" : "", result);
}

void DebugApiCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    m_CommonObject.Initialize();
}

void DebugApiCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    m_CommonObject.Finalize();
}

void DebugApiCheckModule::OnLoadData() NN_NOEXCEPT
{
    m_CommonObject.LoadData();

    // WSD の情報取得とダンプのサンプルコード
    const auto& soundArchivePlayer = m_CommonObject.GetSoundArchivePlayer();
    nn::atk::WaveSoundDataInfo wavInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS( soundArchivePlayer.ReadWaveSoundDataInfo(&wavInfo, WSD_YOSHI_ADPCM) );
    NN_LOG("[info]WSD_YOSHI_ADPCM\n");
    wavInfo.Dump();

    // STRM の情報取得とダンプのサンプルコード
    nn::atk::StreamSoundDataInfo strmInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS( soundArchivePlayer.ReadStreamSoundDataInfo(&strmInfo, "STRM_MARIOKART") );
    NN_LOG("[info]STRM_MARIOKART\n");
    strmInfo.Dump();

    // STRM のリージョン情報の取得とダンプのサンプルコード
    nn::atk::StreamSoundRegionDataInfo regionInfo;
    const size_t regionBufferSize = nn::atk::SoundArchivePlayer::GetRequiredWorkBufferSizeToReadStreamSoundHeader();
    void* regionBuffer = nns::atk::Allocate( regionBufferSize );

    NN_ABORT_UNLESS_RESULT_SUCCESS( soundArchivePlayer.ReadStreamSoundRegionDataInfo( &regionInfo, STRM_NAMEDREGION_TEST, "RegionA", regionBuffer, regionBufferSize ) );
    NN_LOG( "[info]STRM_REGION_JUMP region info \"RegionA\"\n" );
    NN_LOG( "    start    = %d\n", regionInfo.startSamplePosition );
    NN_LOG( "    end      = %d\n", regionInfo.endSamplePosition );
    NN_LOG( "    regionNo = %d\n", regionInfo.regionNo );
    NN_LOG( "    reginName= %s\n", regionInfo.regionName );

    NN_ABORT_UNLESS_RESULT_SUCCESS( soundArchivePlayer.ReadStreamSoundRegionDataInfo( &regionInfo, STRM_NAMEDREGION_TEST, "RegionB", regionBuffer, regionBufferSize ) );
    NN_LOG( "[info]STRM_REGION_JUMP region info \"RegionB\"\n" );
    NN_LOG( "    start    = %d\n", regionInfo.startSamplePosition );
    NN_LOG( "    end      = %d\n", regionInfo.endSamplePosition );
    NN_LOG( "    regionNo = %d\n", regionInfo.regionNo );
    NN_LOG( "    reginName= %s\n", regionInfo.regionName );

    // STRM のマーカー情報の取得とダンプのサンプルコード
    const int MarkerInfoCountMax = 16;     // 適当に大きめのサイズを指定
    nn::atk::StreamSoundMarkerInfo markerInfos[MarkerInfoCountMax];
    int actualMarkerInfoCount = 0;
    m_CommonObject.GetSoundArchivePlayer().ReadMarkerInfoArray(markerInfos, &actualMarkerInfoCount, MarkerInfoCountMax, STRM_MARKERTEST);
    NN_ASSERT_LESS_EQUAL(actualMarkerInfoCount, MarkerInfoCountMax);

    for (auto i = 0; i < actualMarkerInfoCount; ++i)
    {
        NN_LOG("[Marker%d]\n", i);
        NN_LOG("   name = %s\n", markerInfos[i].name);
        NN_LOG("   position = %d\n", markerInfos[i].position);
    }
}

void DebugApiCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    NN_LOG("[A]            PrepareSound SEQ   (SEQ_TEST_PCM16)\n");
    NN_LOG("[R + A]        PrepareSound SEQ   (SEQ_TEST_ADPCM)\n");
    NN_LOG("[X]            PrepareSound WSD   (WSD_YOSHI_ADPCM)\n");
    NN_LOG("[R + X]        PrepareSound WSD   (WSD_YOSHI_PCM16)\n");
    NN_LOG("[Y]            PrepareSound STRM  (STRM_PIANO16_ADPCM)\n");
    NN_LOG("[R + Y]        PrepareSound STRM  (STRM_PIANO16_PCM16)\n");
    NN_LOG("[Left]         PrepareSound STRM  (STRM_MARIOKART)\n");
    NN_LOG("[Right]        PrepareSound STRM  (STRM_MULTITRACK)\n");
    NN_LOG("[Up]           Exec StartPrepared() \n");
    NN_LOG("[Down]         CommandBufferDump \n");
    NN_LOG("[B]            Stop Sound\n");
    NN_LOG("[R + B]        Pause Sound\n");
}

void DebugApiCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    m_CommonObject.UpdateInput();

    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
    {
        m_CommonObject.PlayWithStartSound(STRM_MARIOKART, "STRM_MARIOKART");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
    {
        m_CommonObject.PlayWithStartSound(STRM_MULTITRACK, "STRM_MULTITRACK");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
    {
        if (GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_UseStreamCache))
        {
            nn::atk::StreamSoundHandle streamSoundHandle(&m_CommonObject.GetSoundHandle());
            if ( streamSoundHandle.IsAttachedSound() )
            {
                void* fileStreamAddress = streamSoundHandle.detail_SetFsAccessLog(&m_FsAccessLogger);
                m_FsAccessLogger.SetFileStreamAddress(fileStreamAddress);
            }
        }

        m_CommonObject.GetSoundHandle().StartPrepared();
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
    {
        // サウンドスレッドが持つコマンドバッファの充填状況
        int commandCount = nn::atk::SoundSystem::GetAllocatedDriverCommandCount();
        size_t allocatedBufferSize = nn::atk::SoundSystem::GetAllocatedDriverCommandBufferSize();
        size_t allocatableCommandSize = nn::atk::SoundSystem::GetAllocatableDriverCommandSize();
        size_t commandBufferSize = nn::atk::SoundSystem::GetDriverCommandBufferSize();
        DumpCommandBuffer("SoundThread", commandCount, allocatedBufferSize, allocatableCommandSize, commandBufferSize);

        // タスクスレッドが持つコマンドバッファの充填状況
        commandCount = nn::atk::detail::DriverCommand::GetInstanceForTaskThread().GetAllocatedCommandCount();
        allocatedBufferSize = nn::atk::detail::DriverCommand::GetInstanceForTaskThread().GetAllocatedCommandBufferSize();
        allocatableCommandSize = nn::atk::detail::DriverCommand::GetInstanceForTaskThread().GetAllocatableCommandSize();
        commandBufferSize = nn::atk::detail::DriverCommand::GetInstanceForTaskThread().GetCommandBufferSize();
        DumpCommandBuffer("TaskThread", commandCount, allocatedBufferSize, allocatableCommandSize, commandBufferSize);

#if defined(NN_ATK_ENABLE_VOICE_COMMAND_SAMPLE)
        // ボイスコマンドが持つコマンドバッファの充填状況
        commandCount = nn::atk::detail::LowLevelVoiceCommand::GetInstance().GetAllocatedCommandCount();
        allocatedBufferSize = nn::atk::detail::LowLevelVoiceCommand::GetInstance().GetAllocatedCommandBufferSize();
        allocatableCommandSize = nn::atk::detail::LowLevelVoiceCommand::GetInstance().GetAllocatableCommandSize();
        commandBufferSize = nn::atk::detail::LowLevelVoiceCommand::GetInstance().GetCommandBufferSize();
        DumpCommandBuffer("VoiceCommand", commandCount, allocatedBufferSize, allocatableCommandSize, commandBufferSize);
        // 確保済みの仮想ボイスの数
        NN_LOG("  - VirtualVoiceCount            %8zd\n",
            nn::atk::detail::VirtualVoiceManager::GetInstance().GetAllocatedVirtualVoiceCount());
#endif
    }
}

void DebugApiCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    m_CommonObject.Update();

    // サウンドアーカイブプレイヤーの稼働状況
    // ストリームサウンド、ストリームチャンネル、ストリームトラックのインスタンス稼働状況
    nn::atk::SoundArchivePlayer::StreamSoundInstanceState instanceState;
    m_CommonObject.GetSoundArchivePlayer().ReadStreamSoundInstanceState(&instanceState);
    nn::atk::SoundArchive::SoundArchivePlayerInfo archivePlayerInfo;
    m_CommonObject.GetSoundArchive().ReadSoundArchivePlayerInfo(&archivePlayerInfo);
    NN_LOG("[Stream][Sound]%d/%d [Track]%d/%d [Channel]%d/%d",
        instanceState.activeStreamSoundInstanceCount,
        archivePlayerInfo.streamSoundCount,
        instanceState.activeStreamTrackCount,
        archivePlayerInfo.streamTrackCount,
        instanceState.activeStreamChannelCount,
        archivePlayerInfo.streamChannelCount);

    // ストリームサウンドの稼働状況
    nn::atk::StreamSoundHandle streamSoundHandle(&m_CommonObject.GetSoundHandle());
    if ( streamSoundHandle.IsAttachedSound() )
    {
        nn::atk::StreamSoundDataInfo dataInfo;
        streamSoundHandle.ReadStreamSoundDataInfo(&dataInfo);
        // バッファ充填率、バッファブロックの状態、ループ情報、サンプルレート
        NN_LOG(" [Handle][Buffer]Filled:%.0f%% [Block]Play:%d Wait:%d Done:%d Free:%d Total:%d [loop]%s(%d) start:%d end:%d [rate]%d",
            streamSoundHandle.GetFilledBufferPercentage(),
            streamSoundHandle.GetBufferBlockCount(nn::atk::WaveBuffer::Status_Play),
            streamSoundHandle.GetBufferBlockCount(nn::atk::WaveBuffer::Status_Wait),
            streamSoundHandle.GetBufferBlockCount(nn::atk::WaveBuffer::Status_Done),
            streamSoundHandle.GetBufferBlockCount(nn::atk::WaveBuffer::Status_Free),
            streamSoundHandle.GetTotalBufferBlockCount(),
            dataInfo.loopFlag ? "ON" : "OFF",
            streamSoundHandle.GetPlayLoopCount(),
            dataInfo.loopStart,
            dataInfo.loopEnd,
            dataInfo.sampleRate);
    }

    // ウェーブサウンドの稼働状況
    nn::atk::WaveSoundHandle waveSoundHandle(&m_CommonObject.GetSoundHandle());
    if ( waveSoundHandle.IsAttachedSound() )
    {
        // チャンネル数、ループ情報、サンプルレート
        nn::atk::WaveSoundDataInfo dataInfo;
        waveSoundHandle.ReadWaveSoundDataInfo(&dataInfo);
        NN_LOG(" [Wave][Handle][Channel]Total:%d [loop]%s start:%d end:%d [rate]%d",
            waveSoundHandle.GetChannelCount(),
            dataInfo.loopFlag ? "ON" : "OFF",
            dataInfo.loopStart,
            dataInfo.loopEnd,
            dataInfo.sampleRate);
    }

    // シーケンスサウンドの稼働状況
    nn::atk::SequenceSoundHandle sequenceSoundHandle(&m_CommonObject.GetSoundHandle());
    if ( sequenceSoundHandle.IsAttachedSound() )
    {
        // チック数
        NN_LOG(" [Sequence][Handle][Tick]%u", sequenceSoundHandle.GetTick());
    }

    NN_LOG("\n");
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void DebugApiCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
}
#endif

FlagList& DebugApiCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}

void DebugApiCheckModule::DumpCommandBuffer(const char* label, const int commandCount, const size_t allocatedBufferSize,
                                            const size_t allocatableCommandSize, const size_t bufferSize) const NN_NOEXCEPT
{
    float fillingPercentage = 100.0f * allocatedBufferSize / bufferSize;
    NN_LOG("  %s CommandBuffer\n"
               "  - CommandCount                 %8zd\n"
               "  - AllocatedCommandBufferSize   %8zd bytes\n"
               "  - AllocatableCommandSize       %8zd bytes\n"
               "  - CommandBufferSize            %8zd bytes\n"
               "  - Filling Percentage              %3.3f %%\n"
               , label
               , commandCount
               , allocatedBufferSize
               , allocatableCommandSize
               , bufferSize
               , fillingPercentage);
}
