﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/atk.h>
#include <nn/htcs.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>

#include <nn/atk/viewer/atk_SoundEdit.h>

#include "nns/atk/atk_SampleCommon.h"
#include "ModuleSelector.h"
#include "CheckCode/ICheckModule.h"
#include "FlagList.h"

#include "CheckCode/Simple.h"
#include "CheckCode/Effect.h"
#include "CheckCode/SoundActor.h"
#include "CheckCode/StreamSound.h"
#include "CheckCode/Surround.h"
#include "CheckCode/ExternalFile.h"
#include "CheckCode/MixParameter.h"
#include "CheckCode/StartInfo.h"
#include "CheckCode/DebugApi.h"
#include "CheckCode/Sound3D.h"
#include "CheckCode/Filter.h"
#include "CheckCode/SinglePlay.h"
#include "CheckCode/CustomSubMix.h"
#include "CheckCode/EditGlobalFlag.h"

#include "GfxCode/DebugViewer.h"

namespace
{
    const char Title[] = "AtkSandbox";

    ModuleSelector g_ModuleSelector;

    SimpleCheckModule g_SimpleCheckModule;
    EffectCheckModule g_EffectCheckModule;
    SoundActorCheckModule g_SoundActorCheckModule;
    StreamSoundCheckModule g_StreamSoundCheckModule;
    SurroundCheckModule g_SurroundCheckModule;
    ExternalFileCheckModule g_ExternalFileCheckModule;
    MixModeCheckModule g_MixModeCheckModule;
    StartInfoCheckModule g_StartInfoCheckModule;
    DebugApiCheckModule g_DebugApiCheckModule;
    Sound3DCheckModule g_Sound3DCheckModule;
    FilterCheckModule g_FilterCheckModule;
    SinglePlayCheckModule g_SinglePlayCheckModule;
    CustomSubMixCheckModule g_CustomSubMixCheckModule;
    EditGlobalFlagModule g_EditGlobalFlagModule;

    ICheckModule* g_pCurrentCheckModule;

    ICheckModule* g_pModuleList[] =
    {
        &g_SimpleCheckModule,
        &g_EffectCheckModule,
        &g_SoundActorCheckModule,
        &g_StreamSoundCheckModule,
        &g_SurroundCheckModule,
        &g_ExternalFileCheckModule,
        &g_MixModeCheckModule,
        &g_StartInfoCheckModule,
        &g_DebugApiCheckModule,
        &g_Sound3DCheckModule,
        &g_FilterCheckModule,
        &g_SinglePlayCheckModule,
        &g_CustomSubMixCheckModule,
        &g_EditGlobalFlagModule,
    };

    const int ModuleListCount = sizeof(g_pModuleList) / sizeof(g_pModuleList[0]);

    // ここを変更した場合には、FlagList.h に定義されている。GlobalFlagType も同時に修正してください。
    FlagElement g_GlobalElements[] =
    {
        { "UseMemorySoundArchive", false },
        { "UseStreamCache", false },
        { "SoundSystemUsingApplicationMemoryPool", false },
        { "SoundSystemUsingApplicationCircularBufferSinkBuffer", false },
        { "EnableSoundArchivePlayerAppliedStreamInstanceBuffer", false },
        { "DisableSoundThread", false },
        { "EnableSpy", false },
        { "EnableCpuProfiler", false},
    };

    FlagList g_GlobalFlagList(g_GlobalElements, sizeof(g_GlobalElements) / sizeof(g_GlobalElements[0]));
}

void InitializeHid() NN_NOEXCEPT
{
    nns::atk::InitializeHidDevices();

    // 追加の必要なキーボードのキー入力を DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonZR = nn::hid::KeyboardKey::Z::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void FinalizeHid() NN_NOEXCEPT
{
}

void Initialize() NN_NOEXCEPT
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();

    InitializeHid();

    const nn::Result result = nn::fs::MountHostRoot();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::htcs::Initialize(nns::atk::Allocate, nns::atk::Free);

    AttachGlobalFlagList(&g_GlobalFlagList);

    DebugViewer::GetInstance().Initialize();
}

void Finalize() NN_NOEXCEPT
{
    DebugViewer::GetInstance().Finalize();

    DetachGlobalFlagList();

    nn::htcs::Finalize();
    nn::fs::UnmountHostRoot();

    FinalizeHid();

    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeAtk() NN_NOEXCEPT
{
    g_pCurrentCheckModule->OnInitializeAtk();
}

void FinalizeAtk() NN_NOEXCEPT
{
    g_pCurrentCheckModule->OnFinalizeAtk();
}

void LoadData() NN_NOEXCEPT
{
    g_pCurrentCheckModule->OnLoadData();
}


void PrintUsage() NN_NOEXCEPT
{
    bool isVoiceCommandMode = false;
#if defined(NN_ATK_ENABLE_VOICE_COMMAND_SAMPLE)
    isVoiceCommandMode = true;
#endif

    const char* moduleName = g_pCurrentCheckModule->GetModuleName();
    const char* modeString = isVoiceCommandMode ? "(VoiceCommandMode)" : "";
    NN_LOG("---------------------------------------------------\n");
    NN_LOG("%s(%s) %s\n", Title, moduleName, modeString);
    NN_LOG("---------------------------------------------------\n");
    g_pCurrentCheckModule->OnPrintUsage();
    NN_LOG("[L]            Print Usage\n");
    NN_LOG("[Start][Space] Exit Application\n");
    NN_LOG("---------------------------------------------------\n");

    g_pCurrentCheckModule->OnPostPrintUsage();
}

bool UpdateInput() NN_NOEXCEPT
{
    g_pCurrentCheckModule->OnUpdateInput();

    // PrintUsage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() )
    {
        return false;
    }

    return true;
}

void UpdateAtk() NN_NOEXCEPT
{
    g_pCurrentCheckModule->OnUpdateAtk();
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void UpdateDraw() NN_NOEXCEPT
{
    g_pCurrentCheckModule->OnUpdateDraw();
}
#endif

extern "C" void nnMain() NN_NOEXCEPT
{
    Initialize();

    for (;;)
    {
        g_pCurrentCheckModule = g_ModuleSelector.Select(g_pModuleList, ModuleListCount);
        if (g_pCurrentCheckModule == nullptr)
        {
            break;
        }

        DebugViewer::GetInstance().Reset();

        InitializeAtk();

        LoadData();

        PrintUsage();

        for (;;)
        {
            nns::atk::UpdateHidDevices();

            if ( !UpdateInput() )
            {
                break;
            }
            UpdateAtk();
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
            UpdateDraw();
#endif

            DebugViewer::GetInstance().Update();
            DebugViewer::GetInstance().Present();

            // Vsync の代わり
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }

        FinalizeAtk();
        DebugViewer::GetInstance().Reset();
    }

    Finalize();
}

