﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "AtkProfiler.h"
#include "GfxCode/DebugViewer.h"

#include "nns/atk/atk_SampleCommon.h"

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
namespace
{
    // サウンドプロファイル表示のデフォルトの文字列描画位置
    const int DefaultPrintSoundProfilePosX = 500;
    const int DefaultPrintSoundProfilePosY = 250;
    // タスクプロファイル表示のデフォルトの文字列描画位置
    const int DefaultPrintTaskProfilePosX = 500;
    const int DefaultPrintTaskProfilePosY = 450;
    // サウンドスレッドプロファイル表示のデフォルトの文字列描画位置
    const int DefaultPrintSoundThreadProfilePosX = 500;
    const int DefaultPrintSoundThreadProfilePosY = 600;
}
#endif
namespace
{
    //  SoundThreadInfoRecorder に設定するバッファサイズ
    const size_t SoundThreadInfoRecorderBufferSize = 64 * 1024;
}

//  必要なバッファサイズを計算します
size_t AtkProfiler::GetRequiredMemorySize(int taskProfileCount, int soundThreadUpdateProfileCount) NN_NOEXCEPT
{
    size_t size = 0;

    size += m_TaskProfileReader.GetRequiredMemorySize(taskProfileCount);
    size += sizeof(nn::atk::TaskProfile) * taskProfileCount;

    size += m_SoundThreadUpdateProfileReader.GetRequiredMemorySize(soundThreadUpdateProfileCount);
    size += sizeof(nn::atk::SoundThreadUpdateProfile) * soundThreadUpdateProfileCount;

    size += SoundThreadInfoRecorderBufferSize;

    return size;
}
//  初期化します
void AtkProfiler::Initialize(void* buffer, size_t bufferSize, int taskProfileCount, int soundThreadUpdateProfileCount) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(buffer);
    NN_ABORT_UNLESS_GREATER_EQUAL(bufferSize, GetRequiredMemorySize(taskProfileCount, soundThreadUpdateProfileCount));
    nn::util::BytePtr ptr(buffer);

    {
        size_t size = m_TaskProfileReader.GetRequiredMemorySize(taskProfileCount);
        m_TaskProfileCount = taskProfileCount;
        m_TaskProfileReaderBuffer = ptr.Get();
        m_TaskProfileReader.Initialize(m_TaskProfileReaderBuffer, size, m_TaskProfileCount);
        ptr.Advance(size);

        m_TaskProfile = reinterpret_cast<nn::atk::TaskProfile*>(ptr.Get());
        ptr.Advance(sizeof(nn::atk::TaskProfile) * taskProfileCount);
    }

    {
        size_t size = m_SoundThreadUpdateProfileReader.GetRequiredMemorySize(soundThreadUpdateProfileCount);
        m_SoundThreadUpdateProfileCount = soundThreadUpdateProfileCount;
        m_SoundThreadUpdateProfileReaderBuffer = ptr.Get();
        m_SoundThreadUpdateProfileReader.Initialize(m_SoundThreadUpdateProfileReaderBuffer, size, m_SoundThreadUpdateProfileCount);
        ptr.Advance(size);

        m_SoundThreadUpdateProfile = reinterpret_cast<nn::atk::SoundThreadUpdateProfile*>(ptr.Get());
        ptr.Advance( sizeof(nn::atk::SoundThreadUpdateProfile) * soundThreadUpdateProfileCount );
    }

    {
        const size_t size = SoundThreadInfoRecorderBufferSize;
        m_SoundThreadInfoRecorder.Initialize( ptr.Get(), size );
        ptr.Advance( size );
    }

    nn::atk::SoundSystem::RegisterProfileReader(m_ProfileReader);
    nn::atk::SoundSystem::RegisterTaskProfileReader(m_TaskProfileReader);
    nn::atk::SoundSystem::RegisterSoundThreadUpdateProfileReader(m_SoundThreadUpdateProfileReader);
    nn::atk::SoundSystem::RegisterSoundThreadInfoRecorder(m_SoundThreadInfoRecorder);
    m_IsProfileReaderEnabled = false;
    m_IsUsingThreadInfoRecorder = false;
    m_IsSoundThreadUpdateProfileReaderEnabled = false;
    m_IsTaskProfileReaderEnabled = false;
    m_IsFirstProfileRead = false;
}
//  終了処理をします
void AtkProfiler::Finalize() NN_NOEXCEPT
{
    nn::atk::SoundSystem::UnregisterSoundThreadInfoRecorder(m_SoundThreadInfoRecorder);
    nn::atk::SoundSystem::UnregisterSoundThreadUpdateProfileReader(m_SoundThreadUpdateProfileReader);
    nn::atk::SoundSystem::UnregisterTaskProfileReader(m_TaskProfileReader);
    nn::atk::SoundSystem::UnregisterProfileReader(m_ProfileReader);
    m_SoundThreadUpdateProfileReader.Finalize();
    m_TaskProfileReader.Finalize();
}

//  表示処理をします
void AtkProfiler::Show(nn::atk::SoundHandle& soundHandle) NN_NOEXCEPT
{
    if( m_IsUsingThreadInfoRecorder )
    {
        ShowProfileUsingThreadInfoReader( soundHandle );
    }
    else
    {
        ShowProfile( soundHandle );
    }
}

void AtkProfiler::ShowProfile(nn::atk::SoundHandle& soundHandle) NN_NOEXCEPT
{
    if (m_IsProfileReaderEnabled)
    {
        const int profileCount = m_ProfileReader.Read(m_SoundProfile, SoundProfileCount);
        for(int i = 0; i < profileCount; i++)
        {
            ShowSoundProfile(soundHandle, m_SoundProfile[i]);
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
            DrawSoundProfile(soundHandle, m_SoundProfile[i]);
#endif
        }
    }
    else
    {
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
        DebugPrint("SoundProfile", DefaultPrintSoundProfilePosX, DefaultPrintSoundProfilePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, "[ZR(Z) + A]");
#endif
    }


    if (m_IsSoundThreadUpdateProfileReaderEnabled)
    {
        const int profileCount = m_SoundThreadUpdateProfileReader.Read(m_SoundThreadUpdateProfile, m_SoundThreadUpdateProfileCount);
        for(int i = 0; i < profileCount; i++)
        {
            ShowSoundThreadUpdateProfile(m_SoundThreadUpdateProfile[i]);
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
            DrawSoundThreadUpdateProfile(m_SoundThreadUpdateProfile[i]);
#endif
        }
    }
    else
    {
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
        DebugPrint("SoundThreadProfile", DefaultPrintSoundThreadProfilePosX, DefaultPrintSoundThreadProfilePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, "[ZR(Z) + Y]");
#endif
    }


    if (m_IsTaskProfileReaderEnabled)
    {
        int profileCount = m_TaskProfileReader.Read(m_TaskProfile,m_TaskProfileCount);
        ShowTaskProfile(profileCount);
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
        DrawTaskProfile(profileCount);
#endif
    }
    else
    {
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
        DebugPrint("TaskProfile", DefaultPrintTaskProfilePosX, DefaultPrintTaskProfilePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, "[ZR(Z) + X]");
#endif
    }
}

void AtkProfiler::ShowProfileUsingThreadInfoReader(nn::atk::SoundHandle& soundHandle) NN_NOEXCEPT
{
    nn::atk::SoundProfile soundProfile;
    nn::atk::SoundThreadUpdateProfile soundThreadUpdateProfile;

    bool isShowedSoundProfile = false;
    bool isShowedSoundThreadUpdateProfile = false;

    for( nn::atk::SoundThreadInfoReader reader = m_SoundThreadInfoRecorder; reader.HasFrameInfo(); reader.MoveToNextFrame() )
    {
        reader.SetupFrameInfo();

        if (m_IsProfileReaderEnabled)
        {
            while( reader.ReadSoundProfile( &soundProfile ) )
            {
                ShowSoundProfile(soundHandle, soundProfile);
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
                DrawSoundProfile(soundHandle, soundProfile);
#endif
                isShowedSoundProfile = true;
            }
        }

        if (m_IsSoundThreadUpdateProfileReaderEnabled)
        {
            while( reader.ReadSoundThreadUpdateProfile( &soundThreadUpdateProfile ) )
            {
                ShowSoundThreadUpdateProfile(soundThreadUpdateProfile);
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
                DrawSoundThreadUpdateProfile(soundThreadUpdateProfile);
#endif
                isShowedSoundThreadUpdateProfile = true;
            }
        }
    }

    if( !isShowedSoundProfile )
    {
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
        DebugPrint("SoundProfile", DefaultPrintSoundProfilePosX, DefaultPrintSoundProfilePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, "[ZR(Z) + A]");
#endif
    }


    if( !isShowedSoundThreadUpdateProfile )
    {
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
        DebugPrint("SoundThreadProfile", DefaultPrintSoundThreadProfilePosX, DefaultPrintSoundThreadProfilePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, "[ZR(Z) + Y]");
#endif
    }


    //  TaskProfile は未対応
    if (m_IsTaskProfileReaderEnabled)
    {
        int profileCount = m_TaskProfileReader.Read(m_TaskProfile,m_TaskProfileCount);
        ShowTaskProfile(profileCount);
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
        DrawTaskProfile(profileCount);
#endif
    }
    else
    {
#if defined( NN_ATK_ENABLE_GFX_VIEWING )
        DebugPrint("TaskProfile", DefaultPrintTaskProfilePosX, DefaultPrintTaskProfilePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, "[ZR(Z) + X]");
#endif
    }
}

int AtkProfiler::UpdateInput() NN_NOEXCEPT
{
    int processCount = 0;

    if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >())
    {
        if (nns::atk::IsHold < ::nn::hid::DebugPadButton::ZR >())
        {
            SetProfileReaderEnabled(!IsProfileReaderEnabled());
            processCount++;
        }
    }
    if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >())
    {
        if (nns::atk::IsHold < ::nn::hid::DebugPadButton::ZR >())
        {
            SetTaskProfileReaderEnabled(!IsTaskProfileReaderEnabled());
            processCount++;
        }
    }
    if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >())
    {
        if (nns::atk::IsHold < ::nn::hid::DebugPadButton::ZR >())
        {
            SetSoundThreadUpdateProfileReaderEnabled(!IsSoundThreadUpdateProfileReaderEnabled());
            processCount++;
        }
    }
    if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >())
    {
        if (nns::atk::IsHold < ::nn::hid::DebugPadButton::ZR >())
        {
            SetUsingThreadInfoRecorder(!IsUsingThreadInfoRecorder());
            processCount++;
        }
    }

    return processCount;
}

void AtkProfiler::PrintUsage() NN_NOEXCEPT
{
    NN_LOG("[ZR(Z) + A]    Start/Stop Profile\n");
    NN_LOG("[ZR(Z) + X]    Start/Stop TaskProfile\n");
    NN_LOG("[ZR(Z) + Y]    Start/Stop SoundThreadUpdateProfile\n");
    NN_LOG("[ZR(Z) + B]    On/Off     UsingThreadRecorder\n");
}

bool AtkProfiler::IsProfileReaderEnabled() const NN_NOEXCEPT
{
    return m_IsProfileReaderEnabled;
}

void AtkProfiler::SetProfileReaderEnabled(bool isEnabled) NN_NOEXCEPT
{
    m_IsProfileReaderEnabled = isEnabled;

    if (m_IsProfileReaderEnabled)
    {
        m_IsFirstProfileRead = true;
        NN_LOG("---------------------------------------------------------------------------------------\n");
        NN_LOG("numVoices atkFr(us) dspTotalFr(us) sndFr(us) voFr(us) sMixFr(us) sMix2Fr(us) fMixFr(us) sinkFr(us) CBSFr(us) sndInt(us)\n");
        NN_LOG("---------------------------------------------------------------------------------------\n");
    }
}

bool AtkProfiler::IsUsingThreadInfoRecorder() const NN_NOEXCEPT
{
    return m_IsUsingThreadInfoRecorder;
}

void AtkProfiler::SetUsingThreadInfoRecorder(bool isEnabled) NN_NOEXCEPT
{
    m_IsUsingThreadInfoRecorder = isEnabled;
    NN_LOG("Using ThreadInfoRecorder [%s]\n", m_IsUsingThreadInfoRecorder ? "ON" : "OFF");
}

bool AtkProfiler::IsTaskProfileReaderEnabled() const NN_NOEXCEPT
{
    return m_IsTaskProfileReaderEnabled;
}

void AtkProfiler::SetTaskProfileReaderEnabled(bool isEnabled) NN_NOEXCEPT
{
    m_IsTaskProfileReaderEnabled = isEnabled;
    NN_LOG("Task Profile [%s]\n", m_IsTaskProfileReaderEnabled ? "ON" : "OFF");
}

bool AtkProfiler::IsSoundThreadUpdateProfileReaderEnabled() const NN_NOEXCEPT
{
    return m_IsSoundThreadUpdateProfileReaderEnabled;
}

void AtkProfiler::SetSoundThreadUpdateProfileReaderEnabled(bool isEnabled) NN_NOEXCEPT
{
    m_IsSoundThreadUpdateProfileReaderEnabled = isEnabled;
}

void AtkProfiler::ShowSoundProfile(nn::atk::SoundHandle& soundHandle, const nn::atk::SoundProfile& profile) NN_NOEXCEPT
{
        // 再生中のサウンドの DSP 処理負荷を取得
    nn::os::Tick soundRendererProcess(0);
    nn::atk::WaveSoundHandle waveSoundHandle(&soundHandle);
    nn::atk::StreamSoundHandle streamSoundHandle(&soundHandle);
    nn::atk::SequenceSoundHandle sequenceSoundHandle(&soundHandle);
    if (waveSoundHandle.IsAttachedSound())
    {
        soundRendererProcess = waveSoundHandle.GetProcessTick(profile);
    }
    else if (streamSoundHandle.IsAttachedSound())
    {
        soundRendererProcess = streamSoundHandle.GetProcessTick(profile);
    }
    else if (sequenceSoundHandle.IsAttachedSound())
    {
        soundRendererProcess = sequenceSoundHandle.GetProcessTick(profile);
    }

    NN_LOG("%9d %9lld %14lld %9lld %8lld %10lld %10lld %10lld %10lld %9lld %10lld\n",
        profile.totalVoiceCount,
        profile.nwFrameProcess.GetSpan().GetMicroSeconds(),
        profile.rendererFrameProcess.GetSpan().GetMicroSeconds(),
        soundRendererProcess.ToTimeSpan().GetMicroSeconds(),
        profile.voiceProcess.GetSpan().GetMicroSeconds(),
        profile.mainMixProcess.GetSpan().GetMicroSeconds(),
        profile._additionalSubMixProcess.GetSpan().GetMicroSeconds(),
        profile.finalMixProcess.GetSpan().GetMicroSeconds(),
        profile.sinkProcess.GetSpan().GetMicroSeconds(),
        profile.circularBufferSinkProcess.GetSpan().GetMicroSeconds(),
        m_IsFirstProfileRead ? 0 : profile.GetNwFrameProcessInterval(m_PreviousAtkFrameProcessBegin).GetMicroSeconds()
    );

    m_PreviousAtkFrameProcessBegin = profile.nwFrameProcess.begin;
    m_IsFirstProfileRead = false;
}

void AtkProfiler::ShowTaskProfile(int profileCount) NN_NOEXCEPT
{
    for (int i = 0; i < profileCount; i++)
    {
        const auto& taskProfile = m_TaskProfile[i];
        switch (taskProfile.type)
        {
        case nn::atk::TaskProfile::TaskProfileType_LoadStreamBlock:
        {
            const auto& data = taskProfile.loadStreamBlock;

            NN_LOG("[Load STRM Block] total: %3lld us\n", data.GetTotalTime().GetMicroSeconds());

            break;
        }
        case nn::atk::TaskProfile::TaskProfileType_LoadOpusStreamBlock:
        {
            const auto& data = taskProfile.loadOpusStreamBlock;

            NN_LOG("[Load Opus STRM Block] total: %4lld us  decode( %4lld us, %4d samples ), fs( %3lld us, %3d bytes )\n", data.GetTotalTime().GetMicroSeconds(), data.GetDecodeTime().GetMicroSeconds(), data.GetDecodedSampleCount(), data.GetFsAccessTime().GetMicroSeconds(), data.GetFsReadSize());

            break;
        }
        default:
        {
            NN_UNEXPECTED_DEFAULT;
            break;
        }
        }
    }
}

void AtkProfiler::ShowSoundThreadUpdateProfile(const nn::atk::SoundThreadUpdateProfile& profile) NN_NOEXCEPT
{
    NN_LOG("total: %4lld us, voice: %4lld us, frame: %4lld us, effect: %4lld us, update: %4lld us, wait: %5lld us, update+wait: %5lld us, interval: %5lld us\n",
        profile.soundThreadProcess.GetSpan().GetMicroSeconds(),
        profile._updateLowLevelVoiceProcess.GetSpan().GetMicroSeconds(),
        profile._frameProcess.GetSpan().GetMicroSeconds(),
        profile._userEffectFrameProcess.GetSpan().GetMicroSeconds(),
        profile._updateRendererProcess.GetSpan().GetMicroSeconds(),
        profile._waitRendererEventProcess.GetSpan().GetMicroSeconds(),
        profile._updateRendererProcess.GetSpan().GetMicroSeconds() + profile._waitRendererEventProcess.GetSpan().GetMicroSeconds(),
        (profile.soundThreadProcess.begin - m_PreviousSoundThreadUpdateProcessBegin).ToTimeSpan().GetMicroSeconds()
    );
    m_PreviousSoundThreadUpdateProcessBegin = profile.soundThreadProcess.begin;
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void AtkProfiler::DrawSoundProfile(nn::atk::SoundHandle& soundHandle, const nn::atk::SoundProfile& profile) NN_NOEXCEPT
{
    nn::os::Tick soundRendererProcess(0);
    nn::atk::WaveSoundHandle waveSoundHandle(&soundHandle);
    nn::atk::StreamSoundHandle streamSoundHandle(&soundHandle);
    nn::atk::SequenceSoundHandle sequenceSoundHandle(&soundHandle);
    if (waveSoundHandle.IsAttachedSound())
    {
        soundRendererProcess = waveSoundHandle.GetProcessTick(profile);
    }
    else if (streamSoundHandle.IsAttachedSound())
    {
        soundRendererProcess = streamSoundHandle.GetProcessTick(profile);
    }
    else if (sequenceSoundHandle.IsAttachedSound())
    {
        soundRendererProcess = sequenceSoundHandle.GetProcessTick(profile);
    }

    DebugPrint("SoundProfile", DefaultPrintSoundProfilePosX, DefaultPrintSoundProfilePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
        "[ZR(Z) + A]\n"
        "  VoiceCount          :%8d\n"
        "  NwProc          (us):%8lld\n"
        "  CurrentSoundProc(us):%8lld\n"
        "  RendererProc    (us):%8lld\n"
        "    VoiceProc     (us):%8lld\n"
        "    MainMixProc   (us):%8lld\n"
        "    FinalMixProc  (us):%8lld\n"
        "    SinkProc      (us):%8lld\n"
        "    CBSProc       (us):%8lld"
        , profile.totalVoiceCount
        , profile.nwFrameProcess.GetSpan().GetMicroSeconds()
        , soundRendererProcess.ToTimeSpan().GetMicroSeconds()
        , profile.rendererFrameProcess.GetSpan().GetMicroSeconds()
        , profile.voiceProcess.GetSpan().GetMicroSeconds()
        , profile.mainMixProcess.GetSpan().GetMicroSeconds()
        , profile.finalMixProcess.GetSpan().GetMicroSeconds()
        , profile.sinkProcess.GetSpan().GetMicroSeconds()
        , profile.circularBufferSinkProcess.GetSpan().GetMicroSeconds()
    );
}

void AtkProfiler::DrawTaskProfile(int profileCount) NN_NOEXCEPT
{
    for (int i = 0; i < profileCount; i++)
    {
        const auto& taskProfile = m_TaskProfile[i];
        switch (taskProfile.type)
        {
        case nn::atk::TaskProfile::TaskProfileType_LoadStreamBlock:
        {
            const auto& data = taskProfile.loadStreamBlock;

            DebugPrint("TaskProfile", DefaultPrintTaskProfilePosX, DefaultPrintTaskProfilePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
                "[ZR(Z) + X]\n"
                "  FsAccess(us)  :%8d\n"
                "  CacheLength   :%8u\n"
                "  RemainCache(%):%8.2f"
                , data.GetTotalTime().GetMicroSeconds()
                , data.GetCachedLength()
                , data.GetRemainingCachePercentage()
            );

            if (data.GetCachedLength() != 0)
            {
                const int RemainCacheBarOffsetY = 70;
                int remainCacheBarUsedOffsetX = static_cast<int>(DebugViewer::DefaultDrawBarWidth * (100.f - data.GetRemainingCachePercentage()) / 100.f);
                DebugDrawQuad(
                    "Used",
                    DefaultPrintTaskProfilePosX,
                    DefaultPrintTaskProfilePosY + RemainCacheBarOffsetY,
                    DefaultPrintTaskProfilePosX + remainCacheBarUsedOffsetX,
                    DefaultPrintTaskProfilePosY + RemainCacheBarOffsetY + DebugViewer::DefaultDrawBarHeight,
                    GetDebugColor( 130, 130, 130 )
                );
                DebugDrawQuad(
                    "Remain",
                    DefaultPrintTaskProfilePosX + remainCacheBarUsedOffsetX,
                    DefaultPrintTaskProfilePosY + RemainCacheBarOffsetY,
                    DefaultPrintTaskProfilePosX + DebugViewer::DefaultDrawBarWidth,
                    DefaultPrintTaskProfilePosY + RemainCacheBarOffsetY + DebugViewer::DefaultDrawBarHeight,
                    GetDebugColor(   0,   0, 230 )
                );
            }

            break;
        }
        case nn::atk::TaskProfile::TaskProfileType_LoadOpusStreamBlock:
        {
            const auto& data = taskProfile.loadOpusStreamBlock;

            DebugPrint("TaskProfile", DefaultPrintTaskProfilePosX, DefaultPrintTaskProfilePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
                "[ZR(Z) + X]\n"
                "  Total(us)      :%8d\n"
                "  OpusDecode(us) :%8d\n"
                "  OpusSampleCount:%8d\n"
                "  FsAccess(us)   :%8d\n"
                "  FsReadSize     :%8d\n"
                "  CacheLength    :%8u\n"
                "  RemainCache(%) :%8.2f"
                , data.GetTotalTime().GetMicroSeconds()
                , data.GetDecodeTime().GetMicroSeconds()
                , data.GetDecodedSampleCount()
                , data.GetFsAccessTime().GetMicroSeconds()
                , data.GetFsReadSize()
                , data.GetCachedLength()
                , data.GetRemainingCachePercentage()
            );

            if (data.GetCachedLength() != 0)
            {
                const int RemainCacheBarOffsetY = 130;
                int remainCacheBarUsedOffsetX = static_cast<int>(DebugViewer::DefaultDrawBarWidth * (100.f - data.GetRemainingCachePercentage()) / 100.f);
                DebugDrawQuad(
                    "Used",
                    DefaultPrintTaskProfilePosX,
                    DefaultPrintTaskProfilePosY + RemainCacheBarOffsetY,
                    DefaultPrintTaskProfilePosX + remainCacheBarUsedOffsetX,
                    DefaultPrintTaskProfilePosY + RemainCacheBarOffsetY + DebugViewer::DefaultDrawBarHeight,
                    GetDebugColor( 130, 130, 130 )
                );
                DebugDrawQuad(
                    "Remain",
                    DefaultPrintTaskProfilePosX + remainCacheBarUsedOffsetX,
                    DefaultPrintTaskProfilePosY + RemainCacheBarOffsetY,
                    DefaultPrintTaskProfilePosX + DebugViewer::DefaultDrawBarWidth,
                    DefaultPrintTaskProfilePosY + RemainCacheBarOffsetY + DebugViewer::DefaultDrawBarHeight,
                    GetDebugColor( 0, 0, 230 )
                );
            }

            break;
        }
        default:
        {
            NN_UNEXPECTED_DEFAULT;
            break;
        }
        }
    }
}

void AtkProfiler::DrawSoundThreadUpdateProfile(const nn::atk::SoundThreadUpdateProfile& profile) NN_NOEXCEPT
{
    DebugPrint("SoundThreadProfile", DefaultPrintSoundThreadProfilePosX, DefaultPrintSoundThreadProfilePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
        "[ZR(Z) + Y]\n"
        "  SoundThreadProc   (us):%8lld\n"
        "  LowLevelVoiceProc (us):%8lld\n"
        "  FrameProc         (us):%8lld\n"
        "  UserEffectProc    (us):%8lld\n"
        "  UpdateRendererProc(us):%8lld\n"
        "  WaitRendererProc  (us):%8lld"
        , profile.soundThreadProcess.GetSpan().GetMicroSeconds()
        , profile._updateLowLevelVoiceProcess.GetSpan().GetMicroSeconds()
        , profile._frameProcess.GetSpan().GetMicroSeconds()
        , profile._userEffectFrameProcess.GetSpan().GetMicroSeconds()
        , profile._updateRendererProcess.GetSpan().GetMicroSeconds()
        , profile._waitRendererEventProcess.GetSpan().GetMicroSeconds()
    );
}
#endif
