﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AtkMixMode.cpp,PageSampleAtkMixMode}
 *
 * @brief
 *  nn::atk ライブラリのミックスのボリューム設定の使い方を示すサンプルプログラム
 */

/**
 * @page PageSampleAtkMixMode ミックスのボリューム設定を用いた再生
 * @tableofcontents
 *
 * @brief
 *  nn::atk ライブラリのミックスのボリューム設定に関する機能のサンプルプログラムの解説です。
 *  SoundHandle 経由でミックスのボリューム設定を行います。
 *
 * @section PageSampleAtkMixMode_SectionBrief 概要
 *  Atk ライブラリでミックスのボリューム設定を行うサンプルです。
 *
 * @section PageSampleAtkMixMode_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AtkMixMode Samples/Sources/Applications/AtkMixMode @endlink 以下にあります。
 *
 * @section PageSampleAtkMixMode_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleAtkMixMode_SectionHowToOperate 操作方法
 *  実行中のログが、画面上とコンソールに表示され、
 *  起動直後のログとして操作方法が表示されます。
 *
 *  PC 版ではウィンドウは立ち上がらず、コンソール表示のみになる点にご注意ください。
 *
 * @section PageSampleAtkMixMode_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAtkMixMode_SectionDetail 解説
 *
 * @subsection PageSampleAtkMixMode_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AtkMixMode.cpp
 *  @includelineno AtkMixMode.cpp
 *
 * @subsection PageSampleAtkMixMode_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの Atk に関連している処理の流れは以下の通りです。
 *
 *  - アロケータの初期化
 *    - 内部で２種類（メモリプール管理対象用とそれ以外）のアロケータを初期化。
 *  - Atk ライブラリの初期化 (InitializeAtk())
 *    - AtkSimple で実行している初期化処理
 *    - サウンドシステムの出力モードをサラウンドに変更(追加要素)
 *  - SoundHeap にウェーブサウンドとシーケンスサウンドの再生に必要な情報をロード (LoadData())
 *  - メインループでの処理
 *    - キー操作による処理
 *      - サウンドの再生・停止処理を実行
 *      - ミックスのボリューム設定を変更するチャンネルの選択(追加要素)
 *      - ミックスのボリューム設定の変更(追加要素)
 *    - SoundArchivePlayer の更新
 *  - Atk ライブラリの終了処理 (FinalizeAtk())
 *
 *  キー操作を行うことで、サウンドのミックスのボリューム設定変更を行います。
 *  ステレオ波形のサウンドの場合、各チャンネルに対してミックスのボリューム設定が可能です。
 *  また、ストリームサウンドやシーケンスサウンドの場合は各トラックに対してもミックスのボリューム設定が可能です。
 *
 */

#include "Common.fsid"

#include <nn/atk.h>

#include <nns/atk/atk_SampleCommon.h>

#include <nns/nns_Log.h>

namespace
{
    const char Title[] = "AtkMixMode";
    const char ArchiveRelativePath[] = "Common.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;

    nn::atk::SoundHandle        g_SoundHandle;
    nn::atk::SoundHandle        g_SoundHandleHold;

    const float MixVolumeDelta   = 0.02f;
    const float MixVolumeDefault = 0.5f;
    const float MixVolumeMax     = 1.0f;
    const float MixVolumeMin     = 0.0f;

    int g_SelectedChannel;
    nn::atk::MixVolume g_MixVolume;

    void DumpMixVolume() NN_NOEXCEPT
    {
        NNS_LOG("(FL, FR, RL, RR, FC, LFE) = (%s%.2f, %s%.2f, %s%.2f, %s%.2f, %s%.2f, %s%.2f )\n"
        , (g_SelectedChannel == nn::atk::ChannelIndex_FrontLeft)   ? ">" : " " , g_MixVolume.frontLeft
        , (g_SelectedChannel == nn::atk::ChannelIndex_FrontRight)  ? ">" : " " , g_MixVolume.frontRight
        , (g_SelectedChannel == nn::atk::ChannelIndex_RearLeft)    ? ">" : " " , g_MixVolume.rearLeft
        , (g_SelectedChannel == nn::atk::ChannelIndex_RearRight)   ? ">" : " " , g_MixVolume.rearRight
        , (g_SelectedChannel == nn::atk::ChannelIndex_FrontCenter) ? ">" : " " , g_MixVolume.frontCenter
        , (g_SelectedChannel == nn::atk::ChannelIndex_Lfe)         ? ">" : " " , g_MixVolume.lowFrequencyEffect);
    }
    void SetMixVolume() NN_NOEXCEPT
    {
        g_SoundHandle.SetMixMode(nn::atk::MixMode_MixVolume);
        g_SoundHandle.SetMixVolume(0, g_MixVolume);
        g_SoundHandle.SetMixVolume(1, g_MixVolume);
        DumpMixVolume();
    }
    void ResetMixVolume() NN_NOEXCEPT
    {
        g_MixVolume.frontLeft          = MixVolumeDefault;
        g_MixVolume.frontRight         = MixVolumeDefault;
        g_MixVolume.rearLeft           = MixVolumeMin;
        g_MixVolume.rearRight          = MixVolumeMin;
        g_MixVolume.frontCenter        = MixVolumeMin;
        g_MixVolume.lowFrequencyEffect = MixVolumeMin;
    }
}

void Initialize()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();
}

void Finalize()
{
    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeAtk()
{
    bool isSuccess = true;

    // SoundSystem の初期化
    nn::atk::SoundSystem::SoundSystemParam param;
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_pMemoryForSoundSystem = nns::atk::Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
    isSuccess = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>( g_pMemoryForSoundSystem ),
        memSizeForSoundSystem );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundSystem" );

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate( SoundHeapSize );
    isSuccess = g_SoundHeap.Create( g_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath );

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate(  infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
    isSuccess = g_SoundArchive.LoadHeader( g_pMemoryForInfoBlock, infoBlockSize );
    NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundDataManager = nns::atk::Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize( &g_SoundArchive );
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // SoundArchivePlayer の初期化
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
    isSuccess = g_SoundArchivePlayer.Initialize(
        &g_SoundArchive,
        &g_SoundDataManager,
        g_pMemoryForSoundArchivePlayer, memSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, memSizeForStreamBuffer );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );

    // 出力モードをサラウンドに変更
    nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode_Surround);

    // 音量のリセット
    ResetMixVolume();
}

void FinalizeAtk()
{
    g_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::Finalize();

    nns::atk::FreeForMemoryPool(g_pMemoryForStreamBuffer);
    nns::atk::Free(g_pMemoryForSoundArchivePlayer);
    nns::atk::Free(g_pMemoryForSoundDataManager);
    nns::atk::Free(g_pMemoryForInfoBlock);
    nns::atk::Free(g_pMemoryForSoundHeap);
    nns::atk::Free(g_pMemoryForSoundSystem);
}

void LoadData()
{
    bool isSuccess = true;

    isSuccess = g_SoundDataManager.LoadData( SEQ_MARIOKART, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_MARIOKART) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_YOSHI, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_YOSHI) failed." );
}

void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    g_SoundHandle.Stop( 0 );

    bool result = g_SoundArchivePlayer.StartSound( &g_SoundHandle, soundId ).IsSuccess();
    NNS_LOG("StartSound(%s) ... (%d)\n", debugLabelName, result);
}

void PlayWithHoldSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    bool result = g_SoundArchivePlayer.HoldSound( &g_SoundHandleHold, soundId ).IsSuccess();
    NNS_LOG("HoldSound(%s) ... (%d)\n", debugLabelName, result);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("[A]              StartSound SEQ   (SEQ_MARIOKART)\n");
    NNS_LOG("[X]              StartSound WSD   (SE_YOSHI)\n");
    NNS_LOG("[Y]              StartSound STRM  (STRM_MARIOKART)\n");
    NNS_LOG("[B]              Stop Sound\n");
    NNS_LOG("[RIGHT]          Select Channel   (right)\n");
    NNS_LOG("[LEFT]           Select Channel   (left)\n");
    NNS_LOG("[UP]             Change MixVolume (up)\n");
    NNS_LOG("[DOWN]           Change MixVolume (down)\n");
    NNS_LOG("[R]              Reset  MixVolume\n");
    NNS_LOG("[L]              Print Usage\n");
    NNS_LOG("[+/Start][Space] Exit Application\n");
    NNS_LOG("--------------------------------------------------\n");
}

bool UpdateAtk()
{
    // StartSound / Stop
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
    {
        PlayWithStartSound(SEQ_MARIOKART, "SEQ_MARIOKART");
        SetMixVolume();
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
    {
        PlayWithStartSound(SE_YOSHI, "SE_YOSHI");
        SetMixVolume();
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >() )
    {
        PlayWithStartSound(STRM_MARIOKART, "STRM_MARIOKART");
        SetMixVolume();
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
    {
        g_SoundHandle.Stop( 0 );
    }

    // Change MixVolume
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
    {
        g_SelectedChannel += 1;
        if ( g_SelectedChannel >= nn::atk::ChannelIndex_Count )
        {
            g_SelectedChannel = 0;
        }
        SetMixVolume();
    }
    else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
    {
        g_SelectedChannel -= 1;
        if ( g_SelectedChannel < 0 )
        {
            g_SelectedChannel = nn::atk::ChannelIndex_Count - 1;
        }
        SetMixVolume();
    }
    else if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Up >() )
    {
        g_MixVolume.channel[g_SelectedChannel] += MixVolumeDelta;
        if ( g_MixVolume.channel[g_SelectedChannel] > MixVolumeMax )
        {
            g_MixVolume.channel[g_SelectedChannel] = MixVolumeMax;
        }
        SetMixVolume();
    }
    else if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Down >() )
    {
        g_MixVolume.channel[g_SelectedChannel] -= MixVolumeDelta;
        if ( g_MixVolume.channel[g_SelectedChannel] < MixVolumeMin )
        {
            g_MixVolume.channel[g_SelectedChannel] = MixVolumeMin;
        }
        SetMixVolume();
    }

    // Print Usage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }
    else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::R >() )
    {
        ResetMixVolume();
        SetMixVolume();
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() )
    {
        return false;
    }

    g_SoundArchivePlayer.Update();

    return true;
}

extern "C" void nnMain()
{
    Initialize();
    InitializeAtk();

    LoadData();

    PrintUsage();

    for ( ;; )
    {
        nns::atk::UpdateHidDevices();

        if ( !UpdateAtk() )
        {
            break;
        }

        // Vsync の代わり
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    FinalizeAtk();
    Finalize();
}

