﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AtkFilter.cpp,PageSampleAtkFilter}
 *
 * @brief
 *  nn::atk ライブラリのフィルタの使い方を示すサンプルプログラム
 */

/**
 * @page PageSampleAtkFilter フィルタを用いた再生
 * @tableofcontents
 *
 * @brief
 *  nn::atk ライブラリのフィルタに関する機能のサンプルプログラムの解説です。
 *  SoundHandle や SoundPlayer にフィルタを適用します。
 *
 * @section PageSampleAtkFilter_SectionBrief 概要
 *  Atk ライブラリのフィルタを用いて音声を再生するサンプルです。
 *
 * @section PageSampleAtkFilter_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AtkFilter Samples/Sources/Applications/AtkFilter @endlink 以下にあります。
 *
 * @section PageSampleAtkFilter_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleAtkFilter_SectionHowToOperate 操作方法
 *  コンソールに操作方法が表示されます。
 *
 * @section PageSampleAtkFilter_SectionPrecaution 注意事項
 *  実行中のログが、画面上とコンソールに表示され、
 *  起動直後のログとして操作方法が表示されます。
 *
 *  PC 版ではウィンドウは立ち上がらず、コンソール表示のみになる点にご注意ください。
 *
 * @section PageSampleAtkFilter_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAtkFilter_SectionDetail 解説
 *
 * @subsection PageSampleAtkFilter_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AtkFilter.cpp
 *  @includelineno AtkFilter.cpp
 *
 * @subsection PageSampleAtkFilter_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの Atk に関連している処理の流れは以下の通りです(AtkSimple と違う部分には追加要素と書かれています)。
 *
 *  - Atk ライブラリの初期化 (InitializeAtk())
 *    - AtkSimple で実行している初期化処理
 *    - オーディオレンダラのレンダリングサンプルレートを 32000 に変更(追加要素)
 *  - SoundHeap にウェーブサウンドとシーケンスサウンドの再生に必要な情報をロード (LoadData())
 *  - メインループでの処理
 *    - キー操作による処理
 *      - サウンドの再生・停止処理を実行
 *      - 適用するフィルターの種類の変更(追加要素)
 *      - 適用したフィルタのかかり具合を変更(追加要素)
 *    - SoundArchivePlayer の更新
 *  - Atk ライブラリの終了処理 (FinalizeAtk())
 *
 *  キー操作を行うことで、サウンドにフィルタを適用することができます。
 *  適用することのできるフィルタは、以下の 6 種類となります。
 *  - ローパスフィルタ
 *  - ローパスフィルタ(双極フィルタ)
 *  - ハイパスフィルタ(双極フィルタ)
 *  - バンドパスフィルタ(双極フィルタ, 中心周波数 512 Hz)
 *  - バンドパスフィルタ(双極フィルタ, 中心周波数 1024 Hz)
 *  - バンドパスフィルタ(双極フィルタ, 中心周波数 2048 Hz)
 *  この 6 種類のフィルタを、SoundHandle および SoundPlayer に対して適用することができます。
 *  また、サンプルでは取り扱っていませんが SetLowPassFilterFrequency と SetBiquadFilter で適用する 2 つのフィルタは併用可能となっており、
 *  併用した場合には 2 つのフィルタの設定が重ねがけされます。
 *
 */

#include "Common.fsid"

#include <nn/atk.h>

#include <nns/atk/atk_SampleCommon.h>

#include <nns/nns_Log.h>

namespace
{
    const char Title[] = "AtkFilter";
    const char ArchiveRelativePath[] = "Common.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;

    nn::atk::SoundHandle        g_SoundHandle;
    nn::atk::SoundHandle        g_SoundHandleHold;

    float g_BiquadFilterValue;
    const float DefaultBiquadFilterValue = 0.8f;
    float g_LowPassFilterValue;
    const float DefaultLowPassFilterValue = -0.3f;

    const float ChangeFilterValue = 0.1f;
    const float LowPassFilterValueMin = -1.0f;
    const float LowPassFilterValueMax = 0.0f;
    const float BiquadFilterValueMin = 0.0f;
    const float BiquadFilterValueMax = 1.0f;
    const float RoundFilterValue = 0.001f;

    enum FilterType
    {
        FilterType_None,
        FilterType_HandleLowPassFilter,
        FilterType_HandleBiquadLowPassFilter,
        FilterType_HandleBiquadHighPassFilter,
        FilterType_HandleBiquadBandPassFilter512,
        FilterType_HandleBiquadBandPassFilter1024,
        FilterType_HandleBiquadBandPassFilter2048,
        FilterType_PlayerLowPassFilter,
        FilterType_PlayerBiquadLowPassFilter,
        FilterType_PlayerBiquadHighPassFilter,
        FilterType_PlayerBiquadBandPassFilter512,
        FilterType_PlayerBiquadBandPassFilter1024,
        FilterType_PlayerBiquadBandPassFilter2048,
        FilterType_Max = FilterType_PlayerBiquadBandPassFilter2048
    };

    int g_FilterType;
}

const char* GetFilterTypeString( int type )
{
    static const char* FilterType_String[] =
    {
        "NONE",
        "HANDLE_LPF",
        "HANDLE_BIQUAD_LPF",
        "HANDLE_BIQUAD_HPF",
        "HANDLE_BIQUAD_BPF_512",
        "HANDLE_BIQUAD_BPF_1024",
        "HANDLE_BIQUAD_BPF_2048",
        "PLAYER_LPF",
        "PLAYER_BIQUAD_LPF",
        "PLAYER_BIQUAD_HPF",
        "PLAYER_BIQUAD_BPF_512",
        "PLAYER_BIQUAD_BPF_1024",
        "PLAYER_BIQUAD_BPF_2048",
    };
    if ( type > FilterType_Max || type < 0 )
    {
        return "INVALID";
    }
    return FilterType_String[ type ];
}

void SetFilter()
{
    nn::atk::SoundPlayer& player = g_SoundArchivePlayer.GetSoundPlayer(PLAYER_DEFAULT);

    switch ( g_FilterType )
    {
    case FilterType_None:
        // ハンドルもプレイヤーもデフォルト値
        g_SoundHandle.SetLowPassFilterFrequency(LowPassFilterValueMax);
        g_SoundHandle.SetBiquadFilter( nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin );
        player.SetLowPassFilterFrequency(LowPassFilterValueMax);
        player.SetBiquadFilter( nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin );
        break;
    case FilterType_HandleLowPassFilter:
        // ハンドルの Low Pass Filter のみ設定
        g_SoundHandle.SetLowPassFilterFrequency(g_LowPassFilterValue);
        // 他はデフォルト値
        g_SoundHandle.SetBiquadFilter( nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin );
        player.SetLowPassFilterFrequency(LowPassFilterValueMax);
        player.SetBiquadFilter( nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin );
        break;
    case FilterType_HandleBiquadLowPassFilter:
    case FilterType_HandleBiquadHighPassFilter:
    case FilterType_HandleBiquadBandPassFilter512:
    case FilterType_HandleBiquadBandPassFilter1024:
    case FilterType_HandleBiquadBandPassFilter2048:
        // ハンドルの Biquad Filter のみ設定
        g_SoundHandle.SetBiquadFilter( g_FilterType - FilterType_HandleLowPassFilter, g_BiquadFilterValue );
        // 他はデフォルト値
        g_SoundHandle.SetLowPassFilterFrequency(LowPassFilterValueMax);
        player.SetLowPassFilterFrequency(LowPassFilterValueMax);
        player.SetBiquadFilter( nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin );
        break;
    case FilterType_PlayerLowPassFilter:
        // プレイヤーの Low Pass Filter のみ設定
        player.SetLowPassFilterFrequency(g_LowPassFilterValue);
        // 他はデフォルト値
        g_SoundHandle.SetLowPassFilterFrequency(LowPassFilterValueMax);
        g_SoundHandle.SetBiquadFilter( nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin );
        player.SetBiquadFilter( nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin );
        break;
    case FilterType_PlayerBiquadLowPassFilter:
    case FilterType_PlayerBiquadHighPassFilter:
    case FilterType_PlayerBiquadBandPassFilter512:
    case FilterType_PlayerBiquadBandPassFilter1024:
    case FilterType_PlayerBiquadBandPassFilter2048:
        // プレイヤーの Biquad Filter のみ設定
        player.SetBiquadFilter( g_FilterType - FilterType_PlayerLowPassFilter, g_BiquadFilterValue );
        // 他はデフォルト値
        g_SoundHandle.SetLowPassFilterFrequency(LowPassFilterValueMax);
        g_SoundHandle.SetBiquadFilter( nn::atk::BiquadFilterType_Inherit, BiquadFilterValueMin );
        player.SetLowPassFilterFrequency(LowPassFilterValueMax);
        break;
    default:
        NNS_LOG("Invalid filter type.\n");
        break;
    }

    NNS_LOG("filter( %-22s )   ", GetFilterTypeString( g_FilterType ));
    if ( g_FilterType == FilterType_None )
    {
        NNS_LOG("\n");
    }
    else if ( g_FilterType == FilterType_HandleLowPassFilter ||
              g_FilterType == FilterType_PlayerLowPassFilter )
    {
        NNS_LOG("value( %.1f )\n", g_LowPassFilterValue );
    }
    else
    {
        NNS_LOG("value( %.1f )\n", g_BiquadFilterValue );
    }
}

void Initialize()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();
}

void Finalize()
{
    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeAtk()
{
    bool isSuccess = true;

    // SoundSystem の初期化
    nn::atk::SoundSystem::SoundSystemParam param;
    param.rendererSampleRate = 32000; // Biquad フィルタのプリセットは 32kHz レンダリング用
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_pMemoryForSoundSystem = nns::atk::Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
    isSuccess = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>( g_pMemoryForSoundSystem ),
        memSizeForSoundSystem );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundSystem" );

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate( SoundHeapSize );
    isSuccess = g_SoundHeap.Create( g_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath );

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate(  infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
    isSuccess = g_SoundArchive.LoadHeader( g_pMemoryForInfoBlock, infoBlockSize );
    NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundDataManager = nns::atk::Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize( &g_SoundArchive );
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // SoundArchivePlayer の初期化
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
    isSuccess = g_SoundArchivePlayer.Initialize(
        &g_SoundArchive,
        &g_SoundDataManager,
        g_pMemoryForSoundArchivePlayer, memSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, memSizeForStreamBuffer );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );

    g_FilterType = FilterType_None;
    g_LowPassFilterValue = DefaultLowPassFilterValue;
    g_BiquadFilterValue = DefaultBiquadFilterValue;
    SetFilter();    // 初期状態表示のため
}

void FinalizeAtk()
{
    g_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::Finalize();

    nns::atk::FreeForMemoryPool(g_pMemoryForStreamBuffer);
    nns::atk::Free(g_pMemoryForSoundArchivePlayer);
    nns::atk::Free(g_pMemoryForSoundDataManager);
    nns::atk::Free(g_pMemoryForInfoBlock);
    nns::atk::Free(g_pMemoryForSoundHeap);
    nns::atk::Free(g_pMemoryForSoundSystem);
}

void LoadData()
{
    bool isSuccess = true;

    isSuccess = g_SoundDataManager.LoadData( SEQ_MARIOKART, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_MARIOKART) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_YOSHI, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_YOSHI) failed." );
}

void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    g_SoundHandle.Stop( 0 );

    bool result = g_SoundArchivePlayer.StartSound( &g_SoundHandle, soundId ).IsSuccess();
    NNS_LOG("StartSound(%s) ... (%d)\n", debugLabelName, result);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("[A]              StartSound SEQ  (SEQ_MARIOKART)\n");
    NNS_LOG("[X]              StartSound WSD  (SE_YOSHI)\n");
    NNS_LOG("[Y]              StartSound STRM (STRM_MARIOKART)\n");
    NNS_LOG("[B]              Stop Sound\n");
    NNS_LOG("[RIGHT/LEFT]     Change FilterType\n");
    NNS_LOG("[UP/DOWN]        Change FilterValue\n");
    NNS_LOG("[L]              Print Usage\n");
    NNS_LOG("[+/Start][Space] Exit Application\n");
    NNS_LOG("--------------------------------------------------\n");
}

bool UpdateAtk()
{
    // StartSound / Stop
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
    {
        PlayWithStartSound(SEQ_MARIOKART, "SEQ_MARIOKART");
        SetFilter();
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
    {
        PlayWithStartSound(SE_YOSHI, "SE_YOSHI");
        SetFilter();
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >() )
    {
        PlayWithStartSound(STRM_MARIOKART, "STRM_MARIOKART");
        SetFilter();
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
    {
        g_SoundHandle.Stop( 0 );
    }

    // Change Filter Type
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
    {
        g_FilterType -= 1;
        if ( g_FilterType < 0 )
        {
            g_FilterType = FilterType_Max;
        }
        SetFilter();
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
    {
        g_FilterType += 1;
        if ( g_FilterType > FilterType_Max )
        {
            g_FilterType = 0;
        }
        SetFilter();
    }

    // Change Filter Value
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
    {
        if ( g_FilterType == FilterType_None)
        {
        }
        else if( g_FilterType == FilterType_HandleLowPassFilter ||
                 g_FilterType == FilterType_PlayerLowPassFilter ) // Low Pass Filter
        {
            g_LowPassFilterValue += ChangeFilterValue;
            if ( g_LowPassFilterValue > LowPassFilterValueMax - RoundFilterValue )
            {
                g_LowPassFilterValue = LowPassFilterValueMax;
            }
            SetFilter();
        }
        else // Biquad Filter
        {
            g_BiquadFilterValue += ChangeFilterValue;
            if ( g_BiquadFilterValue > BiquadFilterValueMax - RoundFilterValue )
            {
                g_BiquadFilterValue = BiquadFilterValueMax;
            }
            SetFilter();
        }
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
    {
        if ( g_FilterType == FilterType_None)
        {
        }
        else if( g_FilterType == FilterType_HandleLowPassFilter ||
                 g_FilterType == FilterType_PlayerLowPassFilter ) // Low Pass Filter
        {
            g_LowPassFilterValue -= ChangeFilterValue;
            if ( g_LowPassFilterValue < LowPassFilterValueMin + RoundFilterValue )
            {
                g_LowPassFilterValue = LowPassFilterValueMin;
            }
            SetFilter();
        }
        else // Biquad Filter
        {
            g_BiquadFilterValue -= ChangeFilterValue;
            if ( g_BiquadFilterValue < BiquadFilterValueMin + RoundFilterValue )
            {
                g_BiquadFilterValue = BiquadFilterValueMin;
            }
            SetFilter();
        }
    }

    // Print Usage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() )
    {
        return false;
    }

    g_SoundArchivePlayer.Update();

    return true;
}

extern "C" void nnMain()
{
    Initialize();
    InitializeAtk();

    LoadData();

    PrintUsage();

    for ( ;; )
    {
        nns::atk::UpdateHidDevices();

        if ( !UpdateAtk() )
        {
            break;
        }

        // Vsync の代わり
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    FinalizeAtk();
    Finalize();
}

