﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AtkAddonSoundArchive.cpp,PageSampleAtkAddonSoundArchive}
 *
 * @brief
 *  追加サウンドアーカイブを用いた再生方法を示すサンプルプログラム
 */

/**
 * @page PageSampleAtkAddonSoundArchive 追加サウンドアーカイブを用いた再生のサンプル
 * @tableofcontents
 *
 * @brief
 *  nn::atk ライブラリの追加サウンドアーカイブを用いた再生方法を示すサンプルプログラムの解説です。
 *
 * @section PageSampleAtkAddonSoundArchive_SectionBrief 概要
 *  追加サウンドアーカイブを使ってサウンドの再生を行うサンプルです。
 *
 * @section PageSampleAtkAddonSoundArchive_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AtkAddonSoundArchive Samples/Sources/Applications/AtkAddonSoundArchive @endlink 以下にあります。
 *
 * @section PageSampleAtkAddonSoundArchive_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleAtkAddonSoundArchive_SectionHowToOperate 操作方法
 *  コンソールに操作方法が表示されます。
 *
 * @section PageSampleAtkAddonSoundArchive_SectionPrecaution 注意事項
 *  実行中のログが、画面上とコンソールに表示され、
 *  起動直後のログとして操作方法が表示されます。
 *
 *  PC 版ではウィンドウは立ち上がらず、コンソール表示のみになる点にご注意ください。
 *
 * @section PageSampleAtkAddonSoundArchive_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAtkAddonSoundArchive_SectionDetail 解説
 *
 * @subsection PageSampleAtkAddonSoundArchive_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AtkAddonSoundArchive.cpp
 *  @includelineno AtkAddonSoundArchive.cpp
 *
 * @subsection PageSampleAtkAddonSoundArchive_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの Atk に関連している処理の流れは以下の通りです(AtkSimple と違う部分には追加要素と書かれています)。
 *
 *  - Atk ライブラリの初期化 (InitializeAtk())
 *    - AtkSimple で実行している初期化処理
 *    - ラベル文字列の読み込み（追加要素）
 *    - SoundArchivePlayer の初期化時に、追加サウンドアーカイブの数を指定するように変更されています。(追加要素)
 *  - 追加サウンドアーカイブのセットアップ （SetupAddonSoundArchive()）
 *  - SoundHeap にウェーブサウンドとシーケンスサウンドの再生に必要な情報をロード (LoadData())
 *  - メインループでの処理
 *    - キー操作による処理
 *      - サウンドの再生・停止処理を実行
 *      - この際、対象のサウンドアーカイブを指定 (追加要素)
 *    - SoundArchivePlayer の更新
 *  - 追加サウンドアーカイブの破棄処理 （UnsetupAddonSoundArchive()）
 *  - Atk ライブラリの終了処理 (FinalizeAtk())
 *
 * キー操作を行うことで、メインのサウンドアーカイブに含まれるサウンド、および追加サウンドアーカイブに含まれるサウンドを再生します。
 * 追加サウンドアーカイブを用いた再生では、以下のサウンドを再生します。
 * - ウェーブサウンド
 * - ストリームサウンド
 * - 追加サウンドアーカイブに含まれるバンクを使用したシーケンスサウンド
 * - メインのサウンドアーカイブに含まれるバンクを使用したシーケンスサウンド
 *
 * 追加サウンドアーカイブの追加は、メインのサウンドアーカイブ初期化後から可能となります。
 * また、追加サウンドアーカイブは、必要がなくなったタイミングでメインのサウンドアーカイブから削除することができます。
 * その他、サンプルでは取り扱っていませんが、追加サウンドアーカイブを 2 つ以上併用して使用することも可能です。
 *
 */

#include "Base.fsid"
#include "Addon.fsid"

#include <nn/atk.h>

#include <nns/atk/atk_SampleCommon.h>

#include <nns/nns_Log.h>

namespace
{
    const char Title[] = "AtkAddonSoundArchive";
    const char ArchiveRelativePath[] = "Base.bfsar";
    const char AddonArchiveRelativePath[] = "Addon.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForLabelData;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;

    nn::atk::SoundHandle        g_SoundHandle;
    nn::atk::SoundHandle        g_SoundHandleHold;

    const char AddonSoundArchiveName[] = "AddonArchive";
    const int AddonSoundArchiveCount = 1;

    nn::atk::AddonSoundArchive g_AddonSoundArchive;
    nn::atk::SoundDataManager g_AddonSoundDataManager;
    void* g_pMemoryForAddonSoundArchive;
    void* g_pMemoryForAddonSoundDataManager;

}

void Initialize()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();
}

void Finalize()
{
    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeAtk()
{
    bool isSuccess = true;

    // SoundSystem の初期化
    nn::atk::SoundSystem::SoundSystemParam param;
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_pMemoryForSoundSystem = nns::atk::Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
    isSuccess = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>( g_pMemoryForSoundSystem ),
        memSizeForSoundSystem );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundSystem" );

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate( SoundHeapSize );
    isSuccess = g_SoundHeap.Create( g_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath );

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate(  infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
    isSuccess = g_SoundArchive.LoadHeader( g_pMemoryForInfoBlock, infoBlockSize );
    NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

    // SoundArchive のラベル文字列情報をメモリにロード
    std::size_t memSizeForLabelString = g_SoundArchive.GetLabelStringDataSize();
    g_pMemoryForLabelData = nns::atk::Allocate(memSizeForLabelString, nn::atk::FsSoundArchive::BufferAlignSize);
    g_SoundArchive.LoadLabelStringData(g_pMemoryForLabelData, memSizeForLabelString);

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundDataManager = nns::atk::Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize( &g_SoundArchive );
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // SoundArchivePlayer の初期化 （AddonSoundArchive の数を指定して初期化）
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize( &g_SoundArchive, 0, AddonSoundArchiveCount );
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize );

    nn::atk::SoundArchivePlayer::InitializeParam initializeParam;
    initializeParam.pSoundArchive = &g_SoundArchive;
    initializeParam.pSoundDataManager = &g_SoundDataManager;
    initializeParam.pSetupBuffer = g_pMemoryForSoundArchivePlayer;
    initializeParam.setupBufferSize = memSizeForSoundArchivePlayer;
    initializeParam.pStreamBuffer = g_pMemoryForStreamBuffer;
    initializeParam.streamBufferSize = memSizeForStreamBuffer;
    initializeParam.addonSoundArchiveCount = AddonSoundArchiveCount;
    isSuccess = g_SoundArchivePlayer.Initialize( initializeParam );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );
}

void FinalizeAtk()
{
    g_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::Finalize();

    nns::atk::FreeForMemoryPool(g_pMemoryForStreamBuffer);
    nns::atk::Free(g_pMemoryForSoundArchivePlayer);
    nns::atk::Free(g_pMemoryForSoundDataManager);
    nns::atk::Free(g_pMemoryForLabelData);
    nns::atk::Free(g_pMemoryForInfoBlock);
    nns::atk::Free(g_pMemoryForSoundHeap);
    nns::atk::Free(g_pMemoryForSoundSystem);
}

void SetupAddonSoundArchive()
{
    const char* addonArchiveAbsolutePath = nns::atk::GetAbsolutePath(AddonArchiveRelativePath);

    // メモリ上にアーカイブをロード
    nn::fs::FileHandle fileHandle;
    nn::fs::OpenFile(&fileHandle, addonArchiveAbsolutePath, nn::fs::OpenMode_Read);
    int64_t fileSize = 0;
    nn::Result result = nn::fs::GetFileSize(&fileSize, fileHandle);
    NN_ABORT_UNLESS( result.IsSuccess() );
    // メモリ上にロードしたアーカイブには波形が含まれているので、メモリプール管理されているヒープから確保する必要があります。
    g_pMemoryForAddonSoundArchive = nns::atk::AllocateForMemoryPool( static_cast<size_t>(fileSize) );
    size_t readSize = 0;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, g_pMemoryForAddonSoundArchive, static_cast<size_t>(fileSize));
    NN_ABORT_UNLESS( result.IsSuccess() );
    nn::fs::CloseFile(fileHandle);

    // 追加サウンドアーカイブの初期化
    g_AddonSoundArchive.Initialize(g_pMemoryForAddonSoundArchive);
    g_AddonSoundArchive.SetExternalFileRoot("content:/");

    // 追加サウンドアーカイブ用の SoundDataManager の初期化
    size_t memSizeForAddonSoundDataManager = g_AddonSoundDataManager.GetRequiredMemSize( &g_AddonSoundArchive );
    g_pMemoryForAddonSoundDataManager = nns::atk::Allocate( memSizeForAddonSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    bool isSuccess = g_AddonSoundDataManager.Initialize(
        &g_AddonSoundArchive,
        g_pMemoryForAddonSoundDataManager,
        memSizeForAddonSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // SoundArchivePlayer に登録
    g_SoundArchivePlayer.AddAddonSoundArchive(AddonSoundArchiveName, &g_AddonSoundArchive, &g_AddonSoundDataManager);
}

void UnsetupAddonSoundArchive()
{
    g_SoundArchivePlayer.RemoveAddonSoundArchive( &g_AddonSoundArchive );

    g_AddonSoundDataManager.Finalize();
    g_AddonSoundArchive.Finalize();

    nns::atk::Free( g_pMemoryForAddonSoundDataManager );
    nns::atk::FreeForMemoryPool( g_pMemoryForAddonSoundArchive );
}

void LoadData()
{
    bool isSuccess = true;

    isSuccess = g_SoundDataManager.LoadData( SEQ_BASE, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_BASE) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_YOSHI, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_YOSHI) failed." );

    isSuccess = g_SoundDataManager.LoadData( BANK_BASE2, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(BANK_BASE2) failed." );
}

void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* archiveName, const char* debugLabelName)
{
    g_SoundHandle.Stop( 0 );

    bool result = g_SoundArchivePlayer.StartSound( &g_SoundHandle, soundId, archiveName ).IsSuccess();

    NNS_LOG("StartSound(%s - %s) ... (%d)\n", debugLabelName, archiveName == nullptr ? "BaseArchive" : archiveName, result);
}

void PlayWithStartSound(const char* soundName, const char* archiveName)
{
    g_SoundHandle.Stop( 0 );

    bool result = g_SoundArchivePlayer.StartSound( &g_SoundHandle, soundName, archiveName ).IsSuccess();

    NNS_LOG("StartSound(%s - %s) ... (%d)\n", soundName, archiveName == nullptr ? "BaseArchive" : archiveName, result);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("[A]              StartSound(BaseArchive ) SEQ  (SEQ_BASE)\n");
    NNS_LOG("[X]              StartSound(BaseArchive ) WSD  (SE_YOSHI)\n");
    NNS_LOG("[Y]              StartSound(BaseArchive ) STRM (STRM_MARIOKART)\n");
    NNS_LOG("[R + A]          StartSound(AddonArchive) SEQ  (SEQ_ADDON)\n");
    NNS_LOG("[R + X]          StartSound(AddonArchive) WSD  (SE_WIHAHO)\n");
    NNS_LOG("[R + Y]          StartSound(AddonArchive) STRM (STRM_PIANO16)\n");
    NNS_LOG("[RIGHT]          StartSound with LabelString(AddonArchive) SEQ  (SEQ_ADDON)\n");
    NNS_LOG("[UP]             StartSound with LabelString(AddonArchive) WSD  (SE_WIHAHO)\n");
    NNS_LOG("[LEFT]           StartSound with LabelString(AddonArchive) STRM (STRM_PIANO16)\n");
    NNS_LOG("[DOWN]           StartSound with LabelString(AddonArchive) SEQ  (SEQ_ADDON_MAINBANK)\n");
    NNS_LOG("[B]              Stop Sound\n");
    NNS_LOG("[L]              Print Usage\n");
    NNS_LOG("[+/Start][Space] Exit Application\n");
    NNS_LOG("--------------------------------------------------\n");
}

bool UpdateAtk()
{
    // StartSound / Stop
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
    {
        if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::R >() )
        {
            PlayWithStartSound(SEQ_BASE, AddonSoundArchiveName, "SEQ_ADDON");
        }
        else
        {
            PlayWithStartSound(SEQ_BASE, nullptr, "SEQ_BASE");
        }
    }
    else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
    {
        if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::R >() )
        {
            PlayWithStartSound(SE_WIHAHO, AddonSoundArchiveName, "SE_WIHAHO");
        }
        else
        {
            PlayWithStartSound(SE_YOSHI, nullptr, "SE_YOSHI");
        }
    }
    else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >() )
    {
        if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::R >() )
        {
            PlayWithStartSound(STRM_PIANO16, AddonSoundArchiveName, "STRM_PIANO16");
        }
        else
        {
            PlayWithStartSound(STRM_MARIOKART, nullptr, "STRM_MARIOKART");
        }
    }
    else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
    {
        PlayWithStartSound("SEQ_ADDON", AddonSoundArchiveName);
    }
    else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
    {
        PlayWithStartSound("SE_WIHAHO", AddonSoundArchiveName);
    }
    else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
    {
        PlayWithStartSound("STRM_PIANO16", AddonSoundArchiveName);
    }
    else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
    {
        PlayWithStartSound( "SEQ_ADDON_MAINBANK", AddonSoundArchiveName );
    }
    else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
    {
        g_SoundHandle.Stop( 0 );
    }

    // Print Usage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() )
    {
        return false;
    }

    g_SoundArchivePlayer.Update();

    return true;
}

extern "C" void nnMain()
{
    Initialize();
    InitializeAtk();
    SetupAddonSoundArchive();

    LoadData();

    PrintUsage();

    for ( ;; )
    {
        nns::atk::UpdateHidDevices();

        if ( !UpdateAtk() )
        {
            break;
        }

        // Vsync の代わり
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    UnsetupAddonSoundArchive();
    FinalizeAtk();
    Finalize();
}

