﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "CustomSubmix.h"

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

namespace
{
    FlagList g_LocalFlagList(nullptr, 0);
}

void CustomSubmixCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    CommonObject::InitArg arg;
    m_CommonObject.GetAudioEngine().GetDefaultInitArg(arg.GetAudioEngineInitArg(), arg.GetAudioEnginePlatformInitArg());
    arg.GetAudioEnginePlatformInitArg().subMixCount = SubMixCount;
    arg.GetAudioEnginePlatformInitArg().mixBufferCount = SubMixCount * m_CommonObject.GetAudioEngine().GetRendererManager().GetChannelCount();
    m_CommonObject.Initialize(arg);

    for (int i = 0; i < SubMixCount; i++)
    {
        size_t bufferSize = nn::atk2::SubMix::GetRequiredMemorySize(1, ChannelCount);
        m_SubMixBuffer[i] = nns::atk::Allocate(bufferSize);
        bool result = m_SubMix[i].Initialize(&m_CommonObject.GetAudioEngine().GetRendererManager(), 1, ChannelCount, m_SubMixBuffer[i], bufferSize);
        if (!result)
        {
            NN_SDK_LOG("[atk2] Failed to initialize to submix[%d]\n",i);
        }
    }

    nn::atk2::detail::RendererManager* pRendererManager = &m_CommonObject.GetAudioEngine().GetRendererManager();
    nn::atk2::FinalMix* pFinalMix = &pRendererManager->GetFinalMix();
    m_SubMix[0].SetDestination(pRendererManager, &m_SubMix[1]);
    m_SubMix[1].SetDestination(pRendererManager, pFinalMix);
    m_SubMix[2].SetDestination(pRendererManager, pFinalMix);

    for (int i = 0; i < SubMixCount; i++)
    {
        for (int j = 0; j < ChannelCount; j++)
        {
            m_SubMix[i].SetMixVolumeImpl(0, j, 0, j, 1.0f);
        }
    }
}

void CustomSubmixCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    m_CommonObject.Finalize();
    for (int i = 0; i < SubMixCount; i++)
    {
        m_SubMix->Finalize(&m_CommonObject.GetAudioEngine().GetRendererManager());
        nns::atk::Free(m_SubMixBuffer[i]);
    }
}

void CustomSubmixCheckModule::OnLoadData() NN_NOEXCEPT
{
    m_CommonObject.LoadData();
}

void CustomSubmixCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    m_CommonObject.PrintUsage();

    NN_LOG("[Up]           Play Voice\n");
    NN_LOG("[Down]         Stop Voice\n");
    NN_LOG("[Left]         Free Voice\n");
    NN_LOG("[Right]        Alloc Voice\n");
}

void CustomSubmixCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    m_CommonObject.UpdateInput();

    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
    {
        m_Voice.SetVoiceState(nn::atk2::detail::VoiceState_Stop);
        NN_LOG("Set stop state.\n");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
    {
        m_Voice.AppendWaveBuffer(&m_WaveBuffer);
        m_Voice.SetVoiceState(nn::atk2::detail::VoiceState_Play);
        NN_LOG("Set play state.\n");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
    {
        m_Voice.SetVoiceState(nn::atk2::detail::VoiceState_Stop);
        m_Voice.Free();
        NN_LOG("Free voice.\n");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
    {
        bool result = m_Voice.AllocVoice(0, &m_CommonObject.GetAudioEngine());
        if (result)
        {
            // 波形のメタデータ
            m_Voice.SetSampleRate(48000);
            m_Voice.SetSampleFormat(nn::atk2::detail::SampleFormat_PcmS16);
            nn::atk2::detail::AdpcmParam adpcmParam;
            m_Voice.SetAdpcmParam(adpcmParam);

            // ボイスのパラメータ
            m_Voice.SetPriority(0);
            m_Voice.SetVolume(1.0f);
            m_Voice.SetPitch(1.0f);
            m_Voice.SetMonoFilter(false);
            m_Voice.SetBiquadFilter(false);
            //m_Voice.SetInterpolationType(0);
            m_Voice.SetOutputReceiver(&m_SubMix[0]);

            int sampleCount = 48000;
            size_t bufferSize = sampleCount * sizeof(int16_t);
            void* buffer = nns::atk::AllocateForMemoryPool(bufferSize, nn::atk2::detail::RendererManager::BufferAlignSize);
            int16_t* sampleArray = reinterpret_cast<int16_t*>(buffer);
            const float Pi = 3.1415926535897932384626433f;
            for (auto i = 0; i < sampleCount; ++i)
            {
                sampleArray[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * 440 * i / 48000));
            }

            m_WaveBuffer.bufferAddress = buffer;
            m_WaveBuffer.bufferSize = bufferSize;
            m_WaveBuffer.loopFlag = true;
            m_WaveBuffer.sampleLength = sampleCount;
            m_WaveBuffer.sampleOffset = 0;

            NN_LOG("Alloc voice.\n");
        }
        else
        {
            NN_LOG("Fail to alloc voice.\n");
        }
    }
}

void CustomSubmixCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    if (m_Voice.GetVoiceState() == nn::atk2::detail::VoiceState_Play)
    {
        m_Voice.UpdateParam();
    }
    m_CommonObject.Update();
}

void CustomSubmixCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
}

const char* CustomSubmixCheckModule::GetModuleName() const NN_NOEXCEPT
{
    return "CustomSubmix";
}

FlagList& CustomSubmixCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}
