﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AocRuntimeInstall.cpp,PageSampleAocRuntimeInstall}
 *
 * @brief
 *  追加コンテンツの、実行時追加読み込みサンプルプログラム
 */

/**
 * @page PageSampleAocRuntimeInstall 追加コンテンツの実行時追加読み込み
 * @tableofcontents
 *
 * @brief
 *  追加コンテンツが、アプリケーション実行中に追加される対応をするサンプルプログラムの解説です。
 *
 * @section PageSampleAocRuntimeInstall_SectionBrief 概要
 *  ここでは、追加コンテンツライブラリで、アプリケーションを終了せずに新しい追加コンテンツを利用する方法を紹介します。
 *
 *  このサンプルでは、主に新しい追加コンテンツが利用可能になったことを検出する方法の紹介となります。
 *  実際の読み込み処理は、AocSimple も参考にしてください。
 *
 *  @ref nn::aoc "追加コンテンツライブラリの関数リファレンス"
 *  @ref nn::fs "ファイルシステムライブラリの関数リファレンス"
 *
 * @section PageSampleAocRuntimeInstall_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AocRuntimeInstall Samples/Sources/Applications/AocRuntimeInstall @endlink 以下にあります。
 *
 * @section PageSampleAocRuntimeInstall_SectionNecessaryEnvironment 必要な環境
 *  ApplicationId が 0x0100cb3000062000 の追加コンテンツは、あらかじめ削除しておく必要があります。
 *  インストールされている場合は、UninstallAoc.bat を実行してください。
 *
 * @section PageSampleAocRuntimeInstall_SectionHowToOperate 操作方法
 *  アプリケーションを起動したあとで、InstallAoc.bat を実行してください。
 *
 * @section PageSampleAocRuntimeInstall_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 *  また、実行時追加読み込みを有効にするには、nmeta の RuntimeAddOnContentInstall に AllowAppend を指定する必要があります。
 *
 * @section PageSampleAocRuntimeInstall_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *  実行した状態で、InstallAoc.bat を実行してください。
 *
 * @section PageSampleAocRuntimeInstall_SectionDetail 解説
 *
 * @subsection PageSampleAocRuntimeInstall_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AocRuntimeInstall.cpp
 *  @includelineno AocRuntimeInstall.cpp
 *
 * @subsection PageSampleAocRuntimeInstall_SectionSampleDetail サンプルプログラムの解説
 *
 *  このプログラムでは、利用可能な追加コンテンツに変化があったことを検出するイベントを取得し、
 *  それを利用して新しく追加された追加コンテンツの情報を検出、列挙しなおすということを行います。
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  AocRuntimeInstall_OutputExample.txt
 *
 *  実際には、TargetManager 上では AocInstall.bat を実行したときのログも出力されます。
 *
 *  このプログラムは、以下の処理を行います。
 *
 *  - 追加コンテンツの変化を検出するイベントの取得(@ref nn::aoc::GetAddOnContentListChangedEvent "nn::aoc::GetAddOnContentListChangedEvent")
 *  - 利用可能な追加コンテンツリストアップ
 *  - イベントを用いて、新たな追加コンテンツが利用できるようになったことを待つ
 *  - イベントが発生したら、再度リストアップを行う
 *
 *  それぞれの関数の詳細については、関数リファレンスを参照してください。
 *  また、追加コンテンツの中身を実際に参照する処理は、AocSimple のサンプルを参照してください。
 *
 */

#include <new>
#include <nn/aoc.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>

namespace {
    // 追加コンテンツインデックスを列挙するためのバッファ
    const int MaxListupCount = 256;
    nn::aoc::AddOnContentIndex g_AocListupBuffer[MaxListupCount];
}

void ListupAddOnContent()
{
    // 追加コンテンツ数の取得
    int aocCount = nn::aoc::CountAddOnContent();
    NN_LOG("CountAddOnCountent -> %d\n", aocCount);

    // 追加コンテンツのリストアップ
    int listupCount = nn::aoc::ListAddOnContent(g_AocListupBuffer, 0, MaxListupCount);
    NN_LOG("ListAddOnCountent  -> %d\n", listupCount);
    for (int i = 0; i < listupCount; ++i)
    {
        NN_LOG("  Index[%d]: %d\n", i, g_AocListupBuffer[i]);
    }
}

extern "C" void nnMain()
{
    nn::Result result;
    NN_LOG("AocRuntimeInstall\n");

    // 変化を通知するイベントの取得
    nn::os::SystemEvent event;
    nn::aoc::GetAddOnContentListChangedEvent(&event);

    // 取りこぼしをなくすために、イベント取得後に列挙を行う
    ListupAddOnContent();

    NN_LOG("Wait list changed event. Please execute InstallAoc.bat\n");
    event.Wait();

    NN_LOG("Detect list change. Listup again\n");
    ListupAddOnContent();
}
