﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{AlbumSaveScreenshot_Main.cpp,PageSampleAlbumSaveScreenshot}

    @brief @copybrief PageSampleAlbumSaveScreenshot
 */

/**
    @page PageSampleAlbumSaveScreenshot アプリケーションが生成した画像を画面写真として保存・ SNS 投稿する

    @tableofcontents

    @brief アプリケーションが生成した画像を画面写真としてアルバムに保存する方法および保存して SNS 投稿する方法を示します。

    @section PageSampleAlbumSaveScreenshot_SectionBrief 概要
    アルバムライブラリをアプリケーションから使用する方法を示します。

    @section PageSampleAlbumSaveScreenshot_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/AlbumSaveScreenshot
    Samples/Sources/Applications/AlbumSaveScreenshot @endlink 以下にあります。

    @section PageSampleAlbumSaveScreenshot_SectionNecessaryEnvironment 必要な環境
    本サンプルプログラムは開発機環境でビルドして実行することができます。

    @section PageSampleAlbumSaveScreenshot_SectionHowToOperate 操作方法
    一般的なサンプルプログラムと同様に本プログラムをビルドし、実行してください。

    サンプルプログラムは以下を順番に実行した後、終了します。

    - アルバムへの画面写真の保存
    - アルバムへ画面写真を保存して SNS 投稿

    @section PageSampleAlbumSaveScreenshot_SectionDetail 解説
    このサンプルプログラムは次の順序で処理を行います。

    - アルバムへの画面写真の保存
        - アルバムライブラリの初期化
        - 保存する画像データの生成
        - 画像データを画面写真として保存
        - アルバムライブラリの終了
    - アルバムへ画面写真を保存して SNS 投稿
        - アルバムライブラリの初期化
        - 保存する画像データの生成
        - 画像データを画面写真として保存して SNS 投稿
        - アルバムライブラリの終了

    下記ソースコードの随所に補足説明を記載していますので、詳細はそちらを参照してください。

    AlbumSaveScreenshot_Main.cpp
    @includelineno AlbumSaveScreenshot_Main.cpp

    本サンプルで使用している nn::album::SaveScreenshot() では
    記録メディアへの静止画像ファイルの書き込みが行なわれます。

    記録メディアへの書き込み回数や書き込み容量に関しては、
    別途ガイドラインが規定されていますので、
    そちらに注意して API を不必要に乱発しないようにして下さい。

    なお、書き込み頻度を調査するためのデバッグ機能も別途提供されています。
    詳細は「@confluencelink{174228221,FS ライブラリ／デバッグ機能／アクセス頻度確認}」を参照して下さい。

 */

#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/album/album_Result.h>
#include <nn/album/album_SaveScreenshot.h>

// 保存する画面写真のサイズ。
// サポートされる画面写真のサイズはファームウェアにより決められています。
const nn::album::ImageSize ScreenShotSize = nn::album::ImageSize_1280x720;

// 保存結果の通知設定。
// 画像をアルバムに保存した結果をユーザーに自動的に通知する条件を選ぶことができます。
// 自動的に通知しなかった場合でも API の戻り値によってアプリケーションが独自の通知を行うことができます。
const nn::album::AlbumReportOption ReportOption = nn::album::AlbumReportOption_ReportAlways;

namespace {

    // 画面写真として保存する画像データを生成します。
    // 画像データの色フォーマットは R8G8B8A8 ですが、アルファチャンネルは無視されます。
    uint8_t* CreateImageData(size_t* pOutSize, int width, int height) NN_NOEXCEPT
    {
        const int PixelSize = 4;
        size_t size = PixelSize * width * height;
        uint8_t* pImage = reinterpret_cast<uint8_t*>(malloc(size));
        NN_ASSERT_NOT_NULL(pImage);

        for(int y = 0; y < height; y++)
        {
            uint8_t* pLine = pImage + PixelSize * width* y;
            for(int x = 0; x < width; x++)
            {
                uint8_t* pPixel = pLine + PixelSize * x;
                int v = ((x / 32) + (y / 32)) % 2;
                if(v == 0)
                {
                    pPixel[0] = 255; // R
                    pPixel[1] = 255; // G
                    pPixel[2] = 255; // B
                    pPixel[3] = 255; // A （無視される）
                }
                else
                {
                    // +---------+---------+
                    // |   Red   |  Green  |
                    // +---------+---------+
                    // |   Blue  |  Black  |
                    // +---------+---------+
                    if(y < height / 2)
                    {
                        if(x < width / 2)
                        {
                            pPixel[0] = 255; // R
                            pPixel[1] = 0;   // G
                            pPixel[2] = 0;   // B
                            pPixel[3] = 255; // A （無視される）
                        }
                        else
                        {
                            pPixel[0] = 0;   // R
                            pPixel[1] = 255; // G
                            pPixel[2] = 0;   // B
                            pPixel[3] = 255; // A （無視される）
                        }
                    }
                    else
                    {
                        if(x < width / 2)
                        {
                            pPixel[0] = 0;   // R
                            pPixel[1] = 0;   // G
                            pPixel[2] = 255; // B
                            pPixel[3] = 255; // A （無視される）
                        }
                        else
                        {
                            pPixel[0] = 0;   // R
                            pPixel[1] = 0;   // G
                            pPixel[2] = 0;   // B
                            pPixel[3] = 255; // A （無視される）
                        }
                    }
                }

            }
        }

        *pOutSize = size;
        return pImage;
    }

    void DestroyImageData(uint8_t* pImage) NN_NOEXCEPT
    {
        free(pImage);
    }
}

// アルバムへ画面写真を保存します
void ExecuteSaveScreenshot() NN_NOEXCEPT
{
    NN_LOG("Sample: SaveScreenShot\n");
    // アルバムライブラリを初期化します。
    nn::album::Initialize();

    int width = nn::album::GetImageWidth(ScreenShotSize);
    int height = nn::album::GetImageHeight(ScreenShotSize);
    NN_LOG("Screenshot width = %d, height = %d\n", width, height);

    // 保存する画像データを生成します。
    size_t imageDataSize = 0;
    auto pImageData = CreateImageData(&imageDataSize, width, height);

    {
        NN_LOG("SaveScreenshot\n");
        auto result = nn::album::SaveScreenshot(
            pImageData,
            imageDataSize,
            ScreenShotSize,
            ReportOption
        );
        if(result.IsFailure())
        {
            if(nn::album::ResultAlbumUnknown::Includes(result))
            {
                if(nn::album::ResultAlbumError::Includes(result))
                {
                    // 画面写真の作成や保存に失敗した場合にこのエラーが返ります。
                    NN_LOG("  -> Screenshot Failed(AlbumError)\n");
                }
                else if(nn::album::ResultAlbumFull::Includes(result))
                {
                    // 画面写真を保存するストレージが満杯だった場合にこのエラーが返ります。
                    NN_LOG("  -> Screenshot Failed(AlbumFull)\n");
                }
                else if(nn::album::ResultAlbumFileCountLimit::Includes(result))
                {
                    // 画面写真の数がアルバムに保存できる最大数に達している場合にこのエラーが返ります。
                    NN_LOG("  -> Screenshot Failed(AlbumFileCountLimit)\n");
                }
                else
                {
                    NN_LOG("  -> Screenshot Failed(Unknown)\n");
                }
            }
            else if(nn::album::ResultNotSupported::Includes(result))
            {
                NN_LOG("  -> NotSupported\n");
            }
            else
            {
                NN_LOG("  -> Error(Unknown)\n");
            }
            return;
        }
    }
    NN_LOG("  -> OK\n");

    // API 呼出後、保存した画像データは破棄できます。
    DestroyImageData(pImageData);

    // アルバムライブラリを終了します。
    nn::album::Finalize();

    // 保存結果の自動通知が消えるのを待ちます。
    // このサンプルでは連続で保存を行うため、通知が消えるまで待っていますが、
    // アプリケーションが保存 API を使用する場合にはスリープする必要はありません。
    NN_LOG("Waiting for 5 sec ...\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
}

// アルバムへ画面写真を保存します（画像の回転指定つき）
void ExecuteSaveScreenshotWithOrientaiton(nn::album::ImageOrientation orientaiton) NN_NOEXCEPT
{
    NN_LOG("Sample: SaveScreenShot\n");
    // アルバムライブラリを初期化します。
    nn::album::Initialize();

    int width = nn::album::GetImageWidth(ScreenShotSize);
    int height = nn::album::GetImageHeight(ScreenShotSize);
    NN_LOG("Screenshot width = %d, height = %d\n", width, height);

    switch(orientaiton)
    {
    case nn::album::ImageOrientation_None:
        NN_LOG("ImageOrientation: None\n");
        break;
    case nn::album::ImageOrientation_Rotate90:
        NN_LOG("ImageOrientation: Rotate90\n");
        break;
    case nn::album::ImageOrientation_Rotate180:
        NN_LOG("ImageOrientation: Rotate180\n");
        break;
    case nn::album::ImageOrientation_Rotate270:
        NN_LOG("ImageOrientation: Rotate270\n");
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }

    // 保存する画像データを生成します。
    size_t imageDataSize = 0;
    auto pImageData = CreateImageData(&imageDataSize, width, height);

    {
        NN_LOG("SaveScreenshot\n");
        auto result = nn::album::SaveScreenshot(
            pImageData,
            imageDataSize,
            ScreenShotSize,
            orientaiton,
            ReportOption
        );
        if(result.IsFailure())
        {
            if(nn::album::ResultAlbumUnknown::Includes(result))
            {
                if(nn::album::ResultAlbumError::Includes(result))
                {
                    // 画面写真の作成や保存に失敗した場合にこのエラーが返ります。
                    NN_LOG("  -> Screenshot Failed(AlbumError)\n");
                }
                else if(nn::album::ResultAlbumFull::Includes(result))
                {
                    // 画面写真を保存するストレージが満杯だった場合にこのエラーが返ります。
                    NN_LOG("  -> Screenshot Failed(AlbumFull)\n");
                }
                else if(nn::album::ResultAlbumFileCountLimit::Includes(result))
                {
                    // 画面写真の数がアルバムに保存できる最大数に達している場合にこのエラーが返ります。
                    NN_LOG("  -> Screenshot Failed(AlbumFileCountLimit)\n");
                }
                else
                {
                    NN_LOG("  -> Screenshot Failed(Unknown)\n");
                }
            }
            else if(nn::album::ResultNotSupported::Includes(result))
            {
                NN_LOG("  -> NotSupported\n");
            }
            else
            {
                NN_LOG("  -> Error(Unknown)\n");
            }
            return;
        }
    }
    NN_LOG("  -> OK\n");

    // API 呼出後、保存した画像データは破棄できます。
    DestroyImageData(pImageData);

    // アルバムライブラリを終了します。
    nn::album::Finalize();

    // 保存結果の自動通知が消えるのを待ちます。
    // このサンプルでは連続で保存を行うため、通知が消えるまで待っていますが、
    // アプリケーションが保存 API を使用する場合にはスリープする必要はありません。
    NN_LOG("Waiting for 5 sec ...\n");
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));
}

// アルバムへ画面写真を保存して SNS 投稿を行います
void ExecuteSaveAndShareScreenshot() NN_NOEXCEPT
{
    NN_LOG("Sample: SaveAndShareScreenShot\n");

    // アルバムライブラリを初期化します。
    nn::album::Initialize();

    int width = nn::album::GetImageWidth(ScreenShotSize);
    int height = nn::album::GetImageHeight(ScreenShotSize);
    NN_LOG("Screenshot width = %d, height = %d\n", width, height);

    // 保存する画像データを生成します。
    size_t imageDataSize = 0;
    auto pImageData = CreateImageData(&imageDataSize, width, height);

    // 投稿に使用するアカウントを指定します。
    // Uid として nn::account::InvalidUid を指定した場合、投稿時にユーザー選択画面が表示されます。
    // アプリケーション内で既にアカウントが選択されている場合、
    // 選択済の Uid を指定することで投稿時のユーザー選択画面を省略して投稿画面が表示されます。
    nn::account::Uid uid = nn::account::InvalidUid;

    {
        NN_LOG("SaveAndShareScreenshot\n");
        auto result = nn::album::SaveAndShareScreenshot(
            pImageData,
            imageDataSize,
            ScreenShotSize,
            ReportOption,
            uid
        );
        if(result.IsFailure())
        {
            if(nn::album::ResultAlbumUnknown::Includes(result))
            {
                if(nn::album::ResultAlbumError::Includes(result))
                {
                    // 画面写真の作成や保存に失敗した場合にこのエラーが返ります。
                    NN_LOG("  -> Screenshot Failed(AlbumError)\n");
                }
                else if(nn::album::ResultAlbumFull::Includes(result))
                {
                    // 画面写真を保存するストレージが満杯だった場合にこのエラーが返ります。
                    NN_LOG("  -> Screenshot Failed(AlbumFull)\n");
                }
                else if(nn::album::ResultAlbumFileCountLimit::Includes(result))
                {
                    // 画面写真の数がアルバムに保存できる最大数に達している場合にこのエラーが返ります。
                    NN_LOG("  -> Screenshot Failed(AlbumFileCountLimit)\n");
                }
                else
                {
                    NN_LOG("  -> Screenshot Failed(Unknown)\n");
                }
            }
            else if(nn::album::ResultShareUnknown::Includes(result))
            {
                if(nn::album::ResultShareCanceled::Includes(result))
                {
                    // ユーザー操作によって投稿がキャンセルされた場合にこのエラーが返ります。
                    NN_LOG("  -> Share Failed(ShareCanceled)\n");
                }
                else if(nn::album::ResultShareFailed::Includes(result))
                {
                    // 投稿中にエラーが発生した場合にこのエラーが返ります。
                    NN_LOG("  -> Share Failed(ShareFailed)\n");
                }
                else
                {
                    NN_LOG("  -> Share Failed(Unknown)\n");
                }
            }
            else if(nn::album::ResultNotSupported::Includes(result))
            {
                NN_LOG("  -> NotSupported\n");
            }
            else
            {
                NN_LOG("  -> Error(Unknown)\n");
            }
            return;
        }
    }
    NN_LOG("  -> OK\n");

    // API 呼出後、保存した画像データは破棄できます。
    DestroyImageData(pImageData);

    // アルバムライブラリを終了します。
    nn::album::Finalize();

}


extern "C" void nnMain() NN_NOEXCEPT
{
    NN_LOG("Start sample program\n");

    // 画像を画面写真として保存する機能を実行します。
    ExecuteSaveScreenshot();

    // 画像を画面写真として保存する機能を実行します（回転つき）
    ExecuteSaveScreenshotWithOrientaiton(nn::album::ImageOrientation_None);
    ExecuteSaveScreenshotWithOrientaiton(nn::album::ImageOrientation_Rotate90);
    ExecuteSaveScreenshotWithOrientaiton(nn::album::ImageOrientation_Rotate180);
    ExecuteSaveScreenshotWithOrientaiton(nn::album::ImageOrientation_Rotate270);

    // 画像を画面写真として保存して投稿する機能を実行します。
    ExecuteSaveAndShareScreenshot();

    NN_LOG("Complete sample program\n");
}
