﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{AlbumFileAccess_Main.cpp,PageSampleAlbumFileAccess}

    @brief @copybrief PageSampleAlbumFileAccess
 */

/**
    @page PageSampleAlbumFileAccess アプリケーションが保存したアルバム画像をリストアップする

    @tableofcontents

    @brief アプリケーションが保存したアルバム画像のファイルをリストアップし、そのデータを読み込む方法を示します。

    @section PageSampleAlbumFileAccess_SectionBrief 概要
    アルバムライブラリをアプリケーションから使用する方法を示します。

    @section PageSampleAlbumFileAccess_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/AlbumFileAccess
    Samples/Sources/Applications/AlbumFileAccess @endlink 以下にあります。

    @section PageSampleAlbumFileAccess_SectionNecessaryEnvironment 必要な環境
    本サンプルプログラムは開発機環境でビルドして実行することができます。

    @section PageSampleAlbumFileAccess_SectionHowToOperate 操作方法
    一般的なサンプルプログラムと同様に本プログラムをビルドし、実行してください。

    本サンプル実行前に、予め
    Samples/Sources/Applications/AlbumSaveScreenshot にある
    「 @subpage PageSampleAlbumSaveScreenshot 」のサンプルを実行して頂くと、
    アルバム静止画ファイルがいくつか作成されます。

    その後に本サンプルを実行することにより、
    いくつかの静止画ファイルがリストアップされます。

    通常、nn::album にある API は、自アプリケーションプログラムを撮影した
    アルバムファイルにしかアクセスできませんが、上記サンプルと本サンプルは
    同一の ApplicationId となっているため、便宜上同一のアプリケーションとして
    みなされています。


    @section PageSampleAlbumFileAccess_SectionDetail 解説

    このサンプルプログラムは以下の処理が順番に実行されます。

    - アルバムから自プログラムを撮影した画像ファイルの nn::album::AlbumFileEntry のリストを取得
    - 取得した個々の nn::album::AlbumFileEntry を使ってファイル情報をロード
    - 個々のファイル情報をログに出力し、ファイルリスト分だけこれを繰り返す

    下記ソースコードの随所に補足説明を記載していますので、
    詳細はそちらを参照してください。

    AlbumFileAccess_Main.cpp
    @includelineno AlbumFileAccess_Main.cpp

    本サンプルでは記録メディアからのアルバムファイルの読み込みが行なわれます。

    記録メディアからの読み込み回数や読み込みサイズに関しては、
    別途ガイドラインが規定されていますので、
    そちらに注意して API を不必要に乱発しないようにして下さい。

    なお、読み込み頻度を調査するためのデバッグ機能も別途提供されています。
    詳細は「@confluencelink{174228221,FS ライブラリ／デバッグ機能／アクセス頻度確認}」を参照して下さい。

 */

#include <cstdlib>
#include <vector>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/album.h>
#include <nn/album/album_AlbumFileAccess.private.h>
#include <nn/err.h>

namespace {

    NN_ALIGNAS(4096) uint8_t g_ScreenShotImage[ nn::album::AlbumScreenShotImageDataSize ];
    NN_ALIGNAS(4096) uint8_t g_ScreenShotWorkBuffer[ nn::album::RequiredWorkMemorySizeToLoadImage ];
    NN_ALIGNAS(4096) uint8_t g_ThumbnailImage[ nn::album::AlbumThumbnailImageDataSize ];
    NN_ALIGNAS(4096) uint8_t g_ThumbnailWorkBuffer[ nn::album::RequiredWorkMemorySizeToLoadThumbnailImage ];

}   // namespace


// アルバムファイルをリストアップします。
void ExecuteGetAlbumFileList() NN_NOEXCEPT
{
    NN_LOG("Sample: AlbumFileAccess\n");

    // ファイルリストを受け取る配列を確保します。
    // 本サンプルでは、とりあえず最大 20,000 個受け取れるようにしておきます。
    // この場合 48(Byte) x 20,000 = 約 960 KByte の領域が必要となります。
    std::vector<nn::album::AlbumFileEntry> entryList;
    entryList.resize(20000);

    // アルバムの静止画ファイルのファイルリストを取得します。
    // ここの API を nn::album::GetAlbumMovieFileList() に変更すると、
    // アルバムの動画ファイルのファイルリストを取得できます。
    // エラー発生時は、以下のようにエラービューアへ渡して下さい。
    int fileCount = 0;
    auto result = nn::album::GetAlbumScreenshotFileList(&fileCount, entryList.data(), entryList.size());
    if (!result.IsSuccess())
    {
        NN_LOG("Failed to get album file list\n");
        // ここでエラービューアを起動する
        nn::err::ShowError(result);
        return;
    }
    NN_LOG("nn::album::GetAlbumScreenshotFileList(): count=%d\n", fileCount);

    // 取得したファイルリストに関して、個々のファイル情報をログに出力します。
    for (int index=0; index<fileCount; ++index)
    {
        auto& entry = entryList[index];

        // 画像情報をログで出力（前半）
        auto dateTime = nn::album::GetAlbumFileCreatedTime(entry);
        NN_LOG("[%5d] %04d/%02d/%02d %02d:%02d:%02d  ",
                index,
                dateTime.year, dateTime.month,  dateTime.day,
                dateTime.hour, dateTime.minute, dateTime.second);

        // 静止画画像データの取得
        // ここではサイズ情報しか使用しませんが、
        // g_ScreenShotImage[] バッファには展開後の画像データが入ります。
        int width;
        int height;
        {
            auto result = nn::album::LoadAlbumImage(&width, &height, g_ScreenShotImage, sizeof(g_ScreenShotImage), g_ScreenShotWorkBuffer, sizeof(g_ScreenShotWorkBuffer), entry);
            if (!result.IsSuccess())
            {
                // ここでエラービューアを起動する
                nn::err::ShowError(result);
                NN_LOG("Failed\n");
                continue;
            }
        }

        // サムネイル画像データの取得
        // ここではサイズ情報しか使用しませんが、
        // g_ThumbnailImage[] バッファには展開後の画像データが入ります。
        int thumbnailWidth;
        int thumbnailHeight;
        {
            auto result = nn::album::LoadAlbumThumbnailImage(&thumbnailWidth, &thumbnailHeight, g_ThumbnailImage, sizeof(g_ThumbnailImage), g_ThumbnailWorkBuffer, sizeof(g_ThumbnailWorkBuffer), entry);
            if (!result.IsSuccess())
            {
                // ここでエラービューアを起動する
                nn::err::ShowError(result);
                NN_LOG("Failed\n");
                continue;
            }
        }

        // 画像情報をログで出力（後半）
        NN_LOG("Image:%dx%d  Thumbnail:%dx%d\n", width, height, thumbnailWidth, thumbnailHeight);
    }
}

extern "C" void nnMain() NN_NOEXCEPT
{
    NN_LOG("\nStart sample program\n\n");

    // アルバムライブラリを初期化します。
    nn::album::Initialize();

    // アルバムファイルをリストアップする機能を呼び出します。
    ExecuteGetAlbumFileList();

    // アルバムライブラリを終了します。
    nn::album::Finalize();

    NN_LOG("\nComplete sample program\n\n");
}
