﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AccountSelection.h"

#include <nn/nn_Assert.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_Selector.h>
#include <nn/account/account_StateRetention.h>
#include <nn/settings/system/settings_SystemApplication.h>

namespace {
class SceneStart final
    : public RootSequence::SceneBase
{
private:
    typedef RootSequence::SceneBase Base;
    enum Item {
        Advanced = 0,
        Open = 1,
        PresetCount = 2,
    };

    UserStateHolder& m_Holder;
    bool m_IsResetRequired;

    static bool SelectAndOpenUser(nn::account::UserHandle* pOut) NN_NOEXCEPT
    {
        int count;
        nn::account::UserSelectionSettings settings = nn::account::DefaultUserSelectionSettings;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::ListOpenUsers(&count, settings.invalidUidList, std::extent<decltype(settings.invalidUidList)>::value));

        nn::account::Uid uid;
        auto r = nn::account::ShowUserSelector(&uid, settings);
        if (!r.IsSuccess())
        {
            NN_SDK_ASSERT(nn::account::ResultCancelledByUser::Includes(r));
            return false;
        }

        nn::account::UserHandle raw;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&raw, uid));
        *pOut = raw;
        return true;
    }

    static void SelectionHandler(int index, void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);

        auto& obj = *reinterpret_cast<SceneStart*>(data);
        if (index < PresetCount)
        {
            // Preset
            switch (index)
            {
            case Advanced:
                obj.EndScene(RootSequence::SceneKind::Advanced);
                return;
            case Open:
                {
                    nn::account::UserHandle raw;
                    auto r = SelectAndOpenUser(&raw);
                    if (!r) return;

                    obj.m_Holder.Add(raw);
                    obj.m_IsResetRequired = true;
                }
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        else
        {
            auto pos = index - PresetCount;
            auto raw = obj.m_Holder.At(pos);
            obj.m_Holder.Delete(raw);
            nn::account::CloseUser(raw);
            obj.m_IsResetRequired = true;
        }
    }

    void Reset() NN_NOEXCEPT
    {
        auto& components = *GetComponentsPtr();

        components.Reset("Sample: User Account State", nullptr, this);
        components.AddOption("[Advanced]", SelectionHandler);
        components.AddOption("Select and open user", SelectionHandler);

        nn::account::Uid users[nn::account::UserCountMax];
        auto count = m_Holder.ListUid(users, std::extent<decltype(users)>::value);
        for (auto i = 0; i < count; ++ i)
        {
            nn::account::Nickname nickname;
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&nickname, users[i]));

            char label[128];
            nn::util::SNPrintf(label, sizeof(label), "Close \"%s\"", nickname.name);
            components.AddOption(label, SelectionHandler);
        }
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_IsResetRequired)
        {
            Reset();
        }
    }

public:
    explicit SceneStart(UserStateHolder& holder) NN_NOEXCEPT
        : m_Holder(holder)
        , m_IsResetRequired(false)
    {
        Reset();
    }
};

class SceneAdvanced final
    : public RootSequence::SceneBase
{
private:
    typedef RootSequence::SceneBase Base;

    enum Item {
        Cancel = 0,
        OpenPreselected = 1,
        PopOpenUsers = 2,
        PushOpenUsers = 3,
    };

    UserStateHolder& m_Holder;

    static void SelectionHandler(int index, void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);

        auto& obj = *reinterpret_cast<SceneAdvanced*>(data);
        switch (index)
        {
        case Cancel:
            obj.EndScene(RootSequence::SceneKind::Start);
            break;
        case OpenPreselected:
            {
                nn::account::UserHandle raw;
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenPreselectedUser(&raw));
                obj.m_Holder.Add(raw);
                obj.EndScene(RootSequence::SceneKind::Start);
            }
            break;
        case PopOpenUsers:
            {
                int poppedCount;
                nn::account::UserHandle handles[nn::account::UserCountMax];
                nn::account::PopOpenUsers(&poppedCount, handles, std::extent<decltype(handles)>::value);
                for (auto i = 0; i < poppedCount; ++ i)
                {
                    obj.m_Holder.Add(handles[i]);
                }
                obj.EndScene(RootSequence::SceneKind::Start);
            }
            break;
        case PushOpenUsers:
            nn::account::PushOpenUsers();
            obj.m_Holder.Clear();
            obj.EndScene(RootSequence::SceneKind::Start);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void Reset() NN_NOEXCEPT
    {
        auto& components = *GetComponentsPtr();

        components.Reset("Sample: User Account State", "Advanced", this);
        components.AddOption("[Cancel]", SelectionHandler);
        components.AddOption("Open pre-selected user", SelectionHandler);
        components.AddOption("Pop state", SelectionHandler);
        components.AddOption("Push state", SelectionHandler);
    }

public:
    explicit SceneAdvanced(UserStateHolder& holder) NN_NOEXCEPT
        : m_Holder(holder)
    {
        Reset();
    }
};
} // ~namespace <anonymous>

// ---------------------------------------------------------------------------------------------
// UserStateHolder

void UserStateHolder::Add(const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    for (auto& h : m_Handles)
    {
        if (!h.isOpened)
        {
            h.Set(handle);
            return;
        }
    }
    NN_ABORT("Unreachable");
}
void UserStateHolder::Delete(const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    nn::account::Uid given;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&given, handle));

    for (auto& h : m_Handles)
    {
        if (!h.isOpened)
        {
            return;
        }

        nn::account::Uid uid;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&uid, h.raw));
        if (uid == given)
        {
            auto* end = m_Handles + std::extent<decltype(m_Handles)>::value;
            std::memmove(&h, &h + 1, (end - (&h + 1)) * sizeof(m_Handles[0]));
            (end - 1)->Reset();
            return;
        }
    }
    NN_ABORT("Unreachable");
}
void UserStateHolder::Clear() NN_NOEXCEPT
{
    for (auto& h : m_Handles)
    {
        if (!h.isOpened)
        {
            return;
        }
        h.Reset();
    }
}
nn::account::UserHandle UserStateHolder::At(int index) const NN_NOEXCEPT
{
    NN_ASSERT(index < std::extent<decltype(m_Handles)>::value);
    const auto& h = m_Handles[index];
    NN_ASSERT(h.isOpened);
    return h.raw;
}
int UserStateHolder::ListUid(nn::account::Uid buffer[], int bufferLength) const NN_NOEXCEPT
{
    int count = 0;
    for (auto& h : m_Handles)
    {
        if (!h.isOpened || count >= bufferLength)
        {
            break;
        }
        nn::account::Uid uid;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetUserId(&uid, m_Handles[count].raw));
        buffer[count] = uid;
        ++ count;
    }
    return count;
}

// ---------------------------------------------------------------------------------------------
// RootSequence

RootSequence::RootSequence(Window& window) NN_NOEXCEPT
    : SequenceBase(window)
    , m_CurrentScene(SceneKind::Start)
    , m_NextScene(SceneKind::Start)
{
    m_Scene.reset(new SceneStart(m_Holder));
    NotifyCurrentSceneUpdate();

    NNS_MIGRATION_LOG_LN("[Root] sequense created");
}
RootSequence::~RootSequence() NN_NOEXCEPT
{
}
void RootSequence::OnEntering(SceneKind next) NN_NOEXCEPT
{
    switch (next)
    {
    case SceneKind::Start:
        m_Scene.reset(new SceneStart(m_Holder));
        NNS_MIGRATION_LOG_LN("[Root] Scene entered: Start");
        break;

    case SceneKind::Advanced:
        m_Scene.reset(new SceneAdvanced(m_Holder));
        NNS_MIGRATION_LOG_LN("[Root] Scene entered: Advanced");
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    m_CurrentScene = next;
}
const Window::Components* RootSequence::GetCurrentSceneComponentsPtr() const NN_NOEXCEPT
{
    return static_cast<const SceneBase*>(m_Scene.get())->GetComponentsPtr();
}
void RootSequence::UpdateImpl() NN_NOEXCEPT
{
    if (m_SubSequence)
    {
        m_SubSequence.reset();
        m_NextScene = SceneKind::Start;
        NNS_MIGRATION_LOG_LN("[Root] Exit subsequence");
    }

    if (m_NextScene != m_CurrentScene)
    {
        OnEntering(m_NextScene);
        NotifyCurrentSceneUpdate();
    }

    m_Scene->Update();
    if (m_Scene->IsEnded())
    {
        m_NextScene = m_Scene->GetNextScene();
    }
}
SequenceBase* RootSequence::GetSubSequence() NN_NOEXCEPT
{
    return m_SubSequence.get();
}
