﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{AccountApplication.cpp,PageSampleAccountApplication}

    @brief @copybrief PageSampleAccountApplication
 */

/**
    @page PageSampleAccountApplication 基本機能のアプリケーションでの利用

    @tableofcontents

    @brief アカウントライブラリの基本的な機能について、アプリケーションから使用する方法を示します。

    @section PageSampleAccountApplication_SectionBrief 概要
    アカウントライブラリの基本的な機能について、アプリケーションから使用する方法を示します。
    アカウントライブラリの基本的な機能には次が含まれます。

    - ユーザーアカウントの選択機能
    - ユーザーアカウントの属性を取得, 変更する機能
    - ユーザーアカウントの Open, Close 機能
    - ユーザーのネットワークサービスアカウントを使用する機能

    @section PageSampleAccountApplication_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/AccountApplication
    Samples/Sources/Applications/AccountApplication @endlink 以下にあります。

    @section PageSampleAccountApplication_SectionNecessaryEnvironment 必要な環境
    本サンプルプログラムは開発機向けにのみビルドし実行することができます。
    Windows 環境をサポートしていないことに注意してください。

    @section PageSampleAccountApplication_SectionHowToOperate 操作方法
    一般的なサンプルプログラムと同様に本プログラムをビルドし、実行してください。
    このプログラムの実行の経過は画面への文字の描画、及び実行ログとして出力されます。

    @section PageSampleAccountApplication_SectionDetail 解説
    このサンプルプログラムは次の順序で処理を行います。

    - ユーザーアカウントの選択
    - ユーザーアカウントの属性の取得
    - ユーザーアカウントの Open
    - ユーザーアカウントの属性の変更
    - ネットワークサービスアカウントの使用
    - ユーザーアカウントの Close

    下記ソースコードの随所に補足説明を記載していますので、詳細はそちらを参照してください。

    AccountApplication.cpp
    @includelineno AccountApplication.cpp

    実行の結果、次のようなログが出力されます。

    AccountApplication_ExampleOutput.txt
    @verbinclude AccountApplication_ExampleOutput.txt
 */

#include "AccountApplication.h"

#include <cstdlib>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_Selector.h>

#include <nn/nn_Assert.h>
#include <nn/err.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_Result.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiNetworkConnection.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Tick.h>
#include <nn/util/util_ScopeExit.h>

namespace
{
// セーブデータサムネイル
const int ThumbnailWidth = 256;
const int ThumbnailHeight = 144;
const size_t ThumbnailBufferSize = ThumbnailWidth * ThumbnailHeight * sizeof(uint32_t);
NN_STATIC_ASSERT(ThumbnailBufferSize == 147456);
NN_ALIGNAS(4096) char g_ThumbnailBuffer[ThumbnailBufferSize];

// 適当なサムネイル画像を生成する
void CreateThumbnailImage(void* buffer, const int Width, const int Height) NN_NOEXCEPT
{
    NN_ASSERT(Width == ThumbnailWidth);
    NN_ASSERT(Height == ThumbnailHeight);
    for (auto s = 0; s < Height; ++ s)
    {
        const uint8_t R = static_cast<char>(((s + 1) * 255) / Height);
        for (auto t = 0; t < Width; ++ t)
        {
            const uint8_t G = static_cast<char>(((t + 1) * 255) / Width);
            const uint8_t B = 0;
            const uint8_t A = 255;

            uint8_t* pixel = reinterpret_cast<uint8_t*>(buffer) + ((s * Width + t) * 4);
            pixel[0] = R;
            pixel[1] = G;
            pixel[2] = B;
            pixel[3] = A;
        }
    }
}
} // 無名名前空間終わり

// アプリケーションのメイン
void Program::Execute() NN_NOEXCEPT
{
    // 依存するモジュールの初期化
    nn::nifm::Initialize();

    // アカウントライブラリの初期化
    nn::account::Initialize();

    // ユーザーアカウントの選択画面の表示
    // - ユーザー操作によるキャンセル時には nn::account::ResultCancelledByUser が返る。
    nn::account::Uid uid;
    auto r = nn::account::ShowUserSelector(&uid);
    if (nn::account::ResultCancelledByUser::Includes(r))
    {
        Printfln("- User selection cancelled");
        return;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS(r);

    // セーブデータ領域の初期化
    r = nn::fs::EnsureSaveData(uid);
    if (!r.IsSuccess())
    {
        // nn::fs::EnsureSaveData() は内部でエラービューアを呼ぶため表示不要
        return;
    }

    // 選択されたユーザーアカウントのニックネームの取得
    // - ユーザーアカウントのニックネームやアイコンは Open せずに参照できる。
    nn::account::Nickname name;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&name, uid));
    Printfln("- User to open: \"%s\"", name.name);

    // アプリケーションのユーザー向けシーンを開始
    SceneSingleUser(uid);
}

// アプリケーションのユーザー向けシーン
void Program::SceneSingleUser(const nn::account::Uid& uid) NN_NOEXCEPT
{
    // 選択されたユーザーを Open
    // - ユーザーが使用中のユーザーアカウントを Open することで、フレンドプレゼンスが更新されるなど、特定ユーザー向けのサービスが提供される。
    nn::account::UserHandle handle;
    Printfln(">> Opening user");
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handle, uid));
    NN_UTIL_SCOPE_EXIT
    {
        Printfln("<< Closing user");
        nn::account::CloseUser(handle);
    };

    // セーブデータのサムネイル画像を設定する
    CreateThumbnailImage(g_ThumbnailBuffer, ThumbnailWidth, ThumbnailHeight);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::StoreSaveDataThumbnailImage(uid, g_ThumbnailBuffer, ThumbnailBufferSize));

    // :
    // その他オフライン状態で実行可能なアプリケーション実装
    // :

    // シングルユーザーのオンラインモードを想定した処理を開始
    SceneSingleUserWithNetwork(handle);

    // :
    // その他オフライン状態で実行可能なアプリケーション実装
    // :
}

// ネットワークサービスアカウントの ID トークンの取得
Program::Error Program::AcquireNsaIdToken(size_t* pOutSize, char* buffer, size_t bufferSize, const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    // ネットワークサービスアカウントの確定
    // - ユーザー操作によるキャンセル時には nn::account::ResultCancelledByUser が返る。
    auto r = nn::account::EnsureNetworkServiceAccountAvailable(handle);
    if (!r.IsSuccess())
    {
        if (nn::account::ResultCancelledByUser::Includes(r))
        {
            // ユーザーによるキャンセル処理
            return Error_Cancel;
        }
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }

    // ネットワークサービスアカウントの情報
    // - ユーザーはネットワークサービスアカウントを付け替えることができるため、アプリケーション起動ごとに変わっている可能性がある。
    nn::account::NetworkServiceAccountId nsaId;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNetworkServiceAccountId(&nsaId, handle));
    Printfln("- Network Service Account ID: %016llx", nsaId.id);

    // ネットワークサービスアカウントの ID トークンキャッシュを確定
    nn::account::AsyncContext ctx;
    auto durationMsec = -1ll;
    {
        // このスコープの処理時間を測る
        auto begin = nn::os::GetSystemTick();
        NN_UTIL_SCOPE_EXIT
        {
            auto end = nn::os::GetSystemTick();
            durationMsec = (end - begin).ToTimeSpan().GetMilliSeconds();
        };

        // ネットワークサービスアカウントへのログイン + ID トークンのキャッシュ API
        // - すでにキャッシュが存在する場合は即時完了する。
        // - ctx.Cancel() を呼ぶと、可能な限り早くイベントがシグナルされる。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::EnsureNetworkServiceAccountIdTokenCacheAsync(&ctx, handle));
        nn::os::SystemEvent e;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
        e.Wait();
    }
    r = ctx.GetResult();
    if (!r.IsSuccess())
    {
        if (nn::account::ResultNetworkServiceAccountUnavailable::Includes(r))
        {
            // ログイン時にネットワークサービスアカウントが利用できないことが発覚
            // - nn::account::EnsureNetworkServiceAccountAvailable() でエラーを解消できるため、再試行する。
            return Error_Retry;
        }
        else if (nn::account::ResultNetworkCommunicationError::Includes(r))
        {
            // ログイン時に通信関係のエラーが発生
            // - これにはアプリケーションの更新が必要な場合や、本体更新が必要な場合も含まれる。
            // - nn::error::ShowError() に渡し、詳細なエラー内容をユーザーに提示する必要がある。
            nn::err::ShowError(r);
            return Error_Cancel;
        }
        // ctx.Cancel() が呼ばれていると、ここで nn::account::ResultCancelled() が返る場合がある。
        // しかしこのサンプルでは ctx.Cancel() を呼んでいないため、これが返ることはない。
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
    Printfln("- Network Service Account login turnaround: %lld msec", durationMsec);

    // ネットワークサービスアカウントの ID トークンキャッシュを取得する
    size_t idTokenLength;
    r = nn::account::LoadNetworkServiceAccountIdTokenCache(&idTokenLength, buffer, bufferSize, handle);
    if (!r.IsSuccess())
    {
        if (false
            || nn::account::ResultNetworkServiceAccountUnavailable::Includes(r)
            || nn::account::ResultTokenCacheUnavailable::Includes(r))
        {
            // ネットワークサービスアカウントへのログイン後にアプリケーションがフォーカスを失い...
            //  - その間に NSA が無効になった場合
            //  - その間に ID トークンキャッシュが無効になった場合
            return Error_Retry;
        }
        // NSA 利用不可, トークンキャッシュ利用不可以外のエラーが返ることは通常ない。
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
    Printfln("- Network Service Account ID token: %.*s", idTokenLength, buffer);

    *pOutSize = idTokenLength;
    return Error_Success;
}

// シングルユーザーのオンラインモードを想定した処理
void Program::SceneSingleUserWithNetwork(const nn::account::UserHandle& handle) NN_NOEXCEPT
{
    // ネットワークサービスアカウントの ID トークンを取得する
    auto* idToken = reinterpret_cast<char*>(std::malloc(nn::account::NetworkServiceAccountIdTokenLengthMax));
    NN_UTIL_SCOPE_EXIT
    {
        std::free(idToken);
    };
    size_t idTokenLength;

    bool success = false;
    while (!success)
    {
        // ネットワーク接続要求
        while (NN_STATIC_CONDITION(true))
        {
            nn::nifm::SubmitNetworkRequestAndWait();
            if (!nn::nifm::IsNetworkAvailable())
            {
                auto r = nn::nifm::HandleNetworkRequestResult();
                if (!r.IsSuccess())
                {
                    if (!nn::nifm::ResultErrorHandlingCompleted::Includes(r))
                    {
                        // エラーかつ、それを解消できなかった場合はシーンを終了する。
                        return;
                    }
                    // エラーかつ、それを解消できた場合はリトライする。
                    continue;
                }
                // break
            }
            // ネットワーク接続を利用できる。
            break;
        }

        auto error = AcquireNsaIdToken(&idTokenLength, idToken, nn::account::NetworkServiceAccountIdTokenLengthMax, handle);
        switch (error)
        {
        case Error_Success:
            success = true;
            Printfln("-> Success");
            break;
        case Error_Cancel:
            Printfln("-> Cancelled");
            return;
        case Error_Retry:
            Printfln("-> Retry required");
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    // :
    // その他オンライン状態で実行可能なアプリケーション実装
    // :
}
