﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "3dSampleViewer_MenuUtility.h"

namespace nn { namespace g3d { namespace demo {

    void Menu::Update() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_pPages[m_SelectedPageIndex]->Update();
    }

    void Menu::Draw() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        if (m_DrawCallback)
        {
            m_DrawCallback(this, m_pDrawUserData);
        }

        m_pPages[m_SelectedPageIndex]->Draw();
    }

    Content* Menu::GetSelectedContent() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        Content* pContent = m_pPages[m_SelectedPageIndex]->GetSelectedContent();
        NN_ASSERT_NOT_NULL(pContent);
        return pContent;
    }

    const Content* Menu::GetSelectedContent() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        const Content* pContent = m_pPages[m_SelectedPageIndex]->GetSelectedContent();
        NN_ASSERT_NOT_NULL(pContent);
        return pContent;
    }

    bool Menu::SelectContent(Content* pSelectContent) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        if (!pSelectContent->IsSelectable())
        {
            return false;
        }

        // 現在選択されているページを取得
        Page* pSelectedPage = m_pPages[m_SelectedPageIndex];

        for (int pageIndex = 0; pageIndex < m_PageCount; ++pageIndex)
        {
            if (!m_pPages[pageIndex]->IsContentContained(pSelectContent))
            {
                continue;
            }

            // 入力されたコンテンツを選択
            if (!m_pPages[pageIndex]->SelectContent(pSelectContent))
            {
                return false;
            }

            m_SelectedPageIndex = pageIndex;

            // 選択されていたコンテンツを非選択状態に戻す
            if (pSelectedPage != m_pPages[m_SelectedPageIndex])
            {
                pSelectedPage->ClearIsSelectedInfo();
            }
            return true;
        }
        return false;
    }

    bool Menu::IsContentContained(const Content* pContent) const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int pageIndex = 0; pageIndex < m_PageCount; ++pageIndex)
        {
            if (m_pPages[pageIndex]->IsContentContained(pContent))
            {
                return true;
            }
        }
        return false;
    }

    bool Menu::SetContentSelectability(Content* pContent, bool isSelectable) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int pageIndex = 0; pageIndex < m_PageCount; ++pageIndex)
        {
            if (m_pPages[pageIndex]->SetContentSelectability(pContent, isSelectable))
            {
                return true;
            }
        }
        return false;
    }

    bool Menu::SetGroupSelectability(Group* pGroup, bool isSelectable) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int pageIndex = 0; pageIndex < m_PageCount; ++pageIndex)
        {
            if (m_pPages[pageIndex]->SetGroupSelectability(pGroup, isSelectable))
            {
                return true;
            }
        }
        return false;
    }

    void Page::Initialize(const char* name, Group** pGroupArray, int groupCount) NN_NOEXCEPT
    {
        NN_ASSERT(!IsInitialized());
        m_pName = name;
        m_pGroups = pGroupArray;
        m_GroupCount = groupCount;

        for (int groupIndex = 0; groupIndex < m_GroupCount; ++groupIndex)
        {
            if (m_pGroups[groupIndex]->HasSelectableContent())
            {
                m_SelectedGroup = groupIndex;
                m_pGroups[groupIndex]->SelectFirstContent();
                break;
            }
        }
        m_IsInitialized = true;
    }

    void Page::Update() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int groupIndex = 0; groupIndex < m_GroupCount; ++groupIndex)
        {
            m_pGroups[groupIndex]->Update();
        }
    }

    void Page::Draw() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        if (m_DrawCallback)
        {
            m_DrawCallback(this, m_pDrawUserData);
        }

        for (int groupIndex = 0; groupIndex < m_GroupCount; ++groupIndex)
        {
            m_pGroups[groupIndex]->Draw();
        }
    }

    void Page::ChangeNextContent() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        if (m_pGroups[m_SelectedGroup]->HasNextContent())
        {
            m_pGroups[m_SelectedGroup]->ChangeNextContent();
        }
        else
        {
            m_pGroups[m_SelectedGroup]->ClearIsSelectedInfo();
            m_SelectedGroup = GetNextGroupIndex();
            m_pGroups[m_SelectedGroup]->SelectFirstContent();
        }
    }

    void Page::ChangePreviousContent() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        if (m_pGroups[m_SelectedGroup]->HasPreviousContent())
        {
            m_pGroups[m_SelectedGroup]->ChangePreviousContent();
        }
        else
        {
            m_pGroups[m_SelectedGroup]->ClearIsSelectedInfo();
            m_SelectedGroup = GetPreviousGroupIndex();
            m_pGroups[m_SelectedGroup]->SelectLastContent();
        }
    }

    Content* Page::GetSelectedContent() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        Content* pContent = m_pGroups[m_SelectedGroup]->GetSelectedContent();
        NN_ASSERT_NOT_NULL(pContent);
        return pContent;
    }

    const Content* Page::GetSelectedContent() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        const Content* pContent = m_pGroups[m_SelectedGroup]->GetSelectedContent();
        NN_ASSERT_NOT_NULL(pContent);
        return pContent;
    }

    bool Page::IsContentContained(const Content* pContent) const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int groupIndex = 0; groupIndex < m_GroupCount; ++groupIndex)
        {
            if (m_pGroups[groupIndex]->IsContentContained(pContent))
            {
                return true;
            }
        }
        return false;
    }

    bool Page::SetContentSelectability(Content* pContent, bool isSelectable) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int groupIndex = 0; groupIndex < m_GroupCount; ++groupIndex)
        {
            Group::SetSelectabilityResultFlag result = m_pGroups[groupIndex]->SetContentSelectability(pContent, isSelectable);
            switch (result)
            {
            case nn::g3d::demo::Group::SetSelectabilityResult_NotFound:
                continue;
            case nn::g3d::demo::Group::SetSelectabilityResult_Success:
                return true;
            case nn::g3d::demo::Group::SetSelectabilityResult_ChangeGroup:
                // Selectability が設定されたグループには選択できるコンテンツが存在しない。
                if (m_pGroups[m_SelectedGroup] == m_pGroups[groupIndex])
                {
                    ChangeNextContent();
                }
                return true;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        return false;
    }

    bool Page::SetGroupSelectability(Group* pGroup, bool isSelectable) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        bool successed = true;
        for (int contentIndex = 0, contentCount = pGroup->GetContentCount(); contentIndex < contentCount; ++contentIndex)
        {
            Content* pContent = pGroup->GetContent(contentIndex);
            successed &= SetContentSelectability(pContent, isSelectable);
        }
        return successed;
    }

    int Page::GetNextGroupIndex() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int index = 1; index < m_GroupCount; ++index)
        {
            int next = (m_SelectedGroup + index) % m_GroupCount;
            if (m_pGroups[next]->HasSelectableContent())
            {
                return next;
            }
        }

        // 他に選択できるコンテンツがないので、そのまま
        return m_SelectedGroup;
    }

    int Page::GetPreviousGroupIndex() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int index = 1; index < m_GroupCount; ++index)
        {
            int prev = (m_SelectedGroup + (m_GroupCount - index)) % m_GroupCount;
            if (m_pGroups[prev]->HasSelectableContent())
            {
                return prev;
            }
        }

        // 他に選択できるコンテンツがないので、そのまま
        return m_SelectedGroup;
    }

    bool Page::SelectContent(Content* pSelectContent) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        if (!pSelectContent->IsSelectable())
        {
            return false;
        }

        // 現在選択されているグループを取得
        Group* pSelectedGroup = m_pGroups[m_SelectedGroup];

        for (int groupIndex = 0; groupIndex < m_GroupCount; ++groupIndex)
        {
            if (!m_pGroups[groupIndex]->IsContentContained(pSelectContent))
            {
                continue;
            }

            // 入力されたコンテンツを選択
            if (!m_pGroups[groupIndex]->SelectContent(pSelectContent))
            {
                return false;
            }

            m_SelectedGroup = groupIndex;

            // 選択されていたコンテンツを非選択状態に戻す
            if (pSelectedGroup != m_pGroups[m_SelectedGroup])
            {
                pSelectedGroup->ClearIsSelectedInfo();
            }
            return true;
        }
        return false;
    }

    void Page::ClearIsSelectedInfo() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_pGroups[m_SelectedGroup]->ClearIsSelectedInfo();
    }

    void Group::Initialize(const char* name, Content** pContents, int contentCount) NN_NOEXCEPT
    {
        NN_ASSERT(!IsInitialized());
        m_pName = name;
        m_pContents = pContents;
        m_ContentCount = contentCount;

        for (int contentIndex = 0; contentIndex < m_ContentCount; ++contentIndex)
        {
            if (m_pContents[contentIndex]->IsSelectable())
            {
                m_SelectedContent = contentIndex;
                break;
            }
        }

        m_IsInitialized = true;
    }

    void Group::Update() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int contentIndex = 0; contentIndex < m_ContentCount; ++contentIndex)
        {
            m_pContents[contentIndex]->Update();
        }
    }

    void Group::Draw() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        //TODO: PreContentDraw
        if (m_DrawCallback)
        {
            m_DrawCallback(this, m_pDrawUserData);
        }

        for (int contentIndex = 0; contentIndex < m_ContentCount; ++contentIndex)
        {
            m_pContents[contentIndex]->Draw();
        }

        //TODO: PostContentDraw
    }

    Content* Group::GetSelectedContent() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int contentIndex = 0; contentIndex < m_ContentCount; ++contentIndex)
        {
            if (m_pContents[contentIndex]->IsSelected())
            {
                return m_pContents[contentIndex];
            }
        }
        return nullptr;
    }

    const Content* Group::GetSelectedContent() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int contentIndex = 0; contentIndex < m_ContentCount; ++contentIndex)
        {
            if (m_pContents[contentIndex]->IsSelected())
            {
                return m_pContents[contentIndex];
            }
        }
        return nullptr;
    }

    Group::SetSelectabilityResultFlag Group::SetContentSelectability(Content* pContent, bool isSelectable) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int contentIndex = 0; contentIndex < m_ContentCount; ++contentIndex)
        {
            if (m_pContents[contentIndex] != pContent)
            {
                continue;
            }

            m_pContents[contentIndex]->m_IsSelectable = isSelectable;
            if (!isSelectable && contentIndex == m_SelectedContent)
            {
                if (HasPreviousContent())
                {
                    ChangePreviousContent();
                    return SetSelectabilityResult_Success;
                }
                else if (HasNextContent())
                {
                    ChangeNextContent();
                    return SetSelectabilityResult_Success;
                }

                return SetSelectabilityResult_ChangeGroup;
            }

            return SetSelectabilityResult_Success;
        }
        return SetSelectabilityResult_NotFound;
    }

    bool Group::IsContentContained(const Content* pContent) const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int contentIndex = 0; contentIndex < m_ContentCount; ++contentIndex)
        {
            if (pContent == m_pContents[contentIndex])
            {
                return true;
            }
        }
        return false;
    }

    bool Group::HasSelectableContent() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int contentIndex = 0; contentIndex < m_ContentCount; ++contentIndex)
        {
            if (m_pContents[contentIndex]->IsSelectable())
            {
                return true;
            }
        }
        return false;
    }

    void Group::ChangeNextContent() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT(HasNextContent());
        m_pContents[m_SelectedContent]->m_IsSelected = false;
        m_SelectedContent = GetNextContentIndex();
        NN_ASSERT_NOT_EQUAL(m_SelectedContent, InvalidIndex);
        m_pContents[m_SelectedContent]->m_IsSelected = true;
    }

    void Group::ChangePreviousContent() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT(HasPreviousContent());
        m_pContents[m_SelectedContent]->m_IsSelected = false;
        m_SelectedContent = GetPreviousContentIndex();
        NN_ASSERT_NOT_EQUAL(m_SelectedContent, InvalidIndex);
        m_pContents[m_SelectedContent]->m_IsSelected = true;
    }

    void Group::SelectFirstContent() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_SelectedContent = -1;
        m_SelectedContent = GetNextContentIndex();
        m_pContents[m_SelectedContent]->m_IsSelected = true;
    }

    void Group::SelectLastContent() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_SelectedContent = m_ContentCount;
        m_SelectedContent = GetPreviousContentIndex();
        m_pContents[m_SelectedContent]->m_IsSelected = true;
    }

    void Group::ClearIsSelectedInfo() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_pContents[m_SelectedContent]->m_IsSelected = false;
    }

    int Group::GetNextContentIndex() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int contentIndex = m_SelectedContent + 1; contentIndex < m_ContentCount; ++contentIndex)
        {
            if (m_pContents[contentIndex]->IsSelectable())
            {
                return contentIndex;
            }
        }
        return InvalidIndex;
    }

    int Group::GetPreviousContentIndex() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int contentIndex = m_SelectedContent - 1; contentIndex >= 0; --contentIndex)
        {
            if (m_pContents[contentIndex]->IsSelectable())
            {
                return contentIndex;
            }
        }
        return InvalidIndex;
    }

    bool Group::SelectContent(Content* pSelectContent) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        if (!pSelectContent->IsSelectable())
        {
            return false;
        }

        // 現在選択されているコンテンツを取得
        Content* pSelectedContent = m_pContents[m_SelectedContent];

        for (int contentIndex = 0; contentIndex < m_ContentCount; ++contentIndex)
        {
            if (pSelectContent != m_pContents[contentIndex])
            {
                continue;
            }

            // 入力されたコンテンツを選択
            m_pContents[contentIndex]->m_IsSelected = true;
            m_SelectedContent = contentIndex;

            // 選択されていたコンテンツを非選択状態に戻す
            if (pSelectedContent != m_pContents[m_SelectedContent])
            {
                pSelectedContent->m_IsSelected = false;
            }
            return true;
        }
        return false;
    }
}

} }
