﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "3dSampleViewer.h"
#include "3dSampleViewer_Menu.h"
#include "3dSampleViewer_Renderer.h"
#include "3dSampleViewer_Viewer.h"

#include <nn/g3d.h>
#include <nns/g3d.h>
#include <nn/g3d/g3d_Viewer.h>

namespace {

//----------------------------------------
// Resources
//----------------------------------------

#define RESOURCE_PATH  "Resource:/"

const char* BfshaPath[] = {
    RESOURCE_PATH "town.bfsha",  // 3D Editor 用
    RESOURCE_PATH "demo.bfsha",  // 3D Editor 用
    RESOURCE_PATH "visualizeInfo.bfsha"
};
NN_STATIC_ASSERT(NN_ARRAY_SIZE(BfshaPath) == ShaderArchiveType_Count);

const char* BfresPath[] = {
    RESOURCE_PATH "town_env.bfres",
};
NN_STATIC_ASSERT(NN_ARRAY_SIZE(BfresPath) == ResFileType_Count);

g3ddemo::ResourceHolder g_Holder;

enum TextureRefType
{
    TextureRefType_Water,
    TextureRefType_Shadow,
    TextureRefType_Color,
    TextureRefType_Light,
    TextureRefType_Count
};

struct EnvTextureRef
{
    const char* name;
    nn::g3d::TextureRef textureRef;
};

EnvTextureRef g_EnvTextureRef[TextureRefType_Count];
nns::gfx::FrameBuffer g_DummyFrameBuffer;               // Texture2D
nn::gfx::DescriptorSlot g_DummyFrameBufferDescriptorSlot;

nns::gfx::DepthStencilBuffer g_DummyDepthBuffer;        // Texture2DArrayShadow
nn::gfx::DescriptorSlot g_DummyDepthBufferDescriptorSlot;

//----------------------------------------
// View
//----------------------------------------

g3ddemo::CameraController   g_CameraController;
nns::g3d::RenderView        g_RenderView;

//----------------------------------------
// Viewer
//----------------------------------------

Viewer g_Viewer;

void InitializeResources(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // ダミーテクスチャの初期化
    g3ddemo::CreateDummyTextureAndSampler(pDevice);

    // ホルダーの初期化
    g_Holder.Initialize();

    // シェーダーアーカイブの初期化
    for (const char* pPath : BfshaPath)
    {
        nn::g3d::ResShaderFile* pFile = g3ddemo::LoadResource<nn::g3d::ResShaderFile>(pPath);
        pFile->Setup(pDevice);
        g_Holder.shaderFiles.PushBack(pFile);
    }

    // リソースモデルの初期化
    for (const char* pPath : BfresPath)
    {
        nn::g3d::ResFile* pResFile = g3ddemo::LoadResource<nn::g3d::ResFile>(pPath);
        pResFile->Setup(pDevice);
        g_Holder.files.PushBack(pResFile);

        // テクスチャーおよびサンプラーをディスクリプタプールに登録
        g3ddemo::SetupTexture(pDevice, pResFile, g3ddemo::TextureBindCallback);
        g3ddemo::RegisterSamplerToDescriptorPool(pResFile);
    }
}

void FinalizeResources(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // リソースモデルの終了処理
    for (nn::g3d::ResFile* pResFile : g_Holder.files)
    {
        g3ddemo::UnregisterSamplerFromDescriptorPool(pResFile);
        g3ddemo::CleanupTexture(pDevice, pResFile);
        pResFile->Cleanup(pDevice);
    }
    g_Holder.files.Clear();

    // シェーダーファイルの終了処理
    for (nn::g3d::ResShaderFile* pFile : g_Holder.shaderFiles)
    {
        pFile->Cleanup(pDevice);
    }
    g_Holder.shaderFiles.Clear();

    // ホルダーの終了処理
    g_Holder.Shutdown();

    g3ddemo::DeleteDummyTextureAndSampler(pDevice);
}

void InitializeEnvTexture(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    nns::gfx::DebugGraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();

    // ダミーの環境テクスチャを作成
    {
        nns::gfx::FrameBuffer::InfoType info(2, 2);
        info.SetColorBufferFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm);
        g3ddemo::InitializeFrameBuffer(&g_DummyFrameBuffer, &info);
        g_DummyFrameBuffer.SetClearColor(.0f, .0f, .0f, .0f);
        g_DummyFrameBufferDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_DummyFrameBuffer.GetColorBuffer()->GetTextureView());
    }

    //  ダミーのデプスバッファー初期化
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(2);
        info.SetHeight(2);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil | nn::gfx::GpuAccess_Texture);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_D32_Float);
        info.SetMipCount(1);
        info.SetArrayLength(2);
        size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
        size_t align = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);
        ptrdiff_t offset = pGfxFramework->AllocatePoolMemory(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, size, align);
        nn::gfx::MemoryPool* pMemoryPool = pGfxFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget);
        g_DummyDepthBuffer.Initialize(pDevice, info, pMemoryPool, offset, size);
        g_DummyDepthBufferDescriptorSlot = g3ddemo::RegisterTextureToDescriptorPool(g_DummyDepthBuffer.GetTextureView());
    }

    // EnvTexture の設定
    {
        g_EnvTextureRef[TextureRefType_Water] = { "water", nn::g3d::TextureRef(g_DummyFrameBuffer.GetColorBuffer()->GetTextureView(), g_DummyFrameBufferDescriptorSlot) };
        g_EnvTextureRef[TextureRefType_Shadow] = { "shadow", nn::g3d::TextureRef(g_DummyDepthBuffer.GetTextureView(), g_DummyDepthBufferDescriptorSlot) };
        g_EnvTextureRef[TextureRefType_Color] = { "color", nn::g3d::TextureRef(g_DummyFrameBuffer.GetColorBuffer()->GetTextureView(), g_DummyFrameBufferDescriptorSlot) };
        g_EnvTextureRef[TextureRefType_Light] = { "light", nn::g3d::TextureRef(g_DummyFrameBuffer.GetColorBuffer()->GetTextureView(), g_DummyFrameBufferDescriptorSlot) };
    }
}

void FinalizeEnvTexture() NN_NOEXCEPT
{
    g3ddemo::UnregisterTextureFromDescriptorPool(g_DummyDepthBuffer.GetTextureView());
    g3ddemo::UnregisterTextureFromDescriptorPool(g_DummyFrameBuffer.GetColorBuffer()->GetTextureView());
}

// シャドウサンプラーの作成
void CreateShadowSampler(nn::gfx::Device* pDevice, nn::g3d::ResMaterial* pResMaterial) NN_NOEXCEPT
{
    const char* shadowSamplerName = "_d0";
    nn::gfx::Sampler* pSampler = pResMaterial->FindSampler(shadowSamplerName);

    // マテリアルに設定されているサンプラーを破棄し、シャドウサンプラーを作成
    g3ddemo::UnregisterSamplerFromDescriptorPool(pSampler);
    pSampler->Finalize(pDevice);

    nn::gfx::Sampler::InfoType samplerInfo;
    samplerInfo.SetDefault();
    samplerInfo.SetAddressU(nn::gfx::TextureAddressMode_ClampToBorder);
    samplerInfo.SetAddressV(nn::gfx::TextureAddressMode_ClampToBorder);
    samplerInfo.SetAddressW(nn::gfx::TextureAddressMode_ClampToEdge);
    samplerInfo.SetBorderColorType(nn::gfx::TextureBorderColorType_White);
    samplerInfo.SetFilterMode(nn::gfx::FilterMode_Comparison_MinLinear_MagLinear_MipLinear);
    samplerInfo.SetComparisonFunction(nn::gfx::ComparisonFunction_LessEqual);

    pSampler->Initialize(pDevice, samplerInfo);
    nn::gfx::DescriptorSlot samplerDescriptorSlot = g3ddemo::RegisterSamplerToDescriptorPool(pSampler);
    pResMaterial->SetSamplerDescriptorSlot(shadowSamplerName, samplerDescriptorSlot);
}

void SetupEnvModel(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // 環境テクスチャの初期化
    InitializeEnvTexture(pDevice);

    // 環境モデルインスタンスの生成
    nn::g3d::ModelObj* pEnvModel = nullptr;
    {
        NN_ASSERT_EQUAL(g_Holder.shaderFiles.GetCount(), ShaderArchiveType_Count);
        nn::g3d::ResShaderArchive* pTownShaderArchive = g_Holder.shaderFiles[ShaderArchiveType_Town]->GetResShaderArchive();

        NN_ASSERT_EQUAL(g_Holder.files.GetCount(), ResFileType_Count);
        nn::g3d::ResFile* pResFile = g_Holder.files[ResFileType_TownEnv];

        nn::g3d::ResModel* pResModel = pResFile->FindModel("env");
        NN_ASSERT_NOT_NULL(pResModel);

        // RenderModel を生成し、マテリアルとシェーダーを結びつける
        nns::g3d::RenderModel* pRenderModel = nns::g3d::CreateRenderModel(pDevice, pResModel);
        NN_ASSERT_NOT_NULL(pRenderModel);
        pRenderModel->CreateDrawPass(pDevice, pTownShaderArchive);
        g_Holder.renderModels.PushBack(pRenderModel);

        // モデルインスタンスを生成
        {
            nn::g3d::ModelObj::Builder builder(pResModel);
            {
                builder.SetBoundingEnabled();
                builder.ShapeBufferingCount(BufferingCount);
                builder.SkeletonBufferingCount(BufferingCount);
                builder.MaterialBufferingCount(BufferingCount);
                builder.ViewCount(ViewType_Count);
            }
            nns::g3d::ModelAnimObj* pModelAnimObj = nns::g3d::CreateModelAnimObj(pDevice, builder);
            NN_ASSERT_NOT_NULL(pModelAnimObj);
            g_Holder.modelAnimObjs.PushBack(pModelAnimObj);

            pEnvModel = pModelAnimObj->GetModelObj();
        }
        NN_ASSERT_NOT_NULL(pEnvModel);

        nns::g3d::RenderModelObj* pRenderModelObj = nns::g3d::CreateRenderModelObj(pEnvModel, pRenderModel);
        NN_ASSERT_NOT_NULL(pRenderModelObj);
        g_Holder.renderModelObjs.PushBack(pRenderModelObj);

        // ビジビリティを無効化
        {
            for (int materialIndex = 0, materialCount = pEnvModel->GetMaterialCount(); materialIndex < materialCount; ++materialIndex)
            {
                pRenderModelObj->GetModelObj()->SetMaterialVisible(materialIndex, false);
            }
        }

        // ダミーテクスチャのバインド
        pRenderModelObj->BindDummySampler(g3ddemo::GetDummyTextureRef().GetDescriptorSlot(), g3ddemo::GetDummySamplerDescriptorSlot());
    }

    const char* EnvMaterialName = "lambert1";
    nn::g3d::MaterialObj* pMaterialObj = pEnvModel->FindMaterial(EnvMaterialName);
    NN_ASSERT(pMaterialObj);

    // シャドウサンプラーの作成
    {
        nn::g3d::ResMaterial* pResMaterial = const_cast<nn::g3d::ResMaterial*>(pMaterialObj->GetResource());
        CreateShadowSampler(pDevice, pResMaterial);
    }

    for (const EnvTextureRef& envTex : g_EnvTextureRef)
    {
        for (int textureIndex = 0, textureCount = pMaterialObj->GetTextureCount(); textureIndex < textureCount; ++textureIndex)
        {
            const char* pTextureName = pMaterialObj->GetTextureName(textureIndex);
            if (strcmp(envTex.name, pTextureName) != 0)
            {
                continue;
            }
            pMaterialObj->SetTexture(textureIndex, envTex.textureRef);
        }
    }

    for (int bufferindex = 0; bufferindex < BufferingCount; ++bufferindex)
    {
        pMaterialObj->CalculateMaterial(bufferindex);
    }
}

void CleanupEnvModel() NN_NOEXCEPT
{
    // ModelAnimObj や RenderModelObj の破棄は、ResourceHolder の破棄処理で行います。
    FinalizeEnvTexture();
}

// TODO: hid + InputDirector を用いることで PC ・開発機の両対応
nns::nac::Mouse g_Mouse;
nns::nac::Pad g_Pad;

void InitializeInputInterface() NN_NOEXCEPT
{
#ifdef NN_BUILD_APISET_GENERIC
    HWND hWnd = static_cast<HWND>(g3ddemo::GetWindowHandle());

    // Pad
    g_Pad.Initialize(hWnd);

    // Mouse
    g_Mouse.Initialize(hWnd);

#elif NN_BUILD_APISET_NX
    // do nothing. (hid を利用)
#else
    #error
#endif
}

void FinalizeInputInterface() NN_NOEXCEPT
{
    g_Mouse.Finalize();
    g_Pad.Finalize();
}

void InitializeView(nn::gfx::Device* pDevice) NN_NOEXCEPT
{
    // Controller
    {
        nn::util::Vector3fType position = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 160.0f, 450.0f);
        nn::util::Vector3fType target = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f, 0.0f, 0.0f);
        g_CameraController.SetPos(position);
        g_CameraController.SetLookAtPos(target);
        g_CameraController.Preset();
    }

    // RenderView
    {
        g_RenderView.Initialize(pDevice, ViewType_Count, BufferingCount);
        g_RenderView.SetViewMtx(ViewType_Camera, g_CameraController.GetMatrix());
        SetDefaultProjection();
    }
}

void InitializeViewer() NN_NOEXCEPT
{
    g_Viewer.Initialize(g_Holder);

    // フォールバックシェーダーの設定
    const char* fallbackShadingModelName = "basic";
    nn::g3d::ResShaderArchive* pResShaderArchive = g_Holder.shaderFiles[ShaderArchiveType_Demo]->GetResShaderArchive();
    SetFallBackShadingModel(pResShaderArchive->FindShadingModel(fallbackShadingModelName));

    // ユニフォームブロックの設定
    {
        const nn::gfx::Buffer* pUniformBlocks[BufferingCount];

        // env
        {
            nn::g3d::ModelObj* pEnvModel = g_Holder.modelAnimObjs[ResFileType_TownEnv]->GetModelObj();
            NN_ASSERT(strcmp(pEnvModel->GetName(), "env") == 0);
            nn::g3d::MaterialObj* pEnvMaterial = pEnvModel->GetMaterial(0);
            NN_ASSERT_EQUAL(pEnvMaterial->GetBufferingCount(), BufferingCount);

            for (int bufferIndex = 0; bufferIndex < BufferingCount; ++bufferIndex)
            {
                pUniformBlocks[bufferIndex] = pEnvMaterial->GetMaterialBlock(bufferIndex);
            }
            SetUniformBlock(UniformBlockType_Env, pUniformBlocks, BufferingCount);
        }

        // view
        {
            for (int bufferIndex = 0; bufferIndex < BufferingCount; ++bufferIndex)
            {
                pUniformBlocks[bufferIndex] = g_RenderView.GetViewUniformBlock(ViewType_Camera, bufferIndex);
            }
            SetUniformBlock(UniformBlockType_View, pUniformBlocks, BufferingCount);
        }

        // UniformBlocks
        InitializeUniformBlocks();
    }
}

void Initialize() NN_NOEXCEPT
{
    nn::gfx::Device* pDevice = g3ddemo::GetGfxFramework()->GetDevice();

    // メニューの初期化
    InitializeMenu();

    // リソースの初期化
    InitializeResources(pDevice);

    // 環境モデルの初期化
    SetupEnvModel(pDevice);

    // 入力インターフェイスの初期化
    InitializeInputInterface();

    // ビューの初期化
    InitializeView(pDevice);

    // レンダラーの初期化
    InitializeRenderer();

    // ビューアーの初期化
    InitializeViewer();

    // デバッグ描画に利用する描画ステートの初期化
    InitializeRenderState(pDevice);
}

void LoockAtTarget() NN_NOEXCEPT
{
    nn::util::Vector3fType pos, target;
    g_CameraController.GetPos(&pos);
    g_CameraController.GetLookAtPos(&target);

    MenuContext::SelectedInfo selectedInfo = GetSelectedInfo();
    int index = GetMenuContext()->GetSelectedIndex(selectedInfo);
    switch (selectedInfo)
    {
    case MenuContext::SelectedInfo_Model:
        {
            // env モデルのみ存在する
            if (index == MenuContext::InvalidIndex)
            {
                break;
            }

            const int& selectedModelIndex = index;
            nns::g3d::ModelAnimObj* pModelAnimObj = g_Holder.modelAnimObjs[selectedModelIndex];
            g_CameraController.GetPos(&pos);
            g_CameraController.GetLookAtPos(&target);
            g3ddemo::LookAtModel(&pos, &target, pModelAnimObj->GetModelObj());
        }
        break;
    case MenuContext::SelectedInfo_Shape:
        {
            const int selectedModelIndex = GetMenuContext()->GetSelectedIndex(MenuContext::SelectedInfo_Model);
            nn::g3d::ModelObj* pModelObj = g_Holder.modelAnimObjs[selectedModelIndex]->GetModelObj();
            g3ddemo::LookAtShape(&pos, &target, pModelObj->GetShape(index));
        }
        break;
    case MenuContext::SelectedInfo_SubMesh:
        {
            const int selectedModelIndex = GetMenuContext()->GetSelectedIndex(MenuContext::SelectedInfo_Model);
            nn::g3d::ModelObj* pModelObj = g_Holder.modelAnimObjs[selectedModelIndex]->GetModelObj();
            const int selectedShapeIndex = GetMenuContext()->GetSelectedIndex(MenuContext::SelectedInfo_Shape);
            nn::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(selectedShapeIndex);
            if (pShapeObj->GetSubMeshCount() > 1)
            {
                g3ddemo::LookAtSubMesh(&pos, &target, pShapeObj, 0, index);
            }
            else
            {
                // サブメッシュ分割されていないので、シェイプに注視するようにする
                g3ddemo::LookAtShape(&pos, &target, pShapeObj);
            }
        }
        break;
    case MenuContext::SelectedInfo_Bone:
        {
            const int selectedModelIndex = GetMenuContext()->GetSelectedIndex(MenuContext::SelectedInfo_Model);
            nn::g3d::ModelObj* pModelObj = g_Holder.modelAnimObjs[selectedModelIndex]->GetModelObj();
            g3ddemo::LookAtBone(&pos, &target, pModelObj, index);
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }

    g_CameraController.SetPos(pos);
    g_CameraController.SetLookAtPos(target);
}

void UpdateView(int bufferIndex) NN_NOEXCEPT
{
    g3ddemo::Pad& pad = g3ddemo::GetPad();
    if (pad.IsTriggered(g3ddemo::Pad::TRIGGER_R))
    {
        LoockAtTarget();
    }

    g_CameraController.UpdateCamera(&g_Pad, &g_Mouse, g_Mouse.IsPointerOn(), true);
    {
        nn::util::Matrix4x3fType viewMtx = g_CameraController.GetMatrix();
        g_RenderView.SetViewMtx(ViewType_Camera, viewMtx);
    }

    g_RenderView.Calculate(bufferIndex);
}

void UpdateModel(nn::gfx::Device* pDevice, int bufferIndex) NN_NOEXCEPT
{
    nn::g3d::viewer::ViewerServer& viewerServer = nn::g3d::viewer::ViewerServer::GetInstance();
    for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
    {
        pModelAnimObj->CalculateAnimations();

        // 3DEditor でスケルタルア二メーションが再生されていた場合には、
        // CalculateAnimations() で更新されたアニメーションを上書きします。
        // また、キャッシュを活用するためにスケルタルアニメーションの計算から
        // ワールド行列計算までを連続して行います。
        viewerServer.CalculateSkeletalAnimations(pModelAnimObj->GetModelObj());

        pModelAnimObj->CalculateWorld();
        pModelAnimObj->CalculateBounding();
        pModelAnimObj->CalculateView(0, 0, g_RenderView.GetViewMtx(ViewType_Camera));
    }

    viewerServer.CalculateAnimations();

    for (nns::g3d::ModelAnimObj* pModelAnimObj : g_Holder.modelAnimObjs)
    {
        pModelAnimObj->CalculateUniformBlock(bufferIndex);
    }

    for (nns::g3d::RenderModelObj* pRenderModelObj : g_Holder.renderModelObjs)
    {
        pRenderModelObj->UpdateShader(pDevice);
    }
}

void Finalize() NN_NOEXCEPT
{
    nn::gfx::Device* pDevice = g3ddemo::GetGfxFramework()->GetDevice();

    // デバッグ描画に利用する描画ステートの終了処理
    FinalizeRenderState(pDevice);

    // TownContext の終了処理
    FinalizeUniformBlocks();

    // ビューアーの終了処理
    g_Viewer.Finalize();

    // レンダラーの終了処理
    FinalizeRenderer();

    // ビューの終了処理
    g_RenderView.Finalize(pDevice);

    // 入力インターフェイスの終了処理
    FinalizeInputInterface();

    // 環境モデルの終了処理
    CleanupEnvModel();

    // リソースホルダーの終了処理
    FinalizeResources(pDevice);
}

void Calculate(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    int* pBufferIndex = reinterpret_cast<int*>(pUserData);

    // 入力インターフェイスの更新
    {
        g_Mouse.Update();
        g_Pad.Update();
        g3ddemo::Pad& pad = g3ddemo::GetPad();
        pad.Read();
    }

    // メニューの更新
    UpdateMenu();

    // ビューの更新
    UpdateView(*pBufferIndex);

    // ビューアーの更新
    g_Viewer.Update();

    // モデルの更新
    UpdateModel(pGraphicsFramework->GetDevice(), *pBufferIndex);

    // ユニフォームブロックの更新
    UpdateUniformBlock(*pBufferIndex);

    // レンダラーの更新
    UpdateRenderer(&g_RenderView, ViewType_Camera);
}

void MakeCommand(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    NN_UNUSED(pUserData);

    pGraphicsFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* pCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

        // キャッシュを無効化
        pCommandBuffer->InvalidateMemory(nn::gfx::GpuAccess_ShaderCode | nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_Texture);

        static bool isDummyTextureCleard = false;
        if(!isDummyTextureCleard)
        {
            g3ddemo::ClearFrameBuffer(pCommandBuffer, &g_DummyFrameBuffer);
            pCommandBuffer->ClearDepthStencil(g_DummyDepthBuffer.GetDepthStencilView(), 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);
            isDummyTextureCleard = true;
        }

        // レンダーターゲットの設定
        {
            nn::gfx::ColorTargetView* pColorTargetView = pGraphicsFramework->GetColorTargetView();
            nn::gfx::DepthStencilView* pDepthStencilView = pGraphicsFramework->GetDepthStencilView();

            nn::util::Color4f clearColor = nn::util::Color4f::Gray();
            pCommandBuffer->ClearColor(pColorTargetView, clearColor.GetR(), clearColor.GetG(), clearColor.GetB(), 1.0f, nullptr);
            pCommandBuffer->ClearDepthStencil(pDepthStencilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, nullptr);
            pCommandBuffer->SetRenderTargets(1, &pColorTargetView, pDepthStencilView);
        }

        // モデルの描画
        MakeModelDrawCommand(pCommandBuffer, bufferIndex);

        // ワイヤーフレームの描画
        MakeWireframeDrawCommand(pCommandBuffer, 1.4f, bufferIndex);

        // バウンディングの描画
        MakeBoundingDrawCommand(pCommandBuffer, bufferIndex);

        // スケルトンの描画
        MakeSkeletonDrawCommand(pCommandBuffer, bufferIndex);

        // グリッドの描画
        MakeGridDrawCommand(pCommandBuffer, bufferIndex);

        // メニューの描画
        DrawMenu(pCommandBuffer);
    }
    pGraphicsFramework->EndFrame(bufferIndex);
}

} // anonymous namespace

g3ddemo::ResourceHolder* GetResourceHolder() NN_NOEXCEPT
{
    return &g_Holder;
}

g3ddemo::CameraController* GetCameraController() NN_NOEXCEPT
{
    return &g_CameraController;
}

void SetProjectionMatrix(const nn::util::Matrix4x4fType& projectionMatrix) NN_NOEXCEPT
{
    g_RenderView.SetProjectionMtx(ViewType_Camera, projectionMatrix);
}

void SetDefaultProjection() NN_NOEXCEPT
{
    nn::util::Matrix4x4fType projectionMatrix;
    const float nearClip = 1.0f;
    const float farClip = 40000.0f;
    MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix, nn::util::DegreeToRadian(45.0f), 16.0f / 9.0f, nearClip, farClip);
    SetProjectionMatrix(projectionMatrix);
}

const nn::g3d::ModelObj * GetSelectedModelObj() NN_NOEXCEPT
{
    const MenuContext* pContext = GetMenuContext();
    int modelIndex = pContext->GetSelectedIndex(MenuContext::SelectedInfo_Model);
    if (modelIndex == MenuContext::InvalidIndex)
    {
        return nullptr;
    }

    g3ddemo::ResourceHolder* pHolder = GetResourceHolder();
    NN_ASSERT_RANGE(modelIndex, 0, pHolder->modelAnimObjs.GetCount());
    const nn::g3d::ModelObj* pModelObj = pHolder->modelAnimObjs[modelIndex]->GetModelObj();
    return pModelObj;
}

int SampleViewer() NN_NOEXCEPT
{
    nns::gfx::DebugGraphicsFramework* pGfxFramework = g3ddemo::GetGfxFramework();

    // 初期化
    Initialize();

    int bufferIndex = 0;
    pGfxFramework->SetCalculateCallback(Calculate, &bufferIndex);
    pGfxFramework->SetMakeCommandCallback(MakeCommand, nullptr);

    // メインループ
    while (NN_STATIC_CONDITION(1))
    {
        // タッチ長押しで終了
        g3ddemo::GetControllerManager()->Update();
        if (g3ddemo::isExitDemo())
        {
            break;
        }

        // フレーム処理実行
        pGfxFramework->ProcessFrame();

        bufferIndex = (bufferIndex + 1) % BufferingCount;
    }
    pGfxFramework->QueueFinish();

    pGfxFramework->SetCalculateCallback(nullptr, nullptr);
    pGfxFramework->SetMakeCommandCallback(nullptr, nullptr);

    // 終了処理
    Finalize();

    return EXIT_SUCCESS;
}

