﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/demo/pad/win/demo_PadWin.h>
#include <nw/demo/pad/demo_PadUtil.h>
#include <nw/ut/ut_Memory.h>
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <windows.h>
#include <mmsystem.h>

namespace nw
{
namespace demo
{

//------------------------------------------------------------------------------
const u8 PadWin::DEFAULT_V_KEY_CONFIG[PAD_IDX_MAX] =
{
    'A', // IDX_A
    'B', // IDX_B
    0, // IDX_C
    'X', // IDX_X
    'Y', // IDX_Y
    'Z', // IDX_Z
    0, // IDX_2
    0, // IDX_1
    'H', // IDX_HOME
    0, // IDX_MINUS
    0, // IDX_PLUS
    'S', // IDX_START
    'D', // IDX_SELECT
    'K', // IDX_ZL
    'E', // IDX_ZR
    'L', // IDX_L
    'R', // IDX_R
    0, // IDX_TOUCH
    VK_UP,      // IDX_UP
    VK_DOWN,    // IDX_DOWN
    VK_LEFT,    // IDX_LEFT
    VK_RIGHT,   // IDX_RIGHT
    0, // IDX_LS_UP
    0, // IDX_LS_DOWN
    0, // IDX_LS_LEFT
    0, // IDX_LS_RIGHT
    0, // IDX_RS_UP
    0, // IDX_RS_DOWN
    0, // IDX_RS_LEFT
    0, // IDX_RS_RIGHT
};

const u8 PadWin::DEFAULT_JOY_PAD_CONFIG[PAD_IDX_MAX] =
{
    4,  // IDX_A
    3,  // IDX_B
    0,  // IDX_C
    2,  // IDX_X
    1,  // IDX_Y
    8,  // IDX_Z
    10, // IDX_2
    9,  // IDX_1
    0,  // IDX_HOME
    0,  // IDX_MINUS
    0,  // IDX_PLUS
    12, // IDX_START
    11, // IDX_SELECT
    7,  // IDX_ZL
    8,  // IDX_ZR
    5,  // IDX_L
    6,  // IDX_R
    0,  // IDX_TOUCH
    0,  // IDX_UP
    0,  // IDX_DOWN
    0,  // IDX_LEFT
    0,  // IDX_RIGHT
    1,  // IDX_LS_UP  -> 上下逆転させる
    0,  // IDX_LS_DOWN
    0,  // IDX_LS_LEFT
    0,  // IDX_LS_RIGHT
    1,  // IDX_RS_UP  -> 上下逆転させる
    0,  // IDX_RS_DOWN
    0,  // IDX_RS_LEFT
    0,  // IDX_RS_RIGHT
};

//------------------------------------------------------------------------------
PadWin::PadWin(int joyPadNumber)
: Pad(),
  m_JoyPadNumber( joyPadNumber ),
  m_StickClampMax( STICK_CLAMP_MAX_DEFAULT ),
  m_StickClampMin( STICK_CLAMP_MIN_DEFAULT ),
  m_Flag( MASK_NORMALIZE_ANALOG_STICK_AS_CIRCLE ),
  m_KeyboardMouseDevice( NULL ),
  m_JoyPadDevice( NULL )
{
    SetVKeyConfigDefault();
    SetJoyPadConfigDefault();
}

//------------------------------------------------------------------------------
void
PadWin::SetVKeyConfig(const u8* vkeyConfig)
{
    nw::ut::MemCpy( m_VKeyConfig, vkeyConfig, sizeof( m_VKeyConfig ) );
}

//------------------------------------------------------------------------------
void
PadWin::SetVKeyConfig(PadIdx bit, u8 vkey)
{
    if (bit < PAD_IDX_MAX)
    {
        m_VKeyConfig[ bit ] = vkey;
    }
    else
    {
        NW_ERR( "range over[%d]\n", bit );
    }
}

//------------------------------------------------------------------------------
void
PadWin::SetVKeyConfigDefault()
{
    nw::ut::MemCpy( m_VKeyConfig, DEFAULT_V_KEY_CONFIG, sizeof(m_VKeyConfig) );
}

//------------------------------------------------------------------------------
bool
PadWin::IsJoyPadEnable() const
{
    if ( m_JoyPadDevice )
    {
        return ( m_JoyPadNumber < m_JoyPadDevice->GetPadNum() && m_JoyPadDevice->GetPads(m_JoyPadNumber).isEnable );
    }
    else
    {
        return false;
    }
}

//------------------------------------------------------------------------------
void
PadWin::GetStickClamp(int* min, int* max) const
{
    NW_ASSERT_NOT_NULL( min );
    NW_ASSERT_NOT_NULL( max );

    *min = m_StickClampMin;
    *max = m_StickClampMax;
}

//------------------------------------------------------------------------------
void
PadWin::SetStickClamp(int min, int max)
{
    if (STICK_CLAMP_VALUE_MIN <= min && min <= STICK_CLAMP_VALUE_MAX &&
        STICK_CLAMP_VALUE_MIN <= max && max <= STICK_CLAMP_VALUE_MAX &&
        min < max)
    {
        m_StickClampMin = min;
        m_StickClampMax = max;
    }
    else
    {
        NW_ERR( "illgal value[%d/%d]", min, max );
    }
}

//------------------------------------------------------------------------------
void
PadWin::SetJoyPadConfig(const u8* joyPadConfig)
{
    nw::ut::MemCpy( m_JoyPadConfig, joyPadConfig, sizeof(m_JoyPadConfig) );
}

//------------------------------------------------------------------------------
void
PadWin::SetJoyPadConfig(PadIdx bit, u8 no)
{
    if (bit < PAD_IDX_MAX)
    {
        m_JoyPadConfig[bit] = no;
    }
    else
    {
        NW_ERR( "range over[%d]\n", bit );
    }
}

//------------------------------------------------------------------------------
void
PadWin::SetJoyPadConfigDefault()
{
    nw::ut::MemCpy( m_JoyPadConfig, DEFAULT_JOY_PAD_CONFIG, sizeof(m_JoyPadConfig) );
}

#if defined( NW_USE_NINTENDO_SDK )
//------------------------------------------------------------------------------
static f32 ConvertFromDebugPadAnalogStickValue( int32_t value )
{
    if ( nn::hid::DebugPadAnalogStickCenter <= value )
    {
        static const f32 scale = 1.0f / ( nn::hid::DebugPadAnalogStickMax - nn::hid::DebugPadAnalogStickCenter );
        return scale * ( value - nn::hid::DebugPadAnalogStickCenter );
    }
    else
    {
        static const f32 scale = -1.0f / ( nn::hid::DebugPadAnalogStickCenter - nn::hid::DebugPadAnalogStickMin );
        return scale * ( nn::hid::DebugPadAnalogStickCenter - value );
    }
}

//------------------------------------------------------------------------------
static int32_t ConvertToDebugPadAnalogStickValue( f32 value )
{
    if ( 0.0f <= value )
    {
        return  nn::hid::DebugPadAnalogStickCenter + static_cast<int32_t>((nn::hid::DebugPadAnalogStickMax - nn::hid::DebugPadAnalogStickCenter) * value);
    }
    else
    {
        return  nn::hid::DebugPadAnalogStickCenter + static_cast<int32_t>((nn::hid::DebugPadAnalogStickCenter - nn::hid::DebugPadAnalogStickMin) * value);
    }
}
#endif

//------------------------------------------------------------------------------
void
PadWin::GetPadStatus( RawPadStatus* pad )
{
    NW_ASSERT_NOT_NULL( pad );
    ::ZeroMemory( pad, sizeof( RawPadStatus ) );

    if ( m_PadStatus )
    {
        *pad = m_PadStatus->padStatuses[ m_JoyPadNumber ];
    }
    else
    {
#if defined( NW_USE_NINTENDO_SDK )
        nn::hid::ButtonsState buttons = { 0 };
        buttons.state.SetBit( nn::hid::DebugPadButton::A::Pos, m_PadHold.IsMaskOn(MASK_A) );
        buttons.state.SetBit( nn::hid::DebugPadButton::B::Pos, m_PadHold.IsMaskOn(MASK_B) );
        buttons.state.SetBit( nn::hid::DebugPadButton::X::Pos, m_PadHold.IsMaskOn(MASK_X) );
        buttons.state.SetBit( nn::hid::DebugPadButton::Y::Pos, m_PadHold.IsMaskOn(MASK_Y) );
        buttons.state.SetBit( nn::hid::DebugPadButton::R::Pos, m_PadHold.IsMaskOn(MASK_R) );
        buttons.state.SetBit( nn::hid::DebugPadButton::L::Pos, m_PadHold.IsMaskOn(MASK_L) );
        buttons.state.SetBit( nn::hid::DebugPadButton::ZR::Pos, m_PadHold.IsMaskOn(MASK_ZR) );
        buttons.state.SetBit( nn::hid::DebugPadButton::ZL::Pos, m_PadHold.IsMaskOn(MASK_ZL) );
        buttons.state.SetBit( nn::hid::DebugPadButton::Start::Pos, m_PadHold.IsMaskOn(MASK_START) );
        buttons.state.SetBit( nn::hid::DebugPadButton::Select::Pos, m_PadHold.IsMaskOn(MASK_SELECT) );
        buttons.state.SetBit( nn::hid::DebugPadButton::Right::Pos, m_PadHold.IsMaskOn(MASK_RIGHT) );
        buttons.state.SetBit( nn::hid::DebugPadButton::Left::Pos, m_PadHold.IsMaskOn(MASK_LEFT) );
        buttons.state.SetBit( nn::hid::DebugPadButton::Up::Pos, m_PadHold.IsMaskOn(MASK_UP) );
        buttons.state.SetBit( nn::hid::DebugPadButton::Down::Pos, m_PadHold.IsMaskOn(MASK_DOWN) );
        pad->buttons = buttons;

        pad->analogStickL.x = ConvertToDebugPadAnalogStickValue(m_LeftStick.x);
        pad->analogStickL.y = ConvertToDebugPadAnalogStickValue(m_LeftStick.y);
        pad->analogStickR.x = ConvertToDebugPadAnalogStickValue(m_RightStick.x);
        pad->analogStickR.y = ConvertToDebugPadAnalogStickValue(m_RightStick.y);
#else
        ut::BitFlag32 padHold;
        padHold.SetAllZero();
        padHold.ChangeMask( NW_WINEXT_PAD_BUTTON_A, m_PadHold.IsMaskOn(MASK_A) );
        padHold.ChangeMask( NW_WINEXT_PAD_BUTTON_B, m_PadHold.IsMaskOn(MASK_B) );
        padHold.ChangeMask( NW_WINEXT_PAD_BUTTON_X, m_PadHold.IsMaskOn(MASK_X) );
        padHold.ChangeMask( NW_WINEXT_PAD_BUTTON_Y, m_PadHold.IsMaskOn(MASK_Y) );
        padHold.ChangeMask( NW_WINEXT_PAD_TRIGGER_R, m_PadHold.IsMaskOn(MASK_R) );
        padHold.ChangeMask( NW_WINEXT_PAD_TRIGGER_L, m_PadHold.IsMaskOn(MASK_L) );
        padHold.ChangeMask( NW_WINEXT_PAD_TRIGGER_Z, m_PadHold.IsMaskOn(MASK_Z) );
        padHold.ChangeMask( NW_WINEXT_PAD_BUTTON_START, m_PadHold.IsMaskOn(MASK_START) );
        padHold.ChangeMask( NW_WINEXT_PAD_BUTTON_RIGHT, m_PadHold.IsMaskOn(MASK_RIGHT) );
        padHold.ChangeMask( NW_WINEXT_PAD_BUTTON_LEFT, m_PadHold.IsMaskOn(MASK_LEFT) );
        padHold.ChangeMask( NW_WINEXT_PAD_BUTTON_UP, m_PadHold.IsMaskOn(MASK_UP) );
        padHold.ChangeMask( NW_WINEXT_PAD_BUTTON_DOWN, m_PadHold.IsMaskOn(MASK_DOWN) );
        pad->button = static_cast<u16>(padHold.GetDirect());

        pad->stickX = static_cast<s8>( m_LeftStick.x * 100.f );
        pad->stickY = static_cast<s8>( m_LeftStick.y * 100.f );
        pad->substickX = static_cast<s8>( m_RightStick.x * 100.f );
        pad->substickY = static_cast<s8>( m_RightStick.y * 100.f );
        pad->triggerLeft = static_cast<s8>( m_LeftAnalogTrigger );
        pad->triggerRight = static_cast<s8>( m_RightAnalogTrigger );
        pad->err = NW_WINEXT_PAD_ERR_NONE;
#endif
    }
}


//------------------------------------------------------------------------------
void
PadWin::UpdateImpl()
{
    // パッドをクリア
    m_PadHold.SetAllZero();

    bool isActive = true;
    if ( m_PadStatus )
    {
        isActive = m_PadStatus->isPadEnable || m_PadStatus->isKeyEnable;
    }
    else if ( m_KeyboardMouseDevice )
    {
        // メインウインドウがアクティブな場合に、キーボード・マウスの入力を受け付けます。
        isActive = ( m_KeyboardMouseDevice->GetActiveWindow() == KeyboardMouseDeviceWin::ACTIVE_WINDOW_MAIN ? true : false );
    }

    // キーボードからの情報取得
    if ( m_PadStatus )
    {
        if ( m_PadStatus->keyStatus && m_PadStatus->isKeyEnable )
        {
            for ( int i = 0; i <= IDX_RIGHT; ++i )
            {
                if ( m_PadStatus->keyStatus[ m_VKeyConfig[i] ] != 0 )
                {
                    m_PadHold.SetBitOn( i );
                }
            }
        }
    }
    else if( m_KeyboardMouseDevice )
    {
        if( isActive && m_KeyboardMouseDevice->IsKeyEnable() )
        {
            for ( int i = 0; i <= IDX_RIGHT; ++i )
            {
                if ( m_KeyboardMouseDevice->IsVkeyHold( m_VKeyConfig[i] ) )
                {
                    m_PadHold.SetBitOn( i );
                }
            }
        }
    }

    // ジョイパッドからの情報取得
    bool isJoypadEnable = false;

    if ( m_PadStatus )
    {
        if ( m_PadStatus->isPadEnable )
        {
            isJoypadEnable = true;

#if defined( NW_USE_NINTENDO_SDK )
            const RawPadStatus& padStatus = m_PadStatus->padStatuses[m_JoyPadNumber];
            nn::hid::ButtonsState buttons = padStatus.buttons;

            // パッドの値をマージします。
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::A::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_A );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::B::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_B );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::X::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_X );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::Y::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_Y );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::R::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_R );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::L::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_L );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::ZR::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_ZR );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::ZL::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_ZL );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::Start::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_START );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::Select::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_SELECT );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::Right::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_RIGHT );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::Left::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_LEFT );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::Up::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_UP );
            }
            if ( buttons.state.GetBit( nn::hid::DebugPadButton::Down::Pos ) )
            {
                m_PadHold.SetBitOn( IDX_DOWN );
            }

            // アナログスティックの値を設定します。
            m_LeftStick.x = ConvertFromDebugPadAnalogStickValue(padStatus.analogStickL.x);
            m_LeftStick.y = ConvertFromDebugPadAnalogStickValue(padStatus.analogStickL.y);
            m_RightStick.x = ConvertFromDebugPadAnalogStickValue(padStatus.analogStickR.x);
            m_RightStick.y = ConvertFromDebugPadAnalogStickValue(padStatus.analogStickR.y);

            m_LeftAnalogTrigger = 0;
            m_RightAnalogTrigger = 0;
#else
            const RawPadStatus& padStatus = m_PadStatus->padStatuses[m_JoyPadNumber];
            u32 padHold = padStatus.button;

            // パッドの値をマージします。
            if ( padHold & NW_WINEXT_PAD_BUTTON_A )
            {
                m_PadHold.SetBitOn( IDX_A );
            }
            if ( padHold & NW_WINEXT_PAD_BUTTON_B )
            {
                m_PadHold.SetBitOn( IDX_B );
            }
            if ( padHold & NW_WINEXT_PAD_BUTTON_X )
            {
                m_PadHold.SetBitOn( IDX_X );
            }
            if ( padHold & NW_WINEXT_PAD_BUTTON_Y )
            {
                m_PadHold.SetBitOn( IDX_Y );
            }
            if ( padHold & NW_WINEXT_PAD_TRIGGER_R )
            {
                m_PadHold.SetBitOn( IDX_R );
            }
            if ( padHold & NW_WINEXT_PAD_TRIGGER_L )
            {
                m_PadHold.SetBitOn( IDX_L );
            }
            if ( padHold & NW_WINEXT_PAD_TRIGGER_Z )
            {
                m_PadHold.SetBitOn( IDX_Z );
            }
            if ( padHold & NW_WINEXT_PAD_BUTTON_START )
            {
                m_PadHold.SetBitOn( IDX_START );
            }
            if ( padHold & NW_WINEXT_PAD_BUTTON_RIGHT )
            {
                m_PadHold.SetBitOn( IDX_RIGHT );
            }
            if ( padHold & NW_WINEXT_PAD_BUTTON_LEFT )
            {
                m_PadHold.SetBitOn( IDX_LEFT );
            }
            if ( padHold & NW_WINEXT_PAD_BUTTON_UP )
            {
                m_PadHold.SetBitOn( IDX_UP );
            }
            if ( padHold & NW_WINEXT_PAD_BUTTON_DOWN )
            {
                m_PadHold.SetBitOn( IDX_DOWN );
            }

            // アナログスティックの値を設定します。
            detail::NormalizeStickValueCafe( &m_LeftStick, padStatus.stickX, padStatus.stickY );
            detail::NormalizeStickValueCafe( &m_RightStick, padStatus.substickX, padStatus.substickY );

            m_LeftAnalogTrigger = padStatus.triggerLeft;
            m_RightAnalogTrigger = padStatus.triggerRight;
#endif
        }
    }
    else if ( m_JoyPadDevice != NULL &&
         ( IsJoyPadEnableWhenWindowInactive() || isActive ) &&
         m_JoyPadNumber < m_JoyPadDevice->GetPadNum()
        )
    {
        const JoyPadDeviceWin::PadStatus& pad = m_JoyPadDevice->GetPads( m_JoyPadNumber );
        JOYINFOEX* padInfo = reinterpret_cast<JOYINFOEX*>(const_cast<u32*>(pad.info));

        if ( pad.isEnable )
        {
            isJoypadEnable = true;

            // パッドの値をマージします。
            for ( int i = 0; i <= IDX_R; ++i )
            {
                int number = m_JoyPadConfig[i];

                if ( number > 0 && ((padInfo->dwButtons & (0x1 << (number - 1))) != 0) )
                {
                    m_PadHold.SetBitOn( i );
                }
            }

            // ハットスイッチから十字キーを取得します。
            if (padInfo->dwPOV < 0x8000)
            {
                int pov = (padInfo->dwPOV & 0xf000) >> 12;

                if (1 <= pov && pov <= 3)
                {
                    m_PadHold.SetMaskOn( MASK_RIGHT );
                }

                if (3 <= pov && pov <= 5)
                {
                    m_PadHold.SetMaskOn( MASK_DOWN );
                }

                if (5 <= pov && pov <= 7)
                {
                    m_PadHold.SetMaskOn( MASK_LEFT );
                }

                if (pov == 0 || pov == 1 || pov == 7)
                {
                    m_PadHold.SetMaskOn( MASK_UP );
                }
            }

            // アナログスティックの値を設定します。
            detail::SetStickValueWithClamp( &m_LeftStick.x, padInfo->dwXpos, m_JoyPadConfig[IDX_LS_RIGHT] != 0, m_StickClampMin, m_StickClampMax );
            detail::SetStickValueWithClamp( &m_LeftStick.y, padInfo->dwYpos, m_JoyPadConfig[IDX_LS_UP] != 0, m_StickClampMin, m_StickClampMax );
            detail::SetStickValueWithClamp( &m_RightStick.x, padInfo->dwZpos, m_JoyPadConfig[IDX_RS_RIGHT] != 0, m_StickClampMin, m_StickClampMax );
            detail::SetStickValueWithClamp( &m_RightStick.y, padInfo->dwRpos, m_JoyPadConfig[IDX_RS_UP] != 0, m_StickClampMin, m_StickClampMax );

            // 設定が有効になっていたら円形への正規化を行います。
            if (IsEnableNormalizeAnalogStickAsCircle())
            {
                detail::NormalizeAnalogStickAsCircle( &m_LeftStick );
                detail::NormalizeAnalogStickAsCircle( &m_RightStick );
            }

            m_LeftAnalogTrigger = ( m_PadHold.IsMaskOn(MASK_L) ? 150.f : 0.f );
            m_RightAnalogTrigger = ( m_PadHold.IsMaskOn(MASK_R) ? 150.f : 0.f );
        }
    }

    if ( !isJoypadEnable )
    {
        m_LeftStick = nw::math::VEC2::Zero();
        m_RightStick = nw::math::VEC2::Zero();
        m_LeftAnalogTrigger = 0.f;
        m_RightAnalogTrigger = 0.f;
    }
}

} // namespace demo
} // namespace nw

