﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/demo/cafe/demo_SystemCafe.h>
#include <nw/demo/cafe/demo_FrameBufferCafe.h>
#include <nw/demo/demo_Define.h>
#include <nw/gfnd/gfnd_Graphics.h>
#include <nw/dev/dev_PrimitiveRenderer.h>
#include <nw/dev/dev_Resource.h>
#include <sdk_ver.h>
#include <cafe/mem.h>

namespace nw
{
namespace demo
{

//------------------------------------------------------------------------------
SystemCafe::SystemCafe( const CreateArg& arg )
: m_Arg( arg ),
  m_ScanBufferPtr( NULL ),
  m_DefaultFrameBuffer( NULL ),
  m_ContextState( NULL ),
  m_Pad( NULL ),
  m_WPad( NULL ),
  m_GpuCounters( NULL ),
  m_MeterDrawer( &m_MeterFrame ),
  m_PerformanceMonitor( &m_DevTextWriter ),
  m_TextWriterInitialized( false ),
  m_PointerTextureImage( NULL ),
  m_DisplayState( HIDE )
{
    m_Flags.SetMaskOn( FLAG_IS_BLACK );
}

//------------------------------------------------------------------------------
SystemCafe::~SystemCafe()
{
}

//------------------------------------------------------------------------------
void
SystemCafe::Initialize()
{
    // 高速キャストを有効化します。
#if defined (NW_COMPILER_GHS)
    OSInitFastCast();
#endif

    // デモファイルシステムの初期化を行います。
    if ( m_Arg.fileDeviceManagerInitialize )
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDeviceManager::CreateArg fileDeviceArg;
        fileDeviceArg.allocator = m_Arg.allocator;
        fileSystem->Initialize( fileDeviceArg );
    }

    // Pad の初期化処理を行います。
    {
        nw::demo::PadDeviceCafe::GetInstance()->Initialize();
        nw::demo::WPadDeviceCafe::GetInstance()->Initialize( m_Arg.allocator );

        m_Pad = new( m_Arg.allocator->Alloc( sizeof( nw::demo::PadCafe ) ) ) nw::demo::PadCafe( 0 );
        m_Pad->SetPadDevice( nw::demo::PadDeviceCafe::GetInstance() );

        m_WPad = new( m_Arg.allocator->Alloc( sizeof( nw::demo::WPadCafe ) ) ) nw::demo::WPadCafe( 0 );
        m_WPad->SetWPadDevice( nw::demo::WPadDeviceCafe::GetInstance() );
    }
}

//------------------------------------------------------------------------------
void
SystemCafe::Finalize()
{
    // Pad の終了処理を行います。
    {
        nw::demo::PadDeviceCafe::GetInstance()->Finalize();
        nw::demo::WPadDeviceCafe::GetInstance()->Finalize();

        nw::ut::SafeFree( m_Pad, m_Arg.allocator );
        nw::ut::SafeFree( m_WPad, m_Arg.allocator );
    }

    // デモファイルシステムの終了処理を行います。
    if ( m_Arg.fileDeviceManagerInitialize )
    {
        nw::dev::FileDeviceManager::GetInstance()->Finalize();
    }
}

//------------------------------------------------------------------------------
void
SystemCafe::InitializeGraphicsSystem( int argc /* = 0 */, char** argv /* = NULL */ )
{
    NW_ASSERT_NOT_NULL( m_Arg.allocator );

    {
        MEMHeapHandle hMEM1 = MEMGetBaseHeapHandle(MEM_ARENA_1);
        u32 mem1Size = MEMGetAllocatableSizeForFrmHeapEx(hMEM1, 4);
        void* mem1 = MEMAllocFromFrmHeapEx(hMEM1, mem1Size, 4);
        m_VRAMAllocator.Initialize(mem1, mem1Size);
    }

    // GX2初期化
    u32 gx2InitAttribs[] =
    {
        GX2_INIT_ATTRIB_ARGC,    (u32)argc,  // number of args
        GX2_INIT_ATTRIB_ARGV,    (u32)argv,  // command-line args
        GX2_INIT_ATTRIB_NULL                 // terminates the list
    };
    GX2Init(gx2InitAttribs);

    // GX2のコンテキストを作成し、Graphicsに設定
    m_ContextState = static_cast<GX2ContextState*>( m_Arg.allocator->Alloc(sizeof(GX2ContextState), GX2_CONTEXT_STATE_ALIGNMENT) );

    GX2SetupContextState( m_ContextState );

    // グラフィックスインスタンス生成.
    {
        nw::gfnd::Graphics::SetInstance( new( m_Arg.allocator->Alloc( sizeof( nw::gfnd::Graphics ) ) ) nw::gfnd::Graphics() );
        nw::gfnd::Graphics::GetInstance()->Initialize();
        nw::gfnd::Graphics::GetInstance()->SetGX2ContextState(m_ContextState);
    }

    // カラーバッファ構造体を初期化する
    if (m_Arg.isSRGBWrite)
    {
        GX2InitColorBuffer(&m_ColorBuffer, m_Arg.width, m_Arg.height, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB, GX2_AA_MODE_1X);
    }
    else
    {
        GX2InitColorBuffer(&m_ColorBuffer, m_Arg.width, m_Arg.height, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, GX2_AA_MODE_1X);
    }
    // カラーバッファのメモリを取得する。
    GX2InitColorBufferPtr(&m_ColorBuffer, m_VRAMAllocator.Alloc(m_ColorBuffer.surface.imageSize, m_ColorBuffer.surface.alignment));

    // デプスバッファ構造体を初期化する
    GX2InitDepthBuffer(&m_DepthBuffer, m_Arg.width, m_Arg.height, GX2_SURFACE_FORMAT_TCD_R32_FLOAT, GX2_AA_MODE_1X);
    // デプスバッファのメモリを取得する。
    GX2InitDepthBufferPtr(&m_DepthBuffer, m_VRAMAllocator.Alloc(m_DepthBuffer.surface.imageSize, m_DepthBuffer.surface.alignment));
    // デプスバッファにHiZのバッファを設定する
    {
        u32 hiZSize, hiZAlign;
        GX2CalcDepthBufferHiZInfo(&m_DepthBuffer, &hiZSize, &hiZAlign);
        GX2InitDepthBufferHiZPtr(&m_DepthBuffer, m_VRAMAllocator.Alloc(hiZSize, hiZAlign));
    }

    // スキャンバッファの設定
    {
        u32 scanSize;
        GX2Boolean scaleNeeded;
        // スキャンバッファのサイズを計算する
        GX2CalcTVSize(GX2_TV_RENDER_720, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, GX2_BUFFERING_DOUBLE, &scanSize, &scaleNeeded);
        // スキャンバッファのメモリを取得する。
        m_ScanBufferPtr = m_Arg.allocator->Alloc(scanSize, GX2_SCAN_BUFFER_ALIGNMENT);
        // スキャンバッファを設定する
        GX2SetTVBuffer(m_ScanBufferPtr, scanSize, GX2_TV_RENDER_720, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, GX2_BUFFERING_DOUBLE);
    }

    // デフォルトのフレームバッファの設定
    {
        m_DefaultFrameBuffer = new( m_Arg.allocator->Alloc( sizeof( FrameBufferCafe ) ) ) FrameBufferCafe( nw::math::VEC2(m_Arg.width, m_Arg.height), &m_ColorBuffer, &m_DepthBuffer );

        NW_ASSERT_NOT_NULL( m_DefaultFrameBuffer );

        m_DefaultFrameBuffer->Bind();
    }

    GX2SetTVScale( m_Arg.width, m_Arg.height );

    GX2SetSwapInterval( m_Arg.waitVBlank );

    // PrimitiveRenderer の初期化を行います。
    {
        nw::dev::PrimitiveRenderer* renderer = nw::dev::PrimitiveRenderer::GetInstance();

        if ( m_Arg.primitiveRendererShaderBinary )
        {
            renderer->InitializeFromBinary( m_Arg.allocator, m_Arg.primitiveRendererShaderBinary, m_Arg.primitiveRendererShaderBinarySize );
        }
        else if ( m_Arg.primitiveRendererShaderPath )
        {
            nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.path = m_Arg.primitiveRendererShaderPath;
            loadArg.allocator = m_Arg.allocator;
            u8* binary = fileSystem->Load( loadArg );

            renderer->InitializeFromBinary( m_Arg.allocator, binary, loadArg.readSize );

            fileSystem->Unload( loadArg, binary );
        }
    }

    // デバッグ文字列描画用行列を計算します。
    {
        f32 l     = 0.f;
        f32 r     = static_cast<f32>( m_Arg.width );
        f32 t     = 0.f;
        f32 b     = static_cast<f32>( m_Arg.height );
        f32 zNear = 0.0f;
        f32 zFar  = 1.f;
        nw::math::MTX44Ortho( &m_TextWriterProjMtx, l, r, b, t, zNear, zFar );
        nw::math::MTX34Identity( &m_TextWriterViewMtx );
    }

    // デバッグ文字描画のための DevTextWriter を初期化します。
    if ( m_Arg.fontBinary )
    {
        m_DevTextWriter.Initialize(
            m_Arg.fontBinary,
            m_Arg.fontBinarySize,
            m_Arg.fontShaderBinary,
            m_Arg.fontShaderBinarySize,
            m_Arg.allocator
        );

        m_DevTextWriter.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
        m_TextWriterInitialized = true;
    }
    else if ( m_Arg.fontPath )
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

        u8* fontShaderBinary = NULL;
        u32 fontShaderBinarySize = 0;

        {
            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.path = m_Arg.fontPath;
            loadArg.allocator = m_Arg.allocator;
            loadArg.alignment = nw::font::RESOURCE_ALIGNMENT;
            m_Arg.fontBinary = fileSystem->Load( loadArg );
            m_Arg.fontBinarySize = loadArg.readSize;
        }

        {
            nw::dev::FileDevice::LoadArg loadArg;
            loadArg.path = m_Arg.fontShaderPath;
            loadArg.allocator = m_Arg.allocator;
            loadArg.alignment = 128;
            fontShaderBinary = fileSystem->Load( loadArg );
            fontShaderBinarySize = loadArg.readSize;
        }

        m_DevTextWriter.Initialize(
            m_Arg.fontBinary,
            m_Arg.fontBinarySize,
            fontShaderBinary,
            fontShaderBinarySize,
            m_Arg.allocator
        );

        m_DevTextWriter.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
        m_TextWriterInitialized = true;

        m_Arg.allocator->Free( fontShaderBinary );
    }

    // 負荷メーターの初期化を行います。
    if( m_Arg.drawMeter )
    {
        m_MeterFrame.SetName( "Frame" );
        m_MeterFrame.SetColor( nw::ut::Color4u8( DEMO_COLOR_CPU_FRAME ) );
        m_MeterCalc.SetName( "Calc" );
        m_MeterCalc.SetColor( ut::Color4u8( DEMO_COLOR_CPU_CALC ) );
        m_MeterDraw.SetName( "Draw" );
        m_MeterDraw.SetColor( ut::Color4u8( DEMO_COLOR_CPU_DRAW ) );
        m_MeterGPU.SetName( "GPU" );
        m_MeterGPU.SetColor( ut::Color4u8( DEMO_COLOR_GPU ) );

        m_GpuCounters = reinterpret_cast<u64*>( m_Arg.allocator->Alloc( m_MeterGPU.GetBufferSize(), GX2_DEFAULT_BUFFER_ALIGNMENT ) );
        m_MeterGPU.Initialize( m_GpuCounters );

        m_MeterDrawer.AttachLoadMeter( &m_MeterCalc );
        m_MeterDrawer.AttachLoadMeter( &m_MeterDraw );
        m_MeterDrawer.AttachLoadMeter( &m_MeterGPU );

        m_MeterDrawer.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
        m_MeterDrawer.SetTextWriter( m_TextWriterInitialized ? &m_DevTextWriter : NULL );
        m_MeterDrawer.SetFrameRate( 60.f / m_Arg.waitVBlank );
        m_MeterDrawer.SetMinimizeBarSize( 0.f );

        // 負荷メーターの位置、大きさを調整します。
        // 安全フレームを考慮して、やや小さくします。
        {
            const f32 METER_WIDTH = m_Arg.width * 0.965f;
            const f32 METER_HEIGHT = m_MeterDrawer.GetAdjustedHeight();
            const f32 METER_POS_X = (m_Arg.width - METER_WIDTH ) / 2.f;
            const f32 METER_POS_Y = m_Arg.height - ( METER_HEIGHT + m_Arg.height / 50.f );

            m_MeterDrawer.SetWidth( METER_WIDTH );
            m_MeterDrawer.SetHeight( METER_HEIGHT );
            m_MeterDrawer.SetPosition( nw::math::VEC2( METER_POS_X, METER_POS_Y) );
        }
    }

    // m_PerformanceMonitor の初期化を行います。
    {
        m_PerformanceMonitor.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
    }

    // ポインタカーソル用テクスチャ画像を生成します。
    if ( m_Arg.usePointerCursor )
    {
        m_PointerTextureImage = nw::dev::CreatePointerTexture( m_Arg.allocator, nw::dev::POINTER_ARROW, &m_PointerTexture, &m_PointerTextureGX2 );
    }

    if ( GetDisplayState() == HIDE )
    {
        m_DisplayState = READY;
    }
}

//------------------------------------------------------------------------------
void
SystemCafe::FinalizeGraphicsSystem()
{
    NW_ASSERT_NOT_NULL( m_Arg.allocator );

    if ( m_Arg.usePointerCursor )
    {
        m_Arg.allocator->Free( m_PointerTextureImage );
    }

    if ( m_Arg.drawMeter )
    {
        m_Arg.allocator->Free( m_GpuCounters );
    }

    if ( m_TextWriterInitialized )
    {
        m_DevTextWriter.Finalize();
    }

    if ( m_Arg.fontPath && m_Arg.fontBinary )
    {
        m_Arg.allocator->Free( m_Arg.fontBinary );
    }

    if ( m_Arg.primitiveRendererShaderBinary || m_Arg.primitiveRendererShaderPath )
    {
        nw::dev::PrimitiveRenderer::GetInstance()->Finalize( m_Arg.allocator );
    }

    GX2SetTVEnable( GX2_FALSE );

    nw::ut::SafeFree( m_DefaultFrameBuffer, m_Arg.allocator );
    nw::ut::SafeFree( m_ScanBufferPtr, m_Arg.allocator );

    if ( m_DepthBuffer.hiZPtr )
    {
        nw::ut::SafeFree( m_DepthBuffer.hiZPtr, &m_VRAMAllocator );
    }
    nw::ut::SafeFree( m_DepthBuffer.surface.imagePtr, &m_VRAMAllocator );

    if ( m_ColorBuffer.auxPtr )
    {
        nw::ut::SafeFree( m_ColorBuffer.auxPtr, &m_VRAMAllocator );
    }
    nw::ut::SafeFree( m_ColorBuffer.surface.imagePtr, &m_VRAMAllocator );

    nw::ut::SafeFree( m_ContextState, m_Arg.allocator );

    nw::gfnd::Graphics* gfx = nw::gfnd::Graphics::GetInstance();
    nw::ut::SafeFree( gfx, m_Arg.allocator );

    GX2Shutdown();

    m_VRAMAllocator.Finalize();
    MEMHeapHandle hMEM1 = MEMGetBaseHeapHandle(MEM_ARENA_1);
    MEMFreeToFrmHeap(hMEM1, MEM_FRMHEAP_FREE_ALL);
}

//------------------------------------------------------------------------------
void
SystemCafe::SetViewport( f32 x, f32 y, f32 width, f32 height, f32 nearZ, f32 farZ )
{
    const FrameBuffer* frameBuffer = FrameBuffer::GetBoundFrameBuffer();
    NW_ASSERT_NOT_NULL( frameBuffer );

    nw::gfnd::Graphics::GetInstance()->SetViewport(
        x,
        y,
        width,
        height,
        nearZ,
        farZ,
        frameBuffer->GetSize().x,
        frameBuffer->GetSize().y
    );
}

//------------------------------------------------------------------------------
void
SystemCafe::SetScissor( f32 x, f32 y, f32 width, f32 height )
{
    const FrameBuffer* frameBuffer = FrameBuffer::GetBoundFrameBuffer();
    NW_ASSERT_NOT_NULL( frameBuffer );

    nw::gfnd::Graphics::GetInstance()->SetScissor(
        x,
        y,
        width,
        height,
        frameBuffer->GetSize().x,
        frameBuffer->GetSize().y
    );
}

//------------------------------------------------------------------------------
void
SystemCafe::ClearFrameBuffers()
{
    ClearFrameBuffersDetail( CLEAR_FLAG_COLOR | CLEAR_FLAG_DEPTH | CLEAR_FLAG_STENCIL, m_Arg.clearColor, m_Arg.clearDepth, 0 );
}

//------------------------------------------------------------------------------
void
SystemCafe::ClearFrameBuffersDetail( u8 flag, const nw::ut::Color4u8& color, f32 depth, u32 stencil )
{
    const FrameBufferCafe* frameBuffer = static_cast<const FrameBufferCafe*>( FrameBuffer::GetBoundFrameBuffer() );
    NW_ASSERT_NOT_NULL( frameBuffer );

    GX2ColorBuffer* colorBuffer = const_cast< GX2ColorBuffer* >( frameBuffer->GetGX2ColorBuffer() );
    GX2DepthBuffer* depthBuffer = const_cast< GX2DepthBuffer* >( frameBuffer->GetGX2DepthBuffer() );

    // Cafeのクリア関数はViewportの設定の影響を受けるため、フレームバッファと等しいサイズに設定する
    GX2SetViewport( 0, 0, (colorBuffer->surface.width >> colorBuffer->viewMip), (colorBuffer->surface.height >> colorBuffer->viewMip), 0.0f, 1.0f );
    GX2SetScissor( 0, 0,  (colorBuffer->surface.width >> colorBuffer->viewMip), (colorBuffer->surface.height >> colorBuffer->viewMip) );

    nw::ut::Color4f fcolor = static_cast<nw::ut::Color4f>( color );

    if ( flag & CLEAR_FLAG_COLOR )
    {
        if ( (flag & CLEAR_FLAG_DEPTH) != 0 && (flag & CLEAR_FLAG_STENCIL) != 0 )
        {
            GX2ClearBuffersEx(colorBuffer, depthBuffer, fcolor.r, fcolor.g, fcolor.b, fcolor.a, depth, stencil, GX2_CLEAR_BOTH);
        }
        else if ( flag & CLEAR_FLAG_DEPTH )
        {
            GX2ClearBuffersEx(colorBuffer, depthBuffer, fcolor.r, fcolor.g, fcolor.b, fcolor.a, depth, stencil, GX2_CLEAR_DEPTH);
        }
        else if ( flag & CLEAR_FLAG_STENCIL )
        {
            GX2ClearBuffersEx(colorBuffer, depthBuffer, fcolor.r, fcolor.g, fcolor.b, fcolor.a, depth, stencil, GX2_CLEAR_STENCIL);
        }
        else
        {
            GX2ClearColor(colorBuffer, fcolor.r, fcolor.g, fcolor.b, fcolor.a);
        }
    }
    else
    {
        if ( (flag & CLEAR_FLAG_DEPTH) != 0 && (flag & CLEAR_FLAG_STENCIL) != 0 )
        {
            GX2ClearDepthStencilEx(depthBuffer, depth, stencil, GX2_CLEAR_BOTH);
        }
        else if ( flag & CLEAR_FLAG_DEPTH )
        {
            GX2ClearDepthStencilEx(depthBuffer, depth, stencil, GX2_CLEAR_DEPTH);
        }
        else if ( flag & CLEAR_FLAG_STENCIL )
        {
            GX2ClearDepthStencilEx(depthBuffer, depth, stencil, GX2_CLEAR_STENCIL);
        }
    }

    // Clearした後にContextStateを戻す
    GX2SetContextState( m_ContextState );
}

//------------------------------------------------------------------------------
void
SystemCafe::SwapBuffer()
{
    if (m_Arg.isSRGBWrite)
    {
        // フレームバッファのフォーマットがGX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGBになっていると、
        // スキャンバッファへのコピー時に逆にsRGBフェッチがかかって元に戻ってしまうため、
        // スキャンバッファへのコピー時はフォーマットをGX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORMにする
        m_ColorBuffer.surface.format = GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM;
        GX2InitColorBufferRegs(&m_ColorBuffer);
    }
    // カラーバッファをスキャンバッファに転送して画面をスワップ
    GX2SwapBuffers(&m_ColorBuffer);
    // context stateを戻す
    GX2SetContextState(nw::gfnd::Graphics::GetInstance()->GetGX2ContextState());
    if (m_Arg.isSRGBWrite)
    {
        // スキャンバッファへのコピーが終わったので元に戻す
        m_ColorBuffer.surface.format = GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_SRGB;
        GX2InitColorBufferRegs(&m_ColorBuffer);
    }
    // カラーバッファを再設定、この処理はスワップの後に必ず行う必要がある
    m_DefaultFrameBuffer->Bind();
    // スワップコマンドが確実に実行されるようにコマンドをフラッシュする
    GX2Flush();

    //  画面表示切り替え
    if ( m_Flags.IsMaskOn( FLAG_CHANGE_BLACK ) )
    {
        SetBlack( ! IsBlack());
        m_Flags.SetMaskOff( FLAG_CHANGE_BLACK );
    }

    // カラーバッファへの描画完了を待つ
    GX2DrawDone();

    // 表示開始.
    if ( GetDisplayState() == READY )
    {
        ReserveSetBlack( false );
        m_DisplayState = SHOW;
    }
}

//------------------------------------------------------------------------------
void
SystemCafe::WaitForVBlank()
{
    GX2WaitForFlip();
}

//------------------------------------------------------------------------------
void
SystemCafe::BeginFrame()
{
    m_MeterFrame.BeginMeasure();
}

//------------------------------------------------------------------------------
void
SystemCafe::EndFrame()
{
    ClearTextWriter();
    m_PerformanceMonitor.EndFrame();
    m_MeterFrame.EndMeasure();
    m_MeterDrawer.EndFrame();
}

//------------------------------------------------------------------------------
void
SystemCafe::UpdatePad()
{
    nw::demo::PadDeviceCafe::GetInstance()->Update();
    nw::demo::WPadDeviceCafe::GetInstance()->Update();

    m_Pad->Update();
    m_WPad->Update();
}

//------------------------------------------------------------------------------
void
SystemCafe::DrawLoadMeters()
{
    if( m_Arg.drawMeter )
    {
        m_MeterDrawer.Draw();
    }
}

//------------------------------------------------------------------------------
void
SystemCafe::ClearTextWriter()
{
    if ( m_TextWriterInitialized )
    {
        m_DevTextWriter.Clear();
    }
}

//------------------------------------------------------------------------------
void
SystemCafe::DrawPointerCursor( f32 posX, f32 posY, f32 width, f32 height )
{
    if ( m_Arg.usePointerCursor )
    {
        static nw::math::MTX44 projMtx;
        static nw::math::MTX34 viewMtx;

        const f32 NEAR_CLIP = 0.0f;
        const f32 FAR_CLIP  = 1.0f;

        nw::math::MTX44Ortho(
            &projMtx,
            -width / 2.0f,  // left
             width / 2.0f,  // right
             height / 2.0f, // bottom
            -height / 2.0f, // top
            NEAR_CLIP,      // near
            FAR_CLIP        // far
        );
        nw::math::MTX34Identity( &viewMtx );


        nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
        graphics->SetBlendEnable( true );
        graphics->SetCullingMode( nw::gfnd::Graphics::CULLING_MODE_NONE );

        nw::dev::PrimitiveRenderer* renderer = nw::dev::PrimitiveRenderer::GetInstance();
        renderer->SetProjectionMtx( &projMtx );
        renderer->SetViewMtx( &viewMtx );

        renderer->Begin();
        {
            renderer->SetModelMatrix( nw::math::MTX34::Identity() );

            nw::dev::PrimitiveRenderer::QuadArg arg;
            arg.SetCornerAndSize(
                posX * width / 2.0f,
                posY * height / 2.0f,
                0.0f,
                static_cast<f32>( nw::dev::POINTER_TEXTURE_SIZE ),
                static_cast<f32>( nw::dev::POINTER_TEXTURE_SIZE )
                );
            arg.SetColor( nw::ut::Color4u8(nw::ut::Color4u8::WHITE) );
            renderer->DrawQuad( m_PointerTexture, arg );
        }
        renderer->End();
    }
}

//------------------------------------------------------------------------------
void
SystemCafe::SetBlack(bool is_black)
{
    if (m_Flags.IsMaskOn( FLAG_IS_BLACK ) != is_black)
    {
        GX2SetTVEnable(static_cast<GX2Boolean>( ! is_black));
        m_Flags.ChangeMask( FLAG_IS_BLACK, is_black );
    }
}

} // namespace nw::demo
} // namespace nw
