﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_W_PAD_WIN_H_
#define NW_DEMO_W_PAD_WIN_H_

#include <nw/demo/pad/demo_Pad.h>
#include <nw/demo/pad/win/demo_KeyboardMouseDeviceWin.h>
#include <nw/demo/pad/win/demo_JoyPadDeviceWin.h>

namespace nw
{
namespace demo
{

//---------------------------------------------------------------------------
//! @brief   キーボードとマウスを用いて PC で WPAD を再現するクラスです。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
class WPadWin : public Pad
{
public:
    NW_UT_RUNTIME_TYPEINFO(Pad)

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    WPadWin();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~WPadWin() {}


    //----------------------------------------
    //! @name 仮想キーコードの割り当て
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        パッドのボタンをどの仮想キーコードに割り当てるかを設定します。
    //!
    //! @param[in]    keyConfig ボタンの割り当てです。
    //---------------------------------------------------------------------------
    void SetVKeyConfig(const u8* keyConfig);

    //---------------------------------------------------------------------------
    //! @brief        パッドのボタンをどの仮想キーコードに割り当てるかを設定します。
    //!
    //! @param[in]    bit       設定するパッドのビット番号です。
    //! @param[in]    vkey      割り当てる仮想キーコードです。
    //---------------------------------------------------------------------------
    void SetVKeyConfig(PadIdx bit, u8 vkey);

    //---------------------------------------------------------------------------
    //! @brief        パッドのボタンをどの仮想キーコードに割り当てるかを、デフォルト値に設定します。
    //---------------------------------------------------------------------------
    void SetVKeyConfigDefault();

    //  @}

    //----------------------------------------
    //! @name ポインタの原点の設定
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        ポインタの座標が中心原点になっているかを取得します。
    //!
    //! @return       中心原点の場合、 true を返します。
    //!               左上原点の場合、 false を返します。
    //---------------------------------------------------------------------------
    bool IsPointerCenterOrigin() const { return m_Flag.IsMaskOn( MASK_IS_POINTER_CENTER_ORIGIN ); }

    //---------------------------------------------------------------------------
    //! @brief        ポインタの座標を中心原点にするかを設定します。
    //!
    //! @param[in]    isCenter  中心原点に設定する場合、 true を指定します。
    //!                         左上原点に設定する場合、 false を指定します。
    //---------------------------------------------------------------------------
    void SetPointerCenterOrigin(bool isCenter) { m_Flag.ChangeMask( MASK_IS_POINTER_CENTER_ORIGIN, isCenter ); }

    //---------------------------------------------------------------------------
    //! @brief        ポインタの座標を中心原点としているときに、ポインタが外れたと判定する閾値を取得します。
    //!
    //! @return       閾値を返します。
    //---------------------------------------------------------------------------
    f32 GetPointerOffThreshold() const { return m_PointerOffThreshold; }

    //---------------------------------------------------------------------------
    //! @brief        ポインタの座標を中心原点としているときに、ポインタが外れたと判定する閾値を設定します。
    //!
    //! @param[in]    threshold 閾値です。
    //---------------------------------------------------------------------------
    void SetPointerOffThreshold(f32 threshold)
    {
        NW_ASSERTMSG( threshold >= 0.f, "invalid threshold: %f", threshold );
        m_PointerOffThreshold = threshold;
    }

    //  @}

    //---------------------------------------------------------------------------
    //! @brief        キーボード・マウスデバイスを設定します。
    //!
    //! @param[in]    device    設定するデバイスへのポインタです。
    //---------------------------------------------------------------------------
    void SetKeyboardMouseDevice( KeyboardMouseDeviceWin* device ){ m_KeyboardMouseDevice = device; }


protected:
    enum FlagMask
    {
        MASK_IS_POINTER_CENTER_ORIGIN = 1  //!< ポインタの座標をWiiリモコン等と同様に中心原点にするかのフラグ値です。
    };


    //! @brief        更新処理の実装です。
    virtual void UpdateImpl();


    u8 m_VKeyConfig[PAD_IDX_MAX];   //!< マウスとキーボードで使われる仮想キーコードの割り当てです。

    nw::ut::BitFlag32 m_Flag;
    f32 m_PointerOffThreshold;                              //!< 中心原点のときに、ポインタを Off と判定する閾値です。
    static const u8 DEFAULT_V_KEY_CONFIG[PAD_IDX_MAX];      //!< デフォルトの仮想キーコードの割り当てです。

    KeyboardMouseDeviceWin* m_KeyboardMouseDevice;          //!< 使用するキーボード・マウスデバイスです。
};


} // namespace demo
} // namespace nw

#endif // NW_DEMO_W_PAD_WIN_H_
