﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_MOUSE_WIN_H_
#define NW_DEMO_MOUSE_WIN_H_

#include <nw/demo/pad/demo_Mouse.h>
#include <nw/demo/pad/win/demo_KeyboardMouseDeviceWin.h>

namespace nw
{
namespace demo
{

//---------------------------------------------------------------------------
//! @brief        PC でマウス入力を扱うクラスです。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
class MouseWin : public Mouse
{
public:
    NW_UT_RUNTIME_TYPEINFO(Mouse)

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    MouseWin();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~MouseWin() {}


    //----------------------------------------
    //! @name ポインタの原点の設定
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        ポインタの座標が中心原点になっているかを取得します。
    //!
    //! @return       中心原点の場合、 true を返します。
    //!               左上原点の場合、 false を返します。
    //---------------------------------------------------------------------------
    bool IsPointerCenterOrigin() const { return m_Flag.IsMaskOn( MASK_IS_POINTER_CENTER_ORIGIN ); }

    //---------------------------------------------------------------------------
    //! @brief        ポインタの座標を中心原点にするかを設定します。
    //!
    //! @param[in]    isCenter  中心原点に設定する場合、 true を指定します。
    //!                         左上原点に設定する場合、 false を指定します。
    //---------------------------------------------------------------------------
    void SetPointerCenterOrigin(bool isCenter) { m_Flag.ChangeMask( MASK_IS_POINTER_CENTER_ORIGIN, isCenter ); }

    //---------------------------------------------------------------------------
    //! @brief        ポインタの座標を中心原点としているときに、ポインタが外れたと判定する閾値を取得します。
    //!
    //! @return       閾値を返します。
    //---------------------------------------------------------------------------
    f32 GetPointerOffThreshold() const { return m_PointerOffThreshold; }

    //---------------------------------------------------------------------------
    //! @brief        ポインタの座標を中心原点としているときに、ポインタが外れたと判定する閾値を設定します。
    //!
    //! @param[in]    threshold 閾値です。
    //---------------------------------------------------------------------------
    void SetPointerOffThreshold(f32 threshold)
    {
        NW_ASSERTMSG( threshold >= 0.f, "invalid threshold: %f", threshold );
        m_PointerOffThreshold = threshold;
    }

    //  @}

    //---------------------------------------------------------------------------
    //! @brief        キーボード・マウスデバイスを設定します。
    //!
    //! @param[in]    device    設定するデバイスへのポインタです。
    //---------------------------------------------------------------------------
    void SetKeyboardMouseDevice( KeyboardMouseDeviceWin* device ) { m_KeyboardMouseDevice = device; }

    //---------------------------------------------------------------------------
    //! @brief        キーボード・マウスデバイスを取得します。
    //!
    //! @return       キーボード・マウスデバイスを返します。
    //---------------------------------------------------------------------------
    KeyboardMouseDeviceWin* GetKeyboardMouseDevice() const { return m_KeyboardMouseDevice; }


protected:
    enum FlagMask
    {
        MASK_IS_POINTER_CENTER_ORIGIN = 1  //!< ポインタの座標をWiiリモコン等と同様に中心原点にするかのフラグ値です。
    };

    //! @brief マウス入力更新の実装です。
    void UpdateImpl();

    nw::ut::BitFlag32       m_Flag;                         //!< 設定フラグです。
    f32                     m_PointerOffThreshold;          //!< 中心原点のときに、ポインタを Off と判定する閾値です。

    KeyboardMouseDeviceWin* m_KeyboardMouseDevice;          //!< 使用するキーボード・マウスデバイスです。
};

} // namespace demo
} // namespace nw

#endif // NW_DEMO_MOUSE_WIN_H_
