﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_MOUSE_H_
#define NW_DEMO_MOUSE_H_

#include <nw/demo/pad/demo_PadBase.h>

namespace nw
{
namespace demo
{

//---------------------------------------------------------------------------
//! @brief マウス状態を表す構造体です。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
struct MouseStatus
{
    u32  button;        //!< マウスボタンの状態です。ビット定義は Mouse::PadMask と同じです。
    f32  posX;          //!< ポインタ位置の X 座標です。
    f32  posY;          //!< ポインタ位置の Y 座標です。
    s32  wheel;         //!< マウスホイールの回転量です。
    bool isDoubleClick; //!< ダブルクリックされたかを表します。
    bool isEnable;      //!< マウスが有効かを表します。

    //! @brief コンストラクタです。
    MouseStatus()
      : button( 0 ),
        posX( 0 ),
        posY( 0 ),
        wheel( 0 ),
        isDoubleClick( false ),
        isEnable( false )
    {}
};

//---------------------------------------------------------------------------
//! @brief        マウス入力を扱うクラスです。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
class Mouse : public PadBase
{
public:
    NW_UT_RUNTIME_TYPEINFO(PadBase)

    //! @brief マウスボタンのビット番号による定義です。
    enum PadIdx
    {
        IDX_LBUTTON = 0,
        IDX_RBUTTON,
        IDX_MBUTTON,
        PAD_IDX_MAX
    };

    //! @brief マウスボタンのマスクによる定義です。
    enum PadMask
    {
        MASK_LBUTTON    = 1 << IDX_LBUTTON,
        MASK_RBUTTON    = 1 << IDX_RBUTTON,
        MASK_MBUTTON    = 1 << IDX_MBUTTON
    };


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    Mouse();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~Mouse() {}


    //---------------------------------------------------------------------------
    //! @brief        マウス入力を更新します。
    //!
    //!               設定された MouseStatus または KeyboardMouseDevice を用いて
    //!               マウス入力を更新します。
    //!               MouseStatus を優先して使用します。
    //---------------------------------------------------------------------------
    virtual void Update();

    //---------------------------------------------------------------------------
    //! @brief        ダブルクリックの有無を取得します。
    //!
    //! @return       ダブルクリックの有無を返します。
    //---------------------------------------------------------------------------
    const bool IsDoubleClick() const { return m_IsDoubleClick; }

    //---------------------------------------------------------------------------
    //! @brief        ホイール回転差分を取得します。
    //!
    //! @return       ホイール回転差分を返します。
    //---------------------------------------------------------------------------
    const s32 GetWheel() const { return m_WheelDelta; }

    //---------------------------------------------------------------------------
    //! @brief        マウス状態を設定します。
    //!
    //! @param[in]    status    設定するマウス状態です。
    //---------------------------------------------------------------------------
    void SetMouseStatus( MouseStatus* status ) { m_MouseStatus = status; }

    //---------------------------------------------------------------------------
    //! @brief        マウス状態を取得します。
    //!
    //! @return       マウス状態を返します。
    //---------------------------------------------------------------------------
    const MouseStatus* GetMouseStatus() const { return m_MouseStatus; }


protected:
    //! @brief マウス入力更新の実装です。
    virtual void UpdateImpl() = 0;

    s32          m_WheelDelta;
    bool         m_IsDoubleClick;
    MouseStatus* m_MouseStatus;
};

} // namespace demo
} // namespace nw

#endif // NW_DEMO_MOUSE_H_
