﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_HEAP_ALLOCATOR_H_
#define NW_DEMO_HEAP_ALLOCATOR_H_

#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_MemoryAllocator.h>

namespace nw
{
namespace demo
{

//---------------------------------------------------------------------------
//! @brief   デモ用のヒープアロケータです。
//---------------------------------------------------------------------------
class IHeapAllocator : public nw::ut::IAllocator
{
public:
    //---------------------------------------------------------------------------
    //! @brief     コンストラクタです。
    //---------------------------------------------------------------------------
    IHeapAllocator() : nw::ut::IAllocator() {}

    //---------------------------------------------------------------------------
    //! @brief     デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~IHeapAllocator() {}


    //---------------------------------------------------------------------------
    //! @brief     メモリを確保します。
    //---------------------------------------------------------------------------
    virtual void* Alloc(size_t size, u32 alignment) = 0;

    //---------------------------------------------------------------------------
    //! @brief     メモリを解放します。
    //---------------------------------------------------------------------------
    virtual void Free(void* memory) = 0;

    //---------------------------------------------------------------------------
    //! @brief     空きメモリサイズを返します。
    //!
    //! @return    空きメモリサイズです。
    //---------------------------------------------------------------------------
    virtual u32 GetFreeSize() { return 0; }

    //---------------------------------------------------------------------------
    //! @brief     メインメモリの状態を表示します。
    //---------------------------------------------------------------------------
    virtual void Dump() {}
};

//---------------------------------------------------------------------------
//! @brief        nw::ut::MemoryAllocator をラップして、nw::demo::IHeapAllocator にしたものです。
//---------------------------------------------------------------------------
class HeapAllocator : public IHeapAllocator
{
public:
    //! @brief コンストラクタです。
    HeapAllocator()
    : nw::demo::IHeapAllocator()
     , m_Allocator( NULL )
    {}

    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //!
    //! @param[out]   allocator 設定するアロケータです。
    //---------------------------------------------------------------------------
    void Initialize( nw::ut::MemoryAllocator* allocator )
    {
        m_Allocator = allocator;
    }

    //---------------------------------------------------------------------------
    //! @brief        終了処理です。
    //---------------------------------------------------------------------------
    void Finalize()
    {
        m_Allocator = NULL;
    }

    //---------------------------------------------------------------------------
    //! @brief     メモリを確保します。
    //---------------------------------------------------------------------------
    virtual void* Alloc(size_t size, u32 alignment)
    {
        NW_ASSERT_NOT_NULL(  m_Allocator );
        return m_Allocator->Alloc( size, alignment );
    }

    //---------------------------------------------------------------------------
    //! @brief     メモリを解放します。
    //---------------------------------------------------------------------------
    virtual void Free(void* memory)
    {
        NW_ASSERT_NOT_NULL( m_Allocator );
        m_Allocator->Free( memory );
    }

    //---------------------------------------------------------------------------
    //! @brief     空きメモリサイズを返します。
    //!
    //! @return    空きメモリサイズです。
    //---------------------------------------------------------------------------
    virtual u32 GetFreeSize() { return static_cast<u32>(m_Allocator->GetFreeSize()); }

    //---------------------------------------------------------------------------
    //! @brief     メモリの状態をダンプします。
    //---------------------------------------------------------------------------
    virtual void Dump()
    {
        m_Allocator->Dump();
    }

private:
    ut::MemoryAllocator* m_Allocator;
};


} // namespace nw::demo
} // namespace nw

#endif // NW_DEMO_HEAP_ALLOCATOR_H_
