﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_SYSTEM_CAFE_H_
#define NW_DEMO_SYSTEM_CAFE_H_

#include <nw/demo/cafe/demo_FrameBufferCafe.h>
#include <nw/demo/cafe/demo_PerformanceMonitorCafe.h>
#include <nw/demo/pad/cafe/demo_PadCafe.h>
#include <nw/demo/pad/cafe/demo_WPadCafe.h>
#include <nw/demo/demo_Utility.h>
#include <nw/dev/dev_MeterDrawer.h>
#include <nw/dev/dev_FileDeviceManager.h>
#include <nw/ut/ut_MemoryAllocator.h>
#include <nw/ut/ut_Color.h>
#include <nw/ut/ut_BitFlag.h>
#include <nw/ut/os/ut_Time.h>
#include <cafe/gx2.h>

namespace nw
{
namespace demo
{

const u32 DEFAULT_SCREEN_WIDTH = 1280; //!< デフォルトの画面幅です。
const u32 DEFAULT_SCREEN_HEIGHT = 720; //!< デフォルトの画面の高さです。

//---------------------------------------------------------------------------
//! @brief        Cafe 用デモシステムクラスです。
//---------------------------------------------------------------------------
class SystemCafe
{
protected:
    //! @brief 画面表示切り替え用フラグです。
    enum
    {
        FLAG_IS_BLACK = 0x1,
        FLAG_CHANGE_BLACK = 0x2
    };

public:
    //! @brief コンストラクタに渡す生成パラメータです。
    struct CreateArg
    {
        nw::ut::IAllocator* allocator;              //!< デモシステムで用いるアロケータです。
        u32                 waitVBlank;             //!< VBlank 待ちする回数 (1なら60fps、2なら30fps, 3なら15fps) です。
        nw::ut::Color4u8    clearColor;             //!< 画面クリア色です。
        f32                 clearDepth;             //!< 画面クリアデプスです。
        u32                 width;                  //!< 画面の横幅です。
        u32                 height;                 //!< 画面の縦幅です。
        u8*                 fontBinary;             //!< デバッグ文字描画に用いるフォントバイナリへのポインタです。
        u32                 fontBinarySize;         //!< デバッグ文字描画に用いるフォントバイナリのサイズです。
        u8*                 fontShaderBinary;       //!< デバッグ文字描画に用いるシェーダーバイナリへのポインタです。
        u32                 fontShaderBinarySize;   //!< デバッグ文字描画に用いるシェーダーバイナリのサイズです。
        const char*         fontPath;               //!< デバッグ文字描画に用いるフォントのパスを指定します。（バイナリを直接設定せずファイルからロードする場合に指定します。デモファイルシステムが初期化されている必要があります。）
        const char*         fontShaderPath;         //!< デバッグ文字描画に用いるシェーダーのパスを指定します。（バイナリを直接設定せずファイルからロードする場合に指定します。デモファイルシステムが初期化されている必要があります。）
        bool                drawMeter;              //!< 負荷メーターを表示するか指定します。
        bool                usePointerCursor;       //!< ポインタカーソルを有効にするかを指定します。
        u8*                 primitiveRendererShaderBinary;      //!< PrimitiveRenderer で用いるシェーダーバイナリへのポインタです。
        u32                 primitiveRendererShaderBinarySize;  //!< PrimitiveRenderer で用いるシェーダーバイナリのサイズです。
        const char*         primitiveRendererShaderPath;        //!< PrimitiveRenderer で用いるシェーダのパスを指定します。（バイナリを直接設定せずファイルからロードする場合に指定します。デモファイルシステムが初期化されている必要があります。）
        bool                fileDeviceManagerInitialize;        //!< デモファイルシステムを初期化するかを指定します。
        bool                isSRGBWrite;                        //!< フレームバッファに書き込む際にリニアからsRGBに変換する処理を有効にするかを指定します。

        //! @brief コンストラクタです。
        CreateArg() :
            allocator( NULL ),
            waitVBlank( 1 ),
            clearColor( SRGBToLinear(nw::ut::Color4u8::GRAY) ),
            clearDepth( 1.f ),
            width( DEFAULT_SCREEN_WIDTH ),
            height( DEFAULT_SCREEN_HEIGHT ),
            fontBinary( NULL ),
            fontBinarySize( 0 ),
            fontShaderBinary( NULL ),
            fontShaderBinarySize( 0 ),
            fontPath( NULL ),
            fontShaderPath( NULL ),
            drawMeter( false ),
            usePointerCursor( false ),
            primitiveRendererShaderBinary( NULL ),
            primitiveRendererShaderBinarySize( 0 ),
            primitiveRendererShaderPath( NULL ),
            fileDeviceManagerInitialize( false ),
            isSRGBWrite( true )
        {}
    };

    //! @brief clear メソッドに渡すビットマスクです。
    enum ClearFlag
    {
        CLEAR_FLAG_NONE     = 0,        //!< 何も指定しない。
        CLEAR_FLAG_COLOR    = 1 << 0,   //!< 色要素をクリアする。
        CLEAR_FLAG_DEPTH    = 1 << 1,   //!< 深度要素をクリアする。
        CLEAR_FLAG_STENCIL  = 1 << 2    //!< ステンシル要素をクリアする。
    };

    //! @brief 画面表示状態です。
    enum DisplayState
    {
        HIDE,       //!< 非表示状態です。
        READY,      //!< 表示準備状態(次のVSyncで表示)です。
        SHOW        //!< 表示中です。
    };

    //! @brief SetProcessPriority に渡すパラメータです。
    enum ProcessPriority
    {
        IDLE,       //!< 低優先度です。
        NORMAL,     //!< 普通優先度です。
        HIGH,       //!< 高優先度です。
        REAL_TIME   //!< 最高優先度です。
    };


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    arg      生成パラメータです。
    //---------------------------------------------------------------------------
    /* ctor */ explicit SystemCafe( const CreateArg& arg );

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~SystemCafe();


    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //---------------------------------------------------------------------------
    void Initialize();

    //---------------------------------------------------------------------------
    //! @brief        終了処理です。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief        グラフィックエンジンの初期化を行います。
    //!
    //! @param[in]    argc     GX2Init に渡す引数、argvの要素数です。
    //! @param[in]    argv     GX2Init に渡す引数です。
    //---------------------------------------------------------------------------
    void InitializeGraphicsSystem( int argc = 0, char** argv = NULL );

    //---------------------------------------------------------------------------
    //! @brief        グラフィックスエンジンの終了処理を行います。
    //---------------------------------------------------------------------------
    void FinalizeGraphicsSystem();


    //---------------------------------------------------------------------------
    //! @brief        現在のフレームレートを計算して取得します。
    //!
    //! @return       現在のフレームレートを返します。
    //---------------------------------------------------------------------------
    f32 CalcFps() { return 60.f / m_Arg.waitVBlank; }

    //---------------------------------------------------------------------------
    //! @brief        タイトルバーに文字列を表示します。
    //!
    //! @param[in]    caption   表示する文字列です。
    //---------------------------------------------------------------------------
    void SetCaption( const char16* caption ) { NW_UNUSED_VARIABLE( caption ); }

    //---------------------------------------------------------------------------
    //! @brief        VBlank 待ち回数を変更します。
    //!
    //! @param[in]    interval VBlank 待ち回数です。
    //---------------------------------------------------------------------------
    void SetVBlankWaitInterval( u32 interval ) { m_Arg.waitVBlank = interval; }

    //---------------------------------------------------------------------------
    //! @brief        黒画面か否かを返します。
    //!
    //! @return       黒画面の場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsBlack() const { return m_Flags.IsMaskOn( FLAG_IS_BLACK ); }

    //---------------------------------------------------------------------------
    //! @brief        次のフレーム更新タイミングで、VISetBlack を行なうリクエストを設定します。
    //---------------------------------------------------------------------------
    void ReserveSetBlack(bool black)
    {
        if (black != IsBlack())
        {
            m_Flags.SetMaskOn( FLAG_CHANGE_BLACK );
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        DefaultFrameBuffer を返します。
    //!
    //! @return       DefaultFrameBuffer です。
    //---------------------------------------------------------------------------
    FrameBuffer* GetDefaultFrameBuffer() { return m_DefaultFrameBuffer; }

    //---------------------------------------------------------------------------
    //! @brief        プロセスの動作優先度を変更します。
    //!
    //! @param[in]    priority 設定する優先度です。
    //!
    //! @return       設定に成功した場合、 true を返します。
    //---------------------------------------------------------------------------
    bool SetProcessPriority( ProcessPriority /*priority*/ ) { return false; }

    //---------------------------------------------------------------------------
    //! @brief        設定されているアロケータを取得します。
    //!
    //! @return       アロケータを返します。
    //---------------------------------------------------------------------------
    nw::ut::IAllocator* GetAllocator() { return m_Arg.allocator; }

    //---------------------------------------------------------------------------
    //! @brief        VRAM アロケータを取得します。
    //!
    //! @return       VRAM アロケータを返します。
    //---------------------------------------------------------------------------
    nw::ut::IAllocator* GetVRAMAllocator() { return &m_VRAMAllocator; }

    //---------------------------------------------------------------------------
    //! @brief        画面の横幅を取得します。
    //!
    //! @return       画面の横幅を返します。
    //---------------------------------------------------------------------------
    s32 GetWidth() { return m_Arg.width; }

    //---------------------------------------------------------------------------
    //! @brief        画面の縦幅を取得します。
    //!
    //! @return       画面の縦幅を返します。
    //---------------------------------------------------------------------------
    s32 GetHeight() { return m_Arg.height; }

    //---------------------------------------------------------------------------
    //! @brief        画面表示状態を取得します。
    //!
    //! @return       画面表示状態を返します。
    //---------------------------------------------------------------------------
    DisplayState GetDisplayState() const { return m_DisplayState; }

    //---------------------------------------------------------------------------
    //! @brief        終了判定を取得します。
    //!
    //! @return       終了判定フラグを返します。
    //---------------------------------------------------------------------------
    bool IsExiting() const
    {
        // メモリーリーク検出が有効な場合、パッド操作で終了判定を true にします。
#if defined(NW_DEBUG_CHECK_MEMORY_LEAK)
        if( m_Pad->IsHoldAll( nw::demo::Pad::MASK_L | nw::demo::Pad::MASK_R | nw::demo::Pad::MASK_X ) )
        {
            return true;
        }
#endif //defined(NW_DEBUG_CHECK_MEMORY_LEAK)

        return false;
    }


    //---------------------------------------------------------------------------
    //! @brief        ビューポートを設定します。
    //!
    //! @param[in]    x        ビューポートの左上原点 X 座標です。
    //! @param[in]    y        ビューポートの左上原点 Y 座標です。
    //! @param[in]    width    ビューポートの幅です。
    //! @param[in]    height   ビューポートの高さです。
    //! @param[in]    nearZ    ビューポートの near 値です。
    //! @param[in]    farZ     ビューポートの far 値です。
    //---------------------------------------------------------------------------
    void SetViewport( f32 x, f32 y, f32 width, f32 height, f32 nearZ, f32 farZ );

    //---------------------------------------------------------------------------
    //! @brief        シザーを設定します。
    //!
    //! @param[in]    x        シザーの左上原点 X 座標です。
    //! @param[in]    y        シザーの左上原点 Y 座標です。
    //! @param[in]    width    シザーの幅です。
    //! @param[in]    height   シザーの高さです。
    //---------------------------------------------------------------------------
    void SetScissor( f32 x, f32 y, f32 width, f32 height );

    //---------------------------------------------------------------------------
    //! @brief        画面クリアを行います。
    //---------------------------------------------------------------------------
    void ClearFrameBuffers();

    //---------------------------------------------------------------------------
    //! @brief        画面クリアを行います。
    //!
    //! @param[in]    flag     クリアする要素をビットマスクで指定します。
    //! @param[in]    color    この色でクリアします。 CLEAR_FLAG_COLOR が指定されていない場合は無視されます。
    //! @param[in]    depth    この値でZバッファを埋めます。 CLEAR_FLAG_DEPTH が指定されていない場合は無視されます。
    //! @param[in]    stencil  この値でステンシルバッファを埋めます。 CLEAR_FLAG_STENCIL が指定されていない場合は無視されます。
    //---------------------------------------------------------------------------
    void ClearFrameBuffersDetail( u8 flag, const nw::ut::Color4u8& color, f32 depth, u32 stencil );

    //---------------------------------------------------------------------------
    //! @brief        画面の切り替え処理を行います。
    //---------------------------------------------------------------------------
    void SwapBuffer();

    //---------------------------------------------------------------------------
    //! @brief        VSync 待ち処理を行います。
    //---------------------------------------------------------------------------
    void WaitForVBlank();

    //---------------------------------------------------------------------------
    //! @brief        フレーム開始時処理を行います。
    //!
    //!               ここでは、以下の処理を行います。
    //!               - DramMeter のフレーム開始処理
    //---------------------------------------------------------------------------
    void BeginFrame();

    //---------------------------------------------------------------------------
    //! @brief        フレーム終了時処理を行います。
    //!
    //!               ここでは、以下の処理を行います。
    //!               - デバッグ用文字描画のための DevTextWriter のバッファのクリア
    //!               - DramMeter のフレーム終了処理
    //---------------------------------------------------------------------------
    void EndFrame();


    //---------------------------------------------------------------------------
    //! @brief        Pad の更新を行います。
    //---------------------------------------------------------------------------
    void UpdatePad();

    //---------------------------------------------------------------------------
    //! @brief        Pad 管理クラスのインスタンスへのポインタを取得します。
    //!
    //! @return       Pad 管理クラスのインスタンスへのポインタを返します。
    //---------------------------------------------------------------------------
    nw::demo::Pad* GetPad() const { return m_Pad; }

    //---------------------------------------------------------------------------
    //! @brief        WPad 管理クラスのインスタンスへのポインタを取得します。
    //!
    //! @return       WPad 管理クラスのインスタンスへのポインタを返します。
    //---------------------------------------------------------------------------
    nw::demo::Pad* GetWPad() const { return m_WPad; }



    //---------------------------------------------------------------------------
    //! @brief        1 フレームを測定する計測メーターを取得します。
    //!
    //! @return       計測メーターを返します。
    //---------------------------------------------------------------------------
    nw::dev::CPUMeter& GetMeterFrame() { return m_MeterFrame; }

    //---------------------------------------------------------------------------
    //! @brief        Calc 計測メーターを取得します。
    //!
    //! @return       計測メーターを返します。
    //---------------------------------------------------------------------------
    nw::dev::CPUMeter& GetMeterCalc() { return m_MeterCalc; }

    //---------------------------------------------------------------------------
    //! @brief        Draw 計測メーターを取得します。
    //!
    //! @return       計測メーターを返します。
    //---------------------------------------------------------------------------
    nw::dev::CPUMeter& GetMeterDraw() { return m_MeterDraw; }

    //---------------------------------------------------------------------------
    //! @brief        GPU 計測メーターを取得します。
    //!
    //! @return       計測メーターを返します。
    //---------------------------------------------------------------------------
    nw::dev::GPUMeter& GetMeterGPU() { return m_MeterGPU; }

    //---------------------------------------------------------------------------
    //! @brief        MeterDrawer を取得します。
    //!
    //! @return       MeterDrawer を返します。
    //---------------------------------------------------------------------------
    nw::dev::MeterDrawer& GetMeterDrawer() { return m_MeterDrawer; }

    //---------------------------------------------------------------------------
    //! @brief        負荷メータを描画します。
    //---------------------------------------------------------------------------
    void DrawLoadMeters();

    //---------------------------------------------------------------------------
    //! @brief        負荷計測結果をログに出力します。
    //---------------------------------------------------------------------------
    void DumpLoadMeters() { m_MeterDrawer.Dump(); }

    //---------------------------------------------------------------------------
    //! @brief        デモ用 PerformanceMonitor 計測・表示機能取得します。
    //!
    //! @return       PerformanceMonitorCafe の参照を返します。
    //---------------------------------------------------------------------------
    PerformanceMonitorCafe& GetPerformanceMonitor() { return m_PerformanceMonitor; }

    //---------------------------------------------------------------------------
    //! @brief        パフォーマンスモニタによる計測を開始します。
    //!
    //!               BeginPMCPUMeasure と EndPMCPUMeasure で挟まれた区間を計測します。
    //!               複数区間を計測することも可能です。最大 32 区間計測できます。
    //!
    //! @param[in]    区間名です。結果表示時にラベルとして表示されます。
    //---------------------------------------------------------------------------
    void BeginPMCPUMeasure ( const char* name ) { m_PerformanceMonitor.BeginMeasure( name ); }

    //---------------------------------------------------------------------------
    //! @brief        パフォーマンスモニタによる計測を終了します。
    //---------------------------------------------------------------------------
    void EndPMCPUMeasure() { m_PerformanceMonitor.EndMeasure(); }

    //---------------------------------------------------------------------------
    //! @brief        計測したパフォーマンスモニタの結果を描画します。
    //---------------------------------------------------------------------------
    void DrawPMCPUResult() { m_PerformanceMonitor.DrawResult(); }


    //---------------------------------------------------------------------------
    //! @brief        デバッグ用文字列描画に用いることができる DevTextWriter を取得します。
    //!
    //! @return       DevTextWriter を返します。
    //---------------------------------------------------------------------------
    nw::dev::DevTextWriter* GetTextWriter() { return &m_DevTextWriter; }

    //---------------------------------------------------------------------------
    //! @brief        デバッグ用文字列描画のための DevTextWriter をクリアします。
    //---------------------------------------------------------------------------
    void ClearTextWriter();

    //---------------------------------------------------------------------------
    //! @brief        ポインタカーソルを描画します。
    //!
    //!               ポインタの座標系はウィンドウの左上が(-1,-1)、右下が(1,1)です。
    //!
    //! @param[in]    posX     ポインタカーソルの x 座標位置です。
    //! @param[in]    posY     ポインタカーソルの y 座標位置です。
    //! @param[in]    width    ウィンドウの幅です。
    //! @param[in]    height   ウィンドウの高さです。
    //---------------------------------------------------------------------------
    void DrawPointerCursor( f32 posX, f32 posY, f32 width, f32 height );

    //---------------------------------------------------------------------------
    //! @brief        フォントバイナリのポインタを取得します。
    //---------------------------------------------------------------------------
    const u8* GetFontBinary() const { return m_Arg.fontBinary; }

    //---------------------------------------------------------------------------
    //! @brief        フォントバイナリのポインタを取得します。
    //---------------------------------------------------------------------------
    u8*       GetFontBinary()  { return m_Arg.fontBinary; }

    //---------------------------------------------------------------------------
    //! @brief        フォントバイナリのサイズを取得します。
    //---------------------------------------------------------------------------
    u32       GetFontBinarySize() const { return m_Arg.fontBinarySize; }

    //---------------------------------------------------------------------------
    //! @brief        フォント用シェーダバイナリのポインタを取得します。
    //---------------------------------------------------------------------------
    const u8* GetFontShaderBinary() const { return m_Arg.fontShaderBinary; }

    //---------------------------------------------------------------------------
    //! @brief        フォント用シェーダバイナリのポインタを取得します。
    //---------------------------------------------------------------------------
    u8*       GetFontShaderBinary()  { return m_Arg.fontShaderBinary; }

    //---------------------------------------------------------------------------
    //! @brief        フォント用シェーダバイナリのサイズを取得します。
    //---------------------------------------------------------------------------
    u32       GetFontShaderBinarySize() const { return m_Arg.fontShaderBinarySize; }


protected:
    //! 黒画面に設定します。
    void SetBlack(bool is_black);

    CreateArg                 m_Arg;                        //!< 生成パラメータです。
    nw::ut::BitFlag32         m_Flags;                      //!< 設定フラグです。

    nw::ut::MemoryAllocator   m_VRAMAllocator;              //!< VRAM アロケーターです。
    FrameBuffer*              m_DefaultFrameBuffer;         //!< フレームバッファです。
    GX2ColorBuffer            m_ColorBuffer;                //!< カラーバッファです。
    GX2DepthBuffer            m_DepthBuffer;                //!< デプスバッファです。
    void*                     m_ScanBufferPtr;              //!< スキャンバッファへのポインタです。
    GX2ContextState*          m_ContextState;               //!< コンテキストステートへのポインタです。

    nw::demo::PadCafe*        m_Pad;                        //!< デバッグ用パッド管理クラスのインスタンスへのポインタです。
    nw::demo::WPadCafe*       m_WPad;                       //!< WPAD 管理クラスのインスタンスへのポインタです。

    nw::dev::CPUMeter         m_MeterFrame;                 //!< 1 フレームの処理負荷メーターです。
    nw::dev::CPUMeter         m_MeterCalc;                  //!< 計算処理負荷メーターです。
    nw::dev::CPUMeter         m_MeterDraw;                  //!< 描画処理負荷メーターです。
    nw::dev::GPUMeter         m_MeterGPU;                   //!< GPU 負荷メーターです。
    u64*                      m_GpuCounters;                //!< GPU 計測用カウンタの配列のポインタです。
    nw::dev::MeterDrawer      m_MeterDrawer;                //!< 負荷メーターを描画します。
    PerformanceMonitorCafe    m_PerformanceMonitor;         //!< デモ用 PerformanceMonitor 計測・表示です。
    nw::math::MTX44           m_TextWriterProjMtx;          //!< DevTextWriter 用プロジェクション行列です。
    nw::math::MTX34           m_TextWriterViewMtx;          //!< DevTextWriter 用ビュー行列です。
    nw::dev::DevTextWriter    m_DevTextWriter;              //!< デバッグ用文字描画のための DevTextWriter です。
    bool                      m_TextWriterInitialized;      //!< DevTextWriter が初期化されているかを表します。

    GX2Texture                m_PointerTextureGX2;          //!< ポインタカーソル用 GX2Texture です。
    nw::gfnd::TextureCafeGX2  m_PointerTexture;             //!< ポインタカーソル用テクスチャです。
    u8*                       m_PointerTextureImage;        //!< ポインタカーソル用画像データです。

    DisplayState              m_DisplayState;               //!< 画面表示を行ってよいかを判定するステート変数です。
};

} // namespace nw::demo
} // namespace nw

#endif // DEMO_FRAMEWORK_CAFE_H_
