﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Base.h>
#include <nn/util/util_BitPack.h>
#include <nn/nn_Common.h>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN  )
#include <nn/hid.h>
#include <nn/hid/hid_Mouse.h>
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
#include <types.h>
#endif

namespace nns {
namespace nac {

class Mouse
{
public:
    //! @brief マウスボタンのビット番号による定義です。
    enum PadIdx
    {
        IDX_LBUTTON = 0,
        IDX_RBUTTON,
        IDX_MBUTTON,
        PAD_IDX_MAX
    };

    //! @brief マウスボタンのマスクによる定義です。
    enum PadMask
    {
        MASK_LBUTTON    = 1 << IDX_LBUTTON,
        MASK_RBUTTON    = 1 << IDX_RBUTTON,
        MASK_MBUTTON    = 1 << IDX_MBUTTON
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    Mouse();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~Mouse();

    //---------------------------------------------------------------------------
    //! @brief      初期化
    //! @param[in]  hWnd    TBD
    //! @return     TBD
    //---------------------------------------------------------------------------
    bool Initialize( void* hWnd );

    //---------------------------------------------------------------------------
    //! @brief        解放処理
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief        毎フレームごとのアップデート
    //---------------------------------------------------------------------------
    void Update( bool isSetFocus = false );

#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかが押されているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       押されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsHold( uint32_t mask ) const
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN  )
        return ( mask & m_Press.storage ) != 0;
#else
        return false;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかがこのフレームに押されたボタンをマスク値で取得します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       押されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsTrig( uint32_t mask ) const
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN  )
        return ( mask & m_Trigger.storage ) != 0;
#else
        return false;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかがこのフレームで離されているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       離されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsRelease( uint32_t mask ) const
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN  )
        return ( mask & m_Release.storage ) != 0;
#else
        return false;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ホイール回転差分を取得します。
    //!
    //! @return       ホイール回転差分を返します。
    //---------------------------------------------------------------------------
    const int GetWheel() const
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN  )
        return m_WheelDelta / 40;
#else
        return 0;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ポインタの横方向位置を取得します。
    //!
    //! @return       ポインタの横方向位置を返します。
    //---------------------------------------------------------------------------
    const float GetPointerX() const
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN  )
        return static_cast< float >( m_Mouse.x );
#else
        return 0.0f;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ポインタの縦方向位置を取得します。
    //!
    //! @return       ポインタの縦方向位置を返します。
    //---------------------------------------------------------------------------
    const float GetPointerY() const
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN  )
        return static_cast< float >( m_Mouse.y );
#else
        return 0.0f;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        前フレームからのポインタの横方向移動量を取得します。
    //!
    //! @return       ポインタの横方向移動量を返します。
    //---------------------------------------------------------------------------
    const float GetPointerDiffX() const
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN  )
        return static_cast< float >( m_Mouse.x - m_MousePrev.x );
#else
        return 0.0f;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        前フレームからのポインタの縦方向移動量を取得します。
    //!
    //! @return       ポインタの縦方向移動量を返します。
    //---------------------------------------------------------------------------
    const float GetPointerDiffY() const
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN  )
        return static_cast< float >( m_Mouse.y - m_MousePrev.y );
#else
        return 0.0f;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        ポインタが画面内に入っているかを取得します。
    //!
    //! @return       ポインタが画面内に入っている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsPointerOn() const
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN  )
        return m_CursorOnClientArea;
#else
        return false;
#endif
    }

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN  )
    //---------------------------------------------------------------------------
    //! @brief        マウス値を取得するかどうかをアプリケーション側からの設定します。
    //!
    //! @param[in]    isEnabled   パッド時を取得するかしないかを設定します。
    //---------------------------------------------------------------------------
    void SetUseKeyboardForcus( bool isEnabled )
    {
        m_IsUseKeyboardForcus = isEnabled;
    }
#endif
#else
    // Cafe用にはマウスが無いので、コンパイル用にダミーで関数を定義します。（引数の型名が違うのに注意）
    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかが押されているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       押されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsHold(u32 mask) const
    {
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief        ホイール回転差分を取得します。
    //!
    //! @return       ホイール回転差分を返します。
    //---------------------------------------------------------------------------
    const s32 GetWheel() const
    {
        return 0;
    }

    //---------------------------------------------------------------------------
    //! @brief        前フレームからのポインタの横方向移動量を取得します。
    //!
    //! @return       ポインタの横方向移動量を返します。
    //---------------------------------------------------------------------------
    const f32 GetPointerDiffX() const
    {
        return 0;
    }

    //---------------------------------------------------------------------------
    //! @brief        前フレームからのポインタの縦方向移動量を取得します。
    //!
    //! @return       ポインタの縦方向移動量を返します。
    //---------------------------------------------------------------------------
    const f32 GetPointerDiffY() const
    {
        return 0;
    }

    //---------------------------------------------------------------------------
    //! @brief        ポインタが画面内に入っているかを取得します。
    //!
    //! @return       ポインタが画面内に入っている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsPointerOn() const
    {
        return false;
    }
#endif

#if defined ( NN_BUILD_TARGET_PLATFORM_OS_WIN  )
private:
    void*                       m_WindowHandle;           //!< ウィンドウハンドル
    nn::hid::MouseState         m_Mouse;                  //!< マウス管理クラスのインスタンス
    nn::hid::MouseState         m_MousePrev;              //!< 前回のマウス管理クラスのインスタンス
    nn::util::BitPack32         m_Press;                  //!< 入力値
    nn::util::BitPack32         m_PressPrev;              //!< 前回の入力値
    nn::util::BitPack32         m_Trigger;                //!< 押しトリガ入力
    nn::util::BitPack32         m_Release;                //!< 離しトリガ入力
    bool                        m_CursorOnClientArea;     //!< カーソルがこのアプリケーションのクライアント領域の上にあるかを表すフラグ。
    bool                        m_PrevCursorOnClientArea; //!< 前回カーソルがこのアプリケーションのクライアント領域の上にあるかを表すフラグ。
    bool                        m_MouseTriggeredNow;      //!< カーソルがこのアプリケーションのクライアント領域に入ったタイミングを表すフラグ。
    int64_t                     m_WheelSamplingNumber;    //!< ホイール回転差分用のマウスサンプリング番号
    int32_t                     m_WheelDelta;             //!< ホイール回転差分
    bool                        m_IsUseKeyboardForcus;     //!< todo
#endif
};

}
}
