﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file mm_MoviePlayerUtils.h
 * @brief Helper classes used by Player.
 *
 */
#include <vector>
#include <utility>
#include <string>
#include <locale>

#include <nn/vi.h>
#include <nn/os.h>
#include <nn/os/os_Tick.h>
#include <nn/fs.h>
#include <nn/nn_Result.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Log.h>

#include <movie/Common.h>
#include <movie/Status.h>
#include <movie/Utils.h>
#include <movie/Extractor.h>
#include <movie/Decoder.h>

/**
 * @brief
 * Mutex utility class which wraps nn::os::MutexType
 *
 * @details
 *
 * Purpose:
 *     The main purpose of Mutex class is to create a scoped lock which can be used in APIs without explicitly calling
 * InitializeMutex(), FinalizeMutex() LockMutex() and UnlockMutex() nn::os APIs.
 *
 * @Note:
 * Since scoped lock can be created easily using std::lock_guard, this utility class will be removed in future release.
 *
 */
class Mutex
{
    nn::os::MutexType m_Mutex;
public:
    /**
     * @brief
     * Helper class which creates mutex on stack and releases when the API exits
     */
    class ScopedLock
    {
        Mutex& m_Mutex;
    public:
        NN_IMPLICIT inline ScopedLock(Mutex& mutex) : m_Mutex(mutex) { mutex.Lock(); }
        inline ~ScopedLock() { m_Mutex.Unlock(); }
    };

    inline Mutex()       { nn::os::InitializeMutex(&m_Mutex, true, 0); }
    inline ~Mutex()      { nn::os::FinalizeMutex(&m_Mutex); }
    inline void Lock()   { nn::os::LockMutex(&m_Mutex); }
    inline void Unlock() { nn::os::UnlockMutex(&m_Mutex); }
};

/**
 * @brief
 * Clock utility class to get time related information.
 * @details
 *
 * Purpose:
 *     The main purpose of Clock class is to provide a simple API to get elapsed time in microseconds.
 *
 */
class Clock
{
public:
    /**
     * @brief
     * Get elapsed time in microseconds.
     *
     * @return   Returns elapsed time in microseconds.
     */
    static int64_t GetNowUs()
    {
        nn::TimeSpan ts = nn::os::ConvertToTimeSpan(nn::os::GetSystemTick());
        return ts.GetMicroSeconds();
    }
};

/**
 * @brief
 * StreamReader utility class for reading data from client reader interface.
 *
 */
class StreamReader : public movie::ClientStreamReader
{
public:
   /**
    * @brief
    * Open given filename
    *
    * @param[in] filename                       File to be open
    * @param[out] fileHandle                    File handle if open is successful
    *
    * @retval true                              File open is successful.
    * @retval false                             File open failed.
    *
    * @details
    *     Client will open the file and returns a handle to operate.
    */

    bool Open(const char *filename, void** fileHandle);

   /**
    * @brief
    * Read given size of data using provided fileHandle.
    *
    * @param[in]  fileHandle                    The File handle.
    * @param[in]  offset                        The File offset to read data.
    * @param[out] data                          Memory filled with data read.
    * @param[in]  size                          The required data size.
    *
    * @return The total number of bytes read
    *
    * @details
    *     Client will read requested amount of data at the requested offset. The API will return
    * the amount of data read.
    *
    */
    size_t ReadAt(void* fileHandle, int64_t offset, void *data, size_t size);

   /**
    * @brief
    * Close file.
    *
    * @param[in] fileHandle                       The valid file handle
    *
    * @retval true                                File close is successful.
    * @retval false                               File close is failed.
    *
    */
    bool Close(void* fileHandle);

   /**
    * @brief
    * Get file size.
    *
    * @param[in] fileHandle                       Valid file handle.
    * @param[out] size                            File size.
    *
    * @return none
    *
    */
    void GetSize(void* fileHandle, int64_t *size);
private:
    nn::fs::FileHandle *mFileHandle;
    int64_t fileLength;
};

/**
 * @brief
 * MediaClock utility class for AV sync.
 *
 */
class MediaClock
{
public:
    MediaClock();

   /**
    * @brief
    * SetMediaStartTime.
    *
    * @param[in]  mediaStartTimeUs                  Time in microseconds.
    *
    * @return none
    *
    * @details
    *     This API can be used to set media statrt time
    *
    */
    void SetMediaStartTime(int64_t mediaStartTimeUs);

   /**
    * @brief
    * UpdateAnchorTime.
    *
    * @param[in]  anchorTimeMediaUs                 Time in microseconds.
    * @param[in]  anchorTimeRealUs                  Time in microseconds.
    * @param[in]  maxTimeMediaUs                    Time in microseconds.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError
    *
    * @details
    *     This API can be used to update anchor time
    *
    */
    movie::Status UpdateAnchorTime(int64_t anchorTimeMediaUs, int64_t anchorTimeRealUs, int64_t maxTimeMediaUs = INT64_MAX);

   /**
    * @brief
    * ClearAnchorTime.
    *
    * @param[in]  anchorTimeMediaUs                 Time in microseconds.
    * @param[in]  anchorTimeRealUs                  Time in microseconds.
    * @param[in]  maxTimeMediaUs                    Time in microseconds.
    *
    * @return none
    *
    * @details
    *     This API can be used to clear anchor time. This API needs to be called after a seek or pause.
    */
    void ClearAnchorTime();

   /**
    * @brief
    * GetRealTimeForMediaTime.
    *
    * @param[in]  targetMediaUs                     Time in microseconds.
    * @param[out]  outRealUs                        Time in microseconds.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError
    *
    * @details
    *     This API can be used to get real time for a given mediaTime.
    *
    */
    movie::Status GetRealTimeForMediaTime(int64_t targetMediaUs, int64_t *outRealUs);
private:
    movie::Status GetMediaTime(int64_t realUs, int64_t *outMediaUs, bool allowPastMaxTime) const;

    nn::os::Mutex m_Lock;
    int64_t m_AnchorTimeMediaUs;
    int64_t m_AnchorTimeRealUs;
    int64_t m_MaxTimeMediaUs;
    int64_t m_MediaStartTimeUs;
};

/**
 * @brief
 * MemoryTracker utility class to track memory allocations.
 *
 */
class MemoryTracker
{
public:
    MemoryTracker();

   /**
    * @brief
    * API to register memory allocation information with the tracker.
    *
    * @param[in]   memory             Allocated memory address.
    * @param[in]   size               Allocated memory size in bytes.
    * @param[in]   threadName         Name of the thread from which memory allocation initiated.
    *
    * @return none
    *
    * @details
    * Use this API to register the memory allocations with tracker. Allocation information including
    * the allocation time is stored.
    */
    void Track(void *memory, size_t size, const char* threadName);

   /**
    * @brief
    * API to remove memory allocation information from the tracker.
    *
    * @param[in]   memory             Memory address to be removed from tracker
    *
    * @return none
    *
    * @details
    * Use this API to remove the allocation information from the tracker. When a tracked memory is freed
    * this API need to be called, otherwise it will be treated as a leak
    */
    void Untrack(void *memory);

   /**
    * @brief
    * API to print memory allocation statistics from the tracker.
    *
    * @return none
    *
    * @details
    * Use this API to print allocation statistics and memory leak.
    */
    void OutputUsage();

private:
    struct AllocationInfo
    {
        AllocationInfo(size_t size = 0, size_t id = -1, void *address = NULL, const char* threadName = nullptr);
        size_t allocSize;
        size_t allocId;
        void* address;
        nn::os::Tick time;
        char allocThreadName[ nn::os::ThreadNameLengthMax ];
    };

    nn::os::Mutex m_Lock;
    nn::os::Tick m_CreationTime;
    size_t m_Total;
    size_t m_MaxSize;
    size_t m_Id;
    static const size_t s_MaxAllocations = 100000;
    AllocationInfo m_Infos[ s_MaxAllocations ];
    bool m_InfoUsed[ s_MaxAllocations ];
};
