﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       TouchScreen のデバイスアセットに関する API の宣言
 */

#pragma once

#include <vector>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nns/hid/hid_DeviceAsset.h>
#include <nns/hid/hid_DeviceAssetId.h>

namespace nns { namespace hid {

/**
 * @brief   TouchScreen のデバイスアセットクラスです。
 */
class TouchScreenAsset : public DeviceAsset
{
    NN_DISALLOW_COPY(TouchScreenAsset);
    NN_DISALLOW_MOVE(TouchScreenAsset);

public:
    /**
     * @brief       TouchScreen の入力状態を表す型です。
     */
    typedef nn::hid::TouchScreenState<nn::hid::TouchStateCountMax
                                      > TouchScreenState;

    /**
     * @brief       TouchScreenAsset のコンストラクタです。
     *
     * @pre
     *              - 事前に 1 回以上 Initialize() が呼び出されている
     *
     * @param[in]   pManager                    TouchScreen のデバイスアセットの管理者です。
     */
    explicit TouchScreenAsset(ControllerManager* pManager) NN_NOEXCEPT;

    /**
     * @brief       TouchScreenAsset のデストラクタです。
     */
    virtual ~TouchScreenAsset() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       TouchScreen を初期化します。
     */
    static void Initialize() NN_NOEXCEPT;

    /**
     * @brief       TouchScreen のデバイスアセット識別子を返します。
     *
     * @return      TouchScreen のデバイスアセット識別子です。
     */
    virtual DeviceAssetId GetDeviceAssetId() NN_NOEXCEPT NN_OVERRIDE
    {
        return DeviceAssetId_TouchScreen;
    }

    /**
     * @brief       TouchScreen のデバイスアセットを更新します。
     *
     * @details     更新はフレーム毎に行う必要があります。
     */
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       TouchScreen の入力状態を返します。
     *
     * @return      TouchScreen の入力状態です。
     */
    const std::vector<TouchScreenState
                      >& GetTouchScreenStates() const NN_NOEXCEPT
    {
        return m_States;
    }

private:
    int64_t m_SamplingNumber;               //!< TouchScreen のサンプリング番号です。

    std::vector<TouchScreenState> m_States; //!< TouchScreen の入力状態です。

    TouchScreenState m_TemporaryStates[
        nn::hid::TouchScreenStateCountMax]; //!< 入力状態読み出し用のバッファです。
};

}} // namespace nns::hid
