﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ゲームパッドのデバイスアセットに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_Npad.h>
#include <nns/hid/hid_DeviceAsset.h>
#include <nns/hid/hid_DeviceAssetId.h>

namespace nns { namespace hid {

/**
 * @brief   ゲームパッドのデバイスアセットクラスです。
 */
class GamePadAsset : public DeviceAsset
{
    NN_DISALLOW_COPY(GamePadAsset);
    NN_DISALLOW_MOVE(GamePadAsset);

public:
    //!< サポートするプレイヤー番号の数です
    const int PlayerNumberCountMax = 4;

    /**
     * @brief       ゲームパッドの入力状態の属性定義です。
     */
    typedef nn::hid::NpadAttribute GamePadAttribute;

    /**
     * @brief       ゲームパッドのデジタルボタンのデジタルボタン定義です。
     */
    typedef nn::hid::NpadButton GamePadButton;

    /**
     * @brief       ゲームパッドの入力状態を表す型です。
     */
    typedef nn::hid::NpadFullKeyState GamePadState;

    /**
     * @brief       GamePadAsset のコンストラクタです。
     *
     * @pre
     *              - 事前に 1 回以上 Initialize() が呼び出されている
     *
     * @param[in]   pManager                    ゲームパッドのデバイスアセットの管理者です。
     */
    explicit GamePadAsset(ControllerManager* pManager) NN_NOEXCEPT;

    /**
     * @brief       GamePadAsset のデストラクタです。
     */
    virtual ~GamePadAsset() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       ゲームパッドを初期化します。
     */
    static void Initialize() NN_NOEXCEPT;

    /**
     * @brief       ゲームパッドのデバイスアセット識別子を返します。
     *
     * @return      ゲームパッドのデバイスアセット識別子です。
     */
    virtual DeviceAssetId GetDeviceAssetId() NN_NOEXCEPT NN_OVERRIDE
    {
        return DeviceAssetId_GamePad;
    }

    /**
     * @brief       ゲームパッドのデバイスアセットを更新します。
     *
     * @details     更新はフレーム毎に行う必要があります。
     */
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       ゲームパッドの入力状態を取得します。
     *
     * @details     指定のプレイヤー番号と対応するゲームパッドから入力状態を取得します
     *
     * @param[out]  pOutValue                   入力状態を読み出すバッファです。
     * @param[in]   playerNumber                ゲームパッドのプレイヤー番号です。
     *
     * @return      ゲームパッドの入力状態の取得に成功したか否かを表す値です。
     */
    bool GetGamePadState(GamePadState* pOutValue, int playerNumber
                         ) const NN_NOEXCEPT;

private:
    GamePadState* m_States; //!< 入力状態読み出し用のバッファです。
    int* m_StatesTable;     //!< PlayerNumber に対応する入力状態のバッファの index を示したテーブルです
};

}} // namespace nns::hid
