﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       コントローラ管理に関する API の宣言
 */

#pragma once

#include <vector>
#include <nn/nn_Macro.h>
#include <nns/hid/hid_Controller.h>
#include <nns/hid/hid_ControllerId.h>
#include <nns/hid/hid_DeviceAsset.h>
#include <nns/hid/hid_DeviceAssetId.h>

namespace nns { namespace hid {

/**
 * @brief   コントローラ管理クラスです。
 */
class ControllerManager
{
    NN_DISALLOW_COPY(ControllerManager);
    NN_DISALLOW_MOVE(ControllerManager);

public:
    /**
     * @brief       ControllerManager のコンストラクタです。
     */
    ControllerManager() NN_NOEXCEPT;

    /**
     * @brief       ControllerManager のデストラクタです。
     */
    ~ControllerManager() NN_NOEXCEPT;

    /**
     * @brief       コントローラの状態を更新します。
     *
     * @details     更新はフレーム毎に行う必要があります。
     */
    void Update() NN_NOEXCEPT;

    /**
     * @brief       デバイスアセットを返します。
     *
     * @details     index 番目のデバイスアセットを返します。
     *              デバイスアセットが見つからなかった場合は NULL を返します。
     *
     * @param[in]   index                       添字の指定です。
     *
     * @return      デバイスアセットです。
     */
    DeviceAsset *GetDeviceAsset(int index) const NN_NOEXCEPT;

    /**
     * @brief       デバイスアセットを返します。
     *
     * @details     指定のデバイスアセット識別子を持つデバイスアセットを探索し、
     *              index 番目に見つかったものを返します。
     *              条件を満たすデバイスアセットが見つからなかった場合は NULL を返します。
     *
     * @param[in]   id                          デバイスアセット識別子です。
     * @param[in]   index                       添字の指定です。
     *
     * @return      デバイスアセットです。
     */
    DeviceAsset *GetDeviceAsset(DeviceAssetId id, int index) const NN_NOEXCEPT;

    /**
     * @brief       コントローラを返します。
     *
     * @details     index 番目のコントローラを返します。
     *              コントローラが見つからなかった場合は NULL を返します。
     *
     * @param[in]   index                       添字の指定です。
     *
     * @return      コントローラです。
     */
    Controller *GetController(int index) const NN_NOEXCEPT;

    /**
     * @brief       コントローラを返します。
     *
     * @details     指定のコントローラ識別子を持つコントローラを探索し、
     *              index 番目に見つかったものを返します。
     *              条件を満たすコントローラが見つからなかった場合は NULL を返します。
     *
     * @param[in]   id                          コントローラ識別子です。
     * @param[in]   index                       添字の指定です。
     *
     * @return      コントローラです。
     */
    Controller *GetController(ControllerId id, int index) const NN_NOEXCEPT;

    /**
     * @brief       デバイスアセットのリストを返します。
     *
     * @details     独自にデバイスアセットを追加したい場合は、
     *              このメソッドを用いてください。
     *
     * @return      デバイスアセットのリストです。
     */
    std::vector<DeviceAsset*>& GetDeviceAssetList() NN_NOEXCEPT
    {
         return m_DeviceAssets;
    }

    /**
     * @brief       コントローラのリストを返します。
     *
     * @details     独自にコントローラを追加したい場合は、
     *              このメソッドを用いてください。
     *
     * @return      コントローラのリストです。
     */
    std::vector<Controller*>& GetControllerList() NN_NOEXCEPT
    {
         return m_Controllers;
    }

private:
    /**
     * @brief   コントローラ管理クラスが管理対象とするデバイスアセットです。
     */
    std::vector<DeviceAsset*> m_DeviceAssets;

    /**
     * @brief   コントローラ管理クラスが管理対象とするコントローラです。
     */
    std::vector<Controller*> m_Controllers;
};

namespace util {

/**
 * @brief       コントローラ管理者を標準構成に設定します。
 *
 * @details     下記の順でデバイスアセットの初期化と追加を行います。
 *              - ゲームパッド
 *              - DebugPad
 *              - Keyboard
 *              - TouchScreen
 *              また、下記の順でコントローラの追加を行います。
 *              - ゲームパッド * 4
 *              - DebugPad
 *              - Keyboard
 *              - TouchScreen
 *
 * @param[in]   pManager                    コントローラ管理者です。
 */
void SetControllerManagerWithDefault(ControllerManager* pManager) NN_NOEXCEPT;

} // namespace util

}} // namespace nns::hid
