﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       コントローラアドオンに関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nns/hid/hid_ControllerAddonId.h>

namespace nns { namespace hid {

// 依存クラスの前方宣言です。
class Controller;

/**
 * @brief   コントローラアドオンの基底クラスです。
 *
 */
class ControllerAddon
{
    NN_DISALLOW_COPY(ControllerAddon);
    NN_DISALLOW_MOVE(ControllerAddon);

public:
    /**
     * @brief       ControllerAddon のコンストラクタです。
     *
     * @param[in]   pController                    コントローラです。
     *
     * @pre
     *              - pController != NULL
     */
    explicit ControllerAddon(Controller* pController) NN_NOEXCEPT
        : m_pController(pController)
    {
    }

    /**
     * @brief       ControllerAddon のデストラクタです。
     */
    virtual ~ControllerAddon() NN_NOEXCEPT{}

    /**
     * @brief       コントローラアドオンの状態を更新します。
     *
     * @details     更新はフレーム毎に行う必要があります。
     */
    virtual void Update() NN_NOEXCEPT = 0;

    /**
     * @brief       コントローラアドオン識別子を返します。
     *
     * @return      コントローラアドオン識別子です。
     */
    virtual ControllerAddonId GetControllerAddonId() NN_NOEXCEPT = 0;

    /**
     * @brief       コントローラが接続状態にあるか否かを返します。
     *
     * @details     接続状態をコントローラが持たない場合は常に true を返します。
     *
     * @return      コントローラが接続状態にあるか否かを表す値です。
     */
    virtual bool IsConnected() const NN_NOEXCEPT
    {
        return true;
    }

    /**
     * @brief       コントローラを返します。
     *
     * @return      コントローラです。
     */
    Controller* GetController() const NN_NOEXCEPT
    {
        return m_pController;
    }

private:
    Controller* m_pController;  //!< コントローラです。
};

}} // namespace nns::hid
