﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   グラフィックドライバに関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/os/os_SystemEvent.h>


namespace nns { namespace gfx0 {

//-----------------------------------------------------------------------------
//! @name グラフィックドライバ関連の API
//! @{

//-----------------------------------------------------------------------------
/**
 * @brief   グラフィックドライバのサービスを初期化して使用可能にします。
 *
 * @details
 *  グラフィックドライバのサービスを初期化して使用可能にします。
 *  グラフィックドライバの他の全ての API よりも先に発行しておく必要があります。
 *
 */
void InitializeGfx0Driver() NN_NOEXCEPT;

//-----------------------------------------------------------------------------
/**
 * @brief   グラフィックドライバのサービスの使用を終了します。
 *
 * @details
 *  グラフィックドライバのサービスの使用を終了します。
 *
 */
void FinalizeGfx0Driver() NN_NOEXCEPT;

//-----------------------------------------------------------------------------
/**
 * @brief   スケルトン用のダミー関数です。
 *
 * @param[in] x     適当な整数値
 * @param[in] y     適当な整数値
 *
 * @pre
 *  - nn::gfx0::InitializeGfx0Driver() を発行済み
 *
 * @details
 *  スケルトン用のダミー関数です。本スケルトンでは x と y の加算結果を返します。
 *
 */
int32_t DummyFuncGfx0(int32_t x, int32_t y) NN_NOEXCEPT;



//-----------------------------------------------------------------------------
/**
 * @brief   作成済みの nn::os::TransferMemory をサーバへ提供します。
 *
 * @param[in] transferMemory nn::os::TransferMemoryType オブジェクトへのポインタ
 * @param[in] size           転送するメモリのサイズ
 *
 * @pre
 *  - nn::gfx0::InitializeGfx0Driver() を発行済み
 *  - transferMemory オブジェクトが生成済み状態
 *  - transferMemory オブジェクトのメモリサイズが size に等しい
 *
 * @post
 *  - transferMemory オブジェクトがデタッチ状態
 *
 * @details
 *  自プロセスで作成した TransferMemory をサーバプロセスへ提供します。
 *
 *  以後、自プロセスでは、この trasnferMemory のメモリ領域を使用できません。
 *  このメモリ領域が再度使用可能になるのは、サーバが対象 transferMemory を
 *  アンマップした時であり、詳細はサーバの振舞いによって決定されます。
 *
 *  指定された transferMemory オブジェクトはデタッチ状態になるため、これ以降、
 *  適当なタイミングで DestroyTransferMemory() する必要があります。
 *
 */
void AttachTransferMemory(nn::os::TransferMemoryType* transferMemory, size_t size) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   特定のメモリ領域をデバイスアドレス空間用のメモリとしてサーバへ提供します。
 *
 * @param[in] address   提供するメモリの先頭アドレス
 * @param[in] size      提供するメモリのサイズ
 *
 * @pre
 *  - nn::gfx0::InitializeGfx0Driver() を発行済み
 *  - address および size が 4KB アライメントされている
 *
 * @details
 *  address と size で表されるメモリ領域の一部をサーバプロセスへ提供します。
 *
 *  サーバプロセスでは、受け取ったメモリをデバイスアドレス空間として
 *  アタッチおよびマップし、メモリを使用した後にアンマップおよびデタッチして
 *  再度提供元へメモリを返却します。
 *
 *  そのため、本関数の呼出し前後によって、指定されたメモリ領域の内容が
 *  サーバ側の処理によって書き換えられている可能性があります。
 *
 */
void AttachDeviceMemory(uintptr_t address, size_t size) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   サーバが作成した nn::os::SharedMemory を取得します。
 *
 * @param[in] sharedMemory  未初期化の nn::os::SharedMemoryType オブジェクトへのポインタ
 *
 * @pre
 *  - nn::gfx0::InitializeGfx0Driver() を発行済み
 *  - sharedMemory オブジェクトが未初期化状態
 *
 * @post
 *  - sharedMemory オブジェクトが初期化状態
 *
 * @details
 *  サーバプロセス上で作成された SharedMemory を取得し、自プロセスからも
 *  アクセスできるように sharedMemory オブジェクトを初期化します。
 *
 *  以後、自プロセスの都合に合わせて、sharedMemory オブジェクトを使って
 *  共有メモリをマップしたり、オブジェクトを破棄したりすることが出来ます。
 *
 */
void QuerySharedMemory(nn::os::SharedMemoryType* sharedMemory) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   サーバが作成した nn::os::SystemEvent を取得します。
 *
 * @param[in] systemEvent  未初期化の nn::os::SystemEventType オブジェクトへのポインタ
 *
 * @pre
 *  - nn::gfx0::InitializeGfx0Driver() を発行済み
 *  - systemEvent オブジェクトが未初期化状態
 *
 * @post
 *  - systemEvent オブジェクトが初期化状態
 *
 * @details
 *  サーバプロセス上で作成された SystemEvent の ReadableHandle を取得し、
 *  自プロセスから待機およびクリアできるように systemEvent オブジェクトを
 *  初期化します。
 *
 *  この systemEvent は gfx0server からの処理完了通知に利用されます。
 *  また、対象の systemEvent は nn::os::EventClearMode_AutoClear として
 *  初期化されます。
 *
 */
void QuerySystemEvent(nn::os::SystemEventType* systemEvent) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   デモプログラムの内部状態を１ステップ分進めます。
 *
 * @pre
 *  - nn::gfx0::InitializeGfx0Driver() を発行済み
 *
 * @details
 *  サーバプロセス内で動作しているデモプログラムの状態を１世代分進めます。
 *  最新のデモプログラムの状態は、 QuerySharedMemory() で取得できる
 *  共有メモリ領域にログ出力可能な文字列として生成されてます。
 *
 *  また、本 API のサーバ側処理が完了する度に QuerySystemEvent() で取得できる
 *  システムイベントがシグナル化されます。これにより、他のプロセスでも
 *  デモプログラムの更新タイミングを検知することができます。
 *
 */
void AdvanceDemo() NN_NOEXCEPT;


//! @}
//-----------------------------------------------------------------------------

}} // namespace nns::gfx0

