﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Allocator.h>
#include <nn/gfx.h>
#include <nn/nn_Assert.h>

namespace nns
{
namespace gfx
{

class ColorBuffer;
class DepthStencilBuffer;

//------------------------------------------------------------------------------
//! @brief レンダーターゲットを管理します。
//------------------------------------------------------------------------------
class FrameBuffer
{
public:
    class InfoType;

    //! @brief クリアカラーの要素の種類です。
    enum ClearColorComponent
    {
        ClearColorComponent_Red = 0,
        ClearColorComponent_Green,
        ClearColorComponent_Blue,
        ClearColorComponent_Alpha,
        ClearColorComponent_Max
    };

    //! @brief  フレームバッファの最大サイズです。
    enum MaxSize
    {
        Max_Width = 8192,
        Max_Height = 8192,
    };

    //! @brief 必要なMemoryPoolのサイズを返します。
    //! @param[in]    pGfxDevice  gfxデバイスです。
    //! @param[in]    info        フレームバッファを初期化する際のパラメータです。
    static size_t GetRequiredMemoryPoolSize(nn::gfx::Device* pGfxDevice, const InfoType& info);

    //! @brief MemoryPoolのアライメントを返します。
    //! @param[in]    pGfxDevice  gfxデバイスです。
    //! @param[in]    info        フレームバッファを初期化する際のパラメータです。
    //! @return                   MemoryPoolのアライメント
    static size_t GetMemoryPoolAlignment(nn::gfx::Device* pGfxDevice, const InfoType& info);

    //! @brief  コンストラクタです。
    FrameBuffer() NN_NOEXCEPT;

    //! @brief  初期化処理です。
    //!
    //! @details
    //! 内部でレンダーターゲットを作成するため nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible が設定されているメモリプールを指定してください。
    //! nns::gfx::GraphicsFramework を使用している場合は nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget のメモリプールを使用することができます。
    //!
    //! @param[in]  pDevice gfx デバイスです。
    //! @param[in]  pInfoType   初期化時のパラメータ情報です。
    //! @param[in]  pMemoryPool 内部のレンダーターゲットを作成する際に使用するメモリプールです。
    //! @param[in]  memoryPoolOffset    メモリプール内のオフセットです。
    //! @param[in]  pAllocateFunction   メモリ確保関数です。
    //! @param[in]  pFreeFunction       メモリ解放関数です。
    //! @param[in]  pAllocateFunctionUserData   メモリ確保・解放関数に渡すユーザーデータへのポインタです。
    void Initialize(
            nn::gfx::Device* pDevice,
            const InfoType* pInfoType,
            nn::gfx::MemoryPool* pMemoryPool,
            ptrdiff_t memoryPoolOffset,
            nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
            nn::FreeFunctionWithUserData pFreeFunction,
            void* pAllocateFunctionUserData
        ) NN_NOEXCEPT;

    //! @brief  終了処理です。
    //!
    //! @param[in]  pDevice gfx デバイスです。
    //! @param[in]  pAllocateFunction   メモリ確保関数です。
    //! @param[in]  pFreeFunction       メモリ解放関数です。
    //! @param[in]  pAllocateFunctionUserData   メモリ確保・解放関数に渡すユーザーデータへのポインタです。
    void Finalize(
            nn::gfx::Device* pDevice,
            nn::AlignedAllocateFunctionWithUserData pAllocateFunction,
            nn::FreeFunctionWithUserData pFreeFunction,
            void* pAllocateFunctionUserData
        ) NN_NOEXCEPT;

    //! @brief  初期化時に渡されたメモリプール内のオフセットを取得します。
    //!
    //! @return 初期化時に渡されたメモリプール内のオフセット指定
    ptrdiff_t GetMemoryPoolOffset() const NN_NOEXCEPT;

    //! @brief  フレームバッファの幅を取得します。
    //!
    //! @return フレームバッファの幅
    int GetWidth() const NN_NOEXCEPT
    {
        return m_Width;
    }

    //! @brief  フレームバッファの高さを取得します。
    //!
    //! @return フレームバッファの高さ
    int GetHeight() const NN_NOEXCEPT
    {
        return m_Height;
    }

    //! @brief  クリアカラーを設定します。
    //!
    //! @param[in]  red クリアカラーの赤成分
    //! @param[in]  green クリアカラーの緑成分
    //! @param[in]  blue クリアカラーの青成分
    //! @param[in]  alpha クリアカラーのアルファ成分
    void SetClearColor(float red, float green, float blue, float alpha = 1.0f) NN_NOEXCEPT
    {
        m_ClearColor[ClearColorComponent_Red] = red;
        m_ClearColor[ClearColorComponent_Green] = green;
        m_ClearColor[ClearColorComponent_Blue] = blue;
        m_ClearColor[ClearColorComponent_Alpha] = alpha;
    }

    //! @brief  クリアカラーを取得します。
    //!
    //! @param[in]  componentType   取得するクリアカラーの成分を指定します。
    //!
    //! @return クリアカラーの 1 要素の値
    float GetClearColor(ClearColorComponent componentType) const
    {
        NN_SDK_ASSERT(componentType < ClearColorComponent_Max);

        return m_ClearColor[componentType];
    }

    //! @brief  カラーバッファを取得します。
    //!
    //! @return カラーバッファ
    nns::gfx::ColorBuffer* GetColorBuffer() NN_NOEXCEPT
    {
        return m_pColorBuffer;
    }

    //! @brief  カラーバッファを取得します。
    //!
    //! @return カラーバッファ
    const nns::gfx::ColorBuffer* GetColorBuffer() const NN_NOEXCEPT
    {
        return m_pColorBuffer;
    }

    //! @brief  デプスバッファを取得します。
    //!
    //! @return デプスバッファ
    nns::gfx::DepthStencilBuffer* GetDepthBuffer() NN_NOEXCEPT
    {
        return m_pDepthBuffer;
    }

    //! @brief  デプスバッファを取得します。
    //!
    //! @return デプスバッファ
    const nns::gfx::DepthStencilBuffer* GetDepthBuffer() const NN_NOEXCEPT
    {
        return m_pDepthBuffer;
    }

    //! @brief  ビューポートシザーステートを取得します。
    //!
    //! @retrun ビューポートシザーステート
    const nn::gfx::ViewportScissorState* GetViewportScissorState() const NN_NOEXCEPT
    {
        return &m_ViewportScissor;
    }

    //! @brief  ビューポートシザーステートを初期化します。
    //!
    //! @param[in]  pDevice gfx デバイスです。
    //! @param[in]  x       ビューポートシザーの x 座標
    //! @param[in]  y       ビューポートシザーの y 座標
    //! @param[in]  width   ビューポートシザーの幅
    //! @param[in]  height  ビューポートシザーの高さ
    void SetupViewport(nn::gfx::Device* pDevice, int x, int y, int width, int height) NN_NOEXCEPT;

private:
    int m_Width;
    int m_Height;
    float m_ClearColor[ClearColorComponent_Max];
    nns::gfx::ColorBuffer* m_pColorBuffer;
    nns::gfx::DepthStencilBuffer* m_pDepthBuffer;
    nn::gfx::ViewportScissorState m_ViewportScissor;

    NN_DISALLOW_COPY(FrameBuffer);
};

//------------------------------------------------------------------------------
//! @brief FrameBuffer の初期化パラメータです。
//------------------------------------------------------------------------------
class FrameBuffer::InfoType
{
public:
    //! @brief  コンストラクタです。
    //!
    //! @param[in]  width   フレームバッファの幅です。
    //! @param[in]  height  フレームバッファの高さです。
    InfoType(int width, int height) NN_NOEXCEPT
        : m_Width(width)
        , m_Height(height)
    {
        SetDefault();
    }

    //! @brief  デフォルト値を設定します。
    void SetDefault() NN_NOEXCEPT
    {
        m_ColorBufferFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;
        m_DepthBufferFormat = nn::gfx::ImageFormat_D32_Float;
        m_UseDepthBuffer = true;
    }

    //! @brief  フレームバッファの幅を取得します。
    //!
    //! @return フレームバッファの幅。
    int GetWidth() const NN_NOEXCEPT
    {
        return m_Width;
    }

    //! @brief  フレームバッファの高さを取得します。
    //!
    //! @return フレームバッファの高さ。
    int GetHeight() const NN_NOEXCEPT
    {
        return m_Height;
    }

    //! @brief  カラーバッファのフォーマットを取得します。
    //!
    //! @return カラーバッファのフォーマット。
    nn::gfx::ImageFormat GetColorBufferFormat() const NN_NOEXCEPT
    {
        return m_ColorBufferFormat;
    }

    //! @brief  デプスステンシルバッファのフォーマットを取得します。
    //!
    //! @return デプスステンシルバッファのフォーマット。
    nn::gfx::ImageFormat GetDepthBufferFormat() const NN_NOEXCEPT
    {
        return m_DepthBufferFormat;
    }

    //! @brief  デプスステンシルバッファを使用するかどうかのフラグを取得します。
    //!
    //! @return デプスステンシルバッファを使用するかどうか。
    bool IsDepthBufferUsed() const NN_NOEXCEPT
    {
        return m_UseDepthBuffer;
    }

    //! @brief  フレームバッファの幅を設定します。
    //!
    //! @param[in]  フレームバッファの幅。
    void SetWidth(int width) NN_NOEXCEPT
    {
        m_Width = width;
    }

    //! @brief  フレームバッファの高さを設定します。
    //!
    //! @param[in]  フレームバッファの高さ。
    void SetHeight(int height) NN_NOEXCEPT
    {
        m_Height = height;
    }

    //! @brief  カラーバッファのフォーマットを設定します。
    //!
    //! @param[in]  カラーバッファのフォーマット。
    void SetColorBufferFormat(nn::gfx::ImageFormat format) NN_NOEXCEPT
    {
        m_ColorBufferFormat = format;
    }

    //! @brief  デプスステンシルバッファのフォーマットを設定します。
    //!
    //! @param[in]  デプスステンシルバッファのフォーマット。
    void SetDepthBufferFormat(nn::gfx::ImageFormat format) NN_NOEXCEPT
    {
        m_DepthBufferFormat = format;

    }

    //! @brief  デプスステンシルバッファを有効に設定します。
    void SetDepthBufferEnabled() NN_NOEXCEPT
    {
        m_UseDepthBuffer = true;
    }

    //! @brief  デプスステンシルバッファを無効に設定します。
    void SetDepthBufferDisabled() NN_NOEXCEPT
    {
        m_UseDepthBuffer = false;
    }

private:
    int m_Width;
    int m_Height;
    nn::gfx::ImageFormat m_ColorBufferFormat;
    nn::gfx::ImageFormat m_DepthBufferFormat;
    bool m_UseDepthBuffer;
};

} // namespace gfx
} // namespace nns
