﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d.h>
#include <nns/g3d/g3d_System.h>
#include <nns/g3d/g3d_RenderModel.h>

namespace nns { namespace g3d {

class RenderUnitObj;

//! @brief シェイプの描画前に呼び出されるコールバック関数です。
//!
//! @details
//! true の場合は描画し、false の場合には描画を行いません。
//!
typedef bool (*DrawCullingCallback)(const RenderUnitObj* pRenderUnitObj, int passIndex);


//--------------------------------------------------------------------------------------------------
//! @brief シェイプの描画クラスです。
class RenderUnitObj
{
    NN_DISALLOW_COPY(RenderUnitObj);
public:
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    RenderUnitObj() NN_NOEXCEPT
        : m_pModelObj(NULL)
        , m_pShapeObj(NULL)
        , m_pRenderShapePtr(NULL)
        , m_pUniformBlockPtr(NULL)
        , m_pShaderStorageBlockPtr(NULL)
        , m_pViewVolumePtr(NULL)
        , m_pCalculateLodLevelFunctorPtr(NULL)
        , m_DrawLodLevel(InvalidLod)
        , m_DrawSingleSubmeshIndex(InvalidIndex)
        , m_IsInitialized(false)
        , m_ShapeIndex(InvalidIndex)
        , m_PassCount(0)
    {
    }

    //! @brief RenderUnitObj を初期化します。
    //!
    //! @param[in] pModelObj シェイプが属するモデルへのポインター。
    //! @param[in] pRenderModel nns::g3d::RenderModel へのポインター。
    //! @param[in] shapeIndex 初期化を行うシェイプのインデックス。
    //!
    //! @return 初期化が成功した場合は true、失敗した場合は false を返します。pRenderShape が初期化済みでない場合、失敗します。
    //!
    //! @pre
    //! - RenderUnitObj が未初期化である。
    //! - pModelObj が NULL でない。
    //! - pRenderModel が NULL でない。
    //! - pRenderModel が初期化されている。
    //!
    //! @post
    //! - RenderUnitObj が初期化されている。
    //!
    bool Initialize(nn::g3d::ModelObj* pModelObj, RenderModel* pRenderModel, int shapeIndex) NN_NOEXCEPT;

    //! @brief RenderUnitObj を破棄します。
    //!
    //! @pre
    //! - RenderUnitObj が初期化されている。
    //!
    //! @post
    //! - RenderUnitObj が未初期化である。
    //!
    void Finalize() NN_NOEXCEPT;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief nn::g3d::ShapeObj を取得します。
    //!
    //! @return nn::g3d::ShapeObj へのポインター。
    //!
    const nn::g3d::ShapeObj* GetShapeObj() const NN_NOEXCEPT
    {
        return m_pShapeObj;
    }

    //! @brief nn::g3d::ModelObj を取得します。
    //!
    //! @return nn::g3d::ModelObj へのポインター。
    //!
    const nn::g3d::ModelObj* GetModelObj() const NN_NOEXCEPT
    {
        return m_pModelObj;
    }

    //! @brief ダミーサンプラーを取得します。
    //!
    //! @return nns::g3d::Sampler へのポインター。
    //!
    const Sampler* GetDummySampler() const NN_NOEXCEPT
    {
        return &m_DummySampler;
    }

    //! @brief ユニフォームブロックを設定します。
    //!
    //! @param[in] id ユニフォームブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] bufferCount pBufferPtr の要素数。
    //!
    //! @details
    //! 複数バッファーを設定することにより、GPU 処理中に GPU が参照していないバッファーを更新出来るようになります。
    //! count には pBufferPtr 配列の要素数を指定します。
    //!
    //! この関数は、全ての描画パスに対してユニフォームブロックを設定することを試みます。
    //!
    void BindUniformBlock(const char* id, const nn::gfx::Buffer** pBufferPtr, int bufferCount) NN_NOEXCEPT;

    //! @brief ユニフォームブロックを設定します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id ユニフォームブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] bufferCount pBufferPtr の要素数。
    //!
    //! @details
    //! 複数バッファーを設定することにより、GPU 処理中に GPU が参照していないバッファーを更新出来るようになります。
    //! count には pBufferPtr 配列の要素数を指定します。
    //!
    void BindUniformBlock(int passIndex, const char* id, const nn::gfx::Buffer** pBufferPtr, int bufferCount) NN_NOEXCEPT;

    //! @brief ユニフォームブロックを取得します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id ユニフォームブロックの識別子
    //!
    //! @details
    //! 対象のユニフォームブロックが見つからなかった場合には、NULL を返します。
    //!
    nns::g3d::UniformBlock* FindUniformBlock(int passIndex, const char* id) NN_NOEXCEPT;

    //! @brief ユニフォームブロックの設定を解除します。
    //!
    //! @param[in] id ユニフォームブロックの識別子。
    //!
    //! @details
    //! この関数は、全ての描画パスに対してユニフォームブロックの設定を解除することを試みます。
    //!
    void UnbindUniformBlock(const char* id) NN_NOEXCEPT;

    //! @brief ユニフォームブロックの設定を解除します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id ユニフォームブロックの識別子。
    //!
    void UnbindUniformBlock(int passIndex, const char* id) NN_NOEXCEPT;

    //! @brief シェーダーストレージブロックを設定します。
    //!
    //! @param[in] id シェーダーストレージブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] bufferCount pBufferPtr の要素数。
    //!
    //! @details
    //! シェーダーストレージブロック内に可変長配列が定義されていない場合にのみ使用してください。
    //! 可変長配列が定義されている場合は、サイズを指定できる BindShaderStorageBlock() を使用してください。
    //! 複数バッファーを設定することにより、GPU 処理中に GPU が参照していないバッファーを更新出来るようになります。
    //! count には pBufferPtr 配列の要素数を指定します。
    //!
    //! この関数は、全ての描画パスに対してシェーダーストレージブロックを設定することを試みます。
    //!
    void BindShaderStorageBlock(const char* id, const nn::gfx::Buffer** pBufferPtr, int bufferCount) NN_NOEXCEPT;

    //! @brief シェーダーストレージブロックを設定します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id シェーダーストレージブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] bufferCount pBufferPtr の要素数。
    //!
    //! @details
    //! シェーダーストレージブロック内に可変長配列が定義されていない場合にのみ使用してください。
    //! 可変長配列が定義されている場合は、サイズを指定できる BindShaderStorageBlock() を使用してください。
    //! 複数バッファーを設定することにより、GPU 処理中に GPU が参照していないバッファーを更新出来るようになります。
    //! count には pBufferPtr 配列の要素数を指定します。
    //!
    void BindShaderStorageBlock(int passIndex, const char* id, const nn::gfx::Buffer** pBufferPtr, int bufferCount) NN_NOEXCEPT;

    //! @brief シェーダーストレージブロックを設定します。
    //!
    //! @param[in] id シェーダーストレージブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] size 1 バッファーあたりのサイズ。
    //! @param[in] bufferCount pBufferPtr の要素数。
    //!
    //! @details
    //! シェーダーストレージブロック内に可変長配列が定義されている場合は、サイズを指定するために本関数を使用してください。
    //! 複数バッファーを設定することにより、GPU 処理中に GPU が参照していないバッファーを更新出来るようになります。
    //! count には pBufferPtr 配列の要素数を指定します。
    //!
    //! この関数は、全ての描画パスに対してシェーダーストレージブロックを設定することを試みます。
    //!
    void BindShaderStorageBlock(const char* id, const nn::gfx::Buffer** pBufferPtr, size_t size, int bufferCount) NN_NOEXCEPT;

    //! @brief シェーダーストレージブロックを設定します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id シェーダーストレージブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] size 1 バッファーあたりのサイズ。
    //! @param[in] bufferCount pBufferPtr の要素数。
    //!
    //! @details
    //! シェーダーストレージブロック内に可変長配列が定義されている場合は、サイズを指定するために本関数を使用してください。
    //! 複数バッファーを設定することにより、GPU 処理中に GPU が参照していないバッファーを更新出来るようになります。
    //! count には pBufferPtr 配列の要素数を指定します。
    //!
    void BindShaderStorageBlock(int passIndex, const char* id, const nn::gfx::Buffer** pBufferPtr, size_t size, int bufferCount) NN_NOEXCEPT;

    //! @brief シェーダーストレージブロックを取得します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id シェーダーストレージブロックの識別子
    //!
    //! @details
    //! 対象のシェーダーストレージブロックが見つからなかった場合には、NULL を返します。
    //!
    nns::g3d::ShaderStorageBlock* FindShaderStorageBlock(int passIndex, const char* id) NN_NOEXCEPT;

    //! @brief シェーダーストレージブロックの設定を解除します。
    //!
    //! @param[in] id シェーダーストレージブロックの識別子。
    //!
    //! @details
    //! この関数は、全ての描画パスに対してシェーダーストレージブロックの設定を解除することを試みます。
    //!
    void UnbindShaderStorageBlock(const char* id) NN_NOEXCEPT;

    //! @brief シェーダーストレージブロックの設定を解除します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id シェーダーストレージブロックの識別子。
    //!
    void UnbindShaderStorageBlock(int passIndex, const char* id) NN_NOEXCEPT;

    //! @brief サンプラーを設定します。
    //!
    //! @param[in] id サンプラーの識別子。
    //! @param[in] textureDescriptorSlot テクスチャーのデスクリプタースロット。
    //! @param[in] samplerDescriptorSlot サンプラーのデスクリプタースロット。
    //!
    //! @details
    //! この関数は、全ての描画パスに対してサンプラーの設定することを試みます。
    //!
    void BindSampler(const char* id, const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT;

    //! @brief サンプラーを設定します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id サンプラーの識別子。
    //! @param[in] textureDescriptorSlot テクスチャーのデスクリプタースロット。
    //! @param[in] samplerDescriptorSlot サンプラーのデスクリプタースロット。
    //!
    void BindSampler(int passIndex, const char* id, const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT;

    //! @brief サンプラーの設定を解除します。
    //!
    //! @param[in] id サンプラーの識別子。
    //!
    //! @details
    //! この関数は、全ての描画パスに対してサンプラーの設定を解除することを試みます。
    //!
    void UnbindSampler(const char* id) NN_NOEXCEPT;

    //! @brief サンプラーの設定を解除します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id サンプラーの識別子。
    //!
    void UnbindSampler(int passIndex, const char* id) NN_NOEXCEPT;

    //! @brief サンプラーを取得します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id サンプラーの識別子。
    //!
    //! @return サンプラーの設定のポインタを返します。
    //!
    //! @details
    //! id が一致するサンプラーが設定されていない場合には、 NULL を返します。
    //!
    nns::g3d::Sampler* FindSampler(int passIndex, const char* id) NN_NOEXCEPT;

    //! @brief ビューボリュームを取得します。
    //!
    //! @param[in] index ビューインデックス。
    //!
    //! @return nn::g3d::ViewVolume へのポインター。
    //!
    const nn::g3d::ViewVolume* GetViewVolume(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_pShapeObj->GetViewCount());
        return m_pViewVolumePtr[index];
    }

    //! @brief ビューボリュームを設定します。
    //!
    //! @param[in] index ビューインデックス。
    //! @param[in] pViewVolume nn::g3d::ViewVolume へのポインター。
    //!
    void SetViewVolume(int index, const nn::g3d::ViewVolume* pViewVolume) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_pShapeObj->GetViewCount());
        m_pViewVolumePtr[index] = pViewVolume;
    }

    //! @brief LOD レベル判定に使用する関数オブジェクトを取得します。
    //!
    //! @param[in] index ビューインデックス。
    //!
    //! @return nn::g3d::ICalculateLodLevelFunctor へのポインター。
    //!
    const nn::g3d::ICalculateLodLevelFunctor* GetCalculateLodLevelFunctor(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_pShapeObj->GetViewCount());
        return m_pCalculateLodLevelFunctorPtr[index];
    }

    //! @brief LODレベル判定に使用する関数オブジェクトを設定します。
    //!
    //! @param[in] index ビューインデックス。
    //! @param[in] pCalculateLodFunctor nn::g3d::ICalculateLodLevelFunctor へのポインター。
    //!
    void SetCalculateLodLevelFunctor(int index, nn::g3d::ICalculateLodLevelFunctor* pCalculateLodFunctor) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_pShapeObj->GetViewCount());
        m_pCalculateLodLevelFunctorPtr[index] = pCalculateLodFunctor;
    }

    //! @brief 常に指定した LOD レベルで描画される機能を有効にします。
    //!
    //! @param[in] lodLevel LOD レベル。
    //!
    void SetDrawLodEnabled(int lodLevel) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(lodLevel, 0, m_pShapeObj->GetMeshCount());
        m_DrawLodLevel = lodLevel;
    }

    //! @brief 常に指定した LOD レベルで描画される機能を無効にします。
    void SetDrawLodDisabled() NN_NOEXCEPT
    {
        m_DrawLodLevel = InvalidLod;
    }

    //! @brief 指定した 1 つのサブメッシュのみを描画する機能を有効にします。
    //!
    //! @param[in] submeshIndex サブメッシュインデックス。
    //!
    void SetDrawSingleSubmeshEnabled(int submeshIndex) NN_NOEXCEPT
    {
        m_DrawSingleSubmeshIndex = submeshIndex;
    }

    //! @brief 指定した 1 つのサブメッシュのみを描画する機能を無効にします。
    void SetDrawSingleSubmeshDisabled() NN_NOEXCEPT
    {
        m_DrawSingleSubmeshIndex = InvalidIndex;
    }

    //! @brief シェイプ描画に使用される nn::g3d::ShaderSelector を取得します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //!
    //! @return nn::g3d::ShaderSelector へのポインター。
    //!
    nn::g3d::ShaderSelector* GetShaderSelector(int passIndex) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
        return m_pRenderShapePtr[passIndex]->GetShaderSelector();
    }

    //! @brief シェイプ描画に使用される nn::g3d::ShaderSelector を取得します。
    //!
    //! @return nn::g3d::ShaderSelector へのポインター。
    //!
    //! @details
    //! 0番の描画パスに設定されている nn::g3d::ShaderSelector を取得します。
    //!
    nn::g3d::ShaderSelector* GetShaderSelector() NN_NOEXCEPT
    {
        return GetShaderSelector(0);
    }

    //! @brief シェイプ描画に使用される nn::g3d::ShaderSelector を取得します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //!
    //! @return nn::g3d::ShaderSelector へのポインター。
    //!
    const nn::g3d::ShaderSelector* GetShaderSelector(int passIndex) const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
        return m_pRenderShapePtr[passIndex]->GetShaderSelector();
    }

    //! @brief シェイプ描画に使用される nn::g3d::ShaderSelector を取得します。
    //!
    //! @return nn::g3d::ShaderSelector へのポインター。
    //!
    //! @details
    //! 0番の描画パスに設定されている nn::g3d::ShaderSelector を取得します。
    //!
    const nn::g3d::ShaderSelector* GetShaderSelector() const NN_NOEXCEPT
    {
        return GetShaderSelector(0);
    }

    //! @brief シェイプに関連付けられた nns::g3d::RenderShape を取得します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //!
    //! @return nns::g3d::RenderShape へのポインター。
    //!
    const nns::g3d::RenderShape* GetRenderShape(int passIndex) const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
        return m_pRenderShapePtr[passIndex];
    }

    //! @brief シェイプに関連付けられた nns::g3d::RenderShape を取得します。
    //!
    //! @return nns::g3d::RenderShape へのポインター。
    //!
    const nns::g3d::RenderShape* GetRenderShape() const NN_NOEXCEPT
    {
        return GetRenderShape(0);
    }

    //! @brief シェーダーを更新します。
    //!
    //! @details
    //! 設定されたシェーダーキーを元に適切なシェーダーを選択します。
    //!
    void UpdateShader(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //! @brief シェーダーを更新します。
    //!
    //! @details
    //! 設定されたシェーダーキーを元に適切なシェーダーを選択します。
    //!
    void UpdateShader(int passIndex, nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //! @brief ダミーサンプラーを設定します。
    //!
    //! @param[in] textureDescriptorSlot テクスチャーのデスクリプタースロット。
    //! @param[in] samplerDescriptorSlot サンプラーのデスクリプタースロット。
    //!
    //! @details
    //! テクスチャーが割り当てられていないサンプラーが存在する場合に、代替として使用するサンプラーを設定します。
    //!
    void BindDummySampler(const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        m_DummySampler.Initialize("DummySampler", samplerDescriptorSlot, textureDescriptorSlot);
    }

    //! @brief 初期化済みかを取得します。
    //!
    //! @return 初期化済みの場合は true、未初期化の場合は false を返します。
    //!
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    //@}

    //----------------------------------------
    //! @name 描画
    //@{


    //! @brief シェイプを描画します。
    //!
    //! @param[in] pCommandBuffer nn::gfx::CommandBuffer へのポインター。
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] bufferIndex バッファーインデックス。
    //! @param[in] pDrawCullingFunction カリング処理を行う関数へのポインター。
    //!
    //! @details
    //! シェイプを描画します。passIndex は RenderModel 生成時に指定した範囲内である必要があります。
    //! viewIndex はモデルが持つビューの範囲内である必要があります。
    //! bufferIndex はスケルトン、シェイプ、マテリアルおよび nns::g3d::RenderUnitObj::BindUniformBlock() で設定したユニフォームブロックすべてで有効なインデックスである必要があります。
    //! 具体的には nns::g3d::RenderUnitObj::BindUniformBlock() で 1 つのバッファーしか設定していない状態で、bufferIndex に 1 を指定することは不正です。
    //!
    //! SetViewVolume() によって nn::g3d::ViewVolume が設定されている場合、ビューフラスタムカリングが行われます。
    //! SetCalculateLodLevelFunctor() によって nn::g3d::ICalculateLodLevelFunctor が設定されている場合は、
    //! 設定された関数オブジェクトを呼び出し、描画 LOD レベルが決定されます。設定されていない場合は、lod レベル 0 のモデルが描画されます。
    //! SetDrawLodEnabled() で描画 LOD レベルが指定されている場合は、その LOD レベルで描画されます。
    //! SetDrawSingleSubmeshEnabled() で描画サブメッシュが指定されている場合、そのサブメッシュのみが描画されます。
    //!
    void Draw(nn::gfx::CommandBuffer* pCommandBuffer, int passIndex, int viewIndex, int bufferIndex, DrawCullingCallback pDrawCullingFunction) const NN_NOEXCEPT;


    //! @brief シェイプを描画します。
    //!
    //! @param[in] pCommandBuffer nn::gfx::CommandBuffer へのポインター。
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] bufferIndex バッファーインデックス。
    //!
    //! @details
    //! シェイプを描画します。passIndex は RenderModel 生成時に指定した範囲内である必要があります。
    //! viewIndex はモデルが持つビューの範囲内である必要があります。
    //! bufferIndex はスケルトン、シェイプ、マテリアルおよび nns::g3d::RenderUnitObj::BindUniformBlock() で設定したユニフォームブロックすべてで有効なインデックスである必要があります。
    //! 具体的には nns::g3d::RenderUnitObj::BindUniformBlock() で 1 つのバッファーしか設定していない状態で、bufferIndex に 1 を指定することは不正です。
    //!
    //! SetViewVolume() によって nn::g3d::ViewVolume が設定されている場合、ビューフラスタムカリングが行われます。
    //! SetCalculateLodLevelFunctor() によって nn::g3d::ICalculateLodLevelFunctor が設定されている場合は、
    //! 設定された関数オブジェクトを呼び出し、描画 LOD レベルが決定されます。設定されていない場合は、lod レベル 0 のモデルが描画されます。
    //! SetDrawLodEnabled() で描画 LOD レベルが指定されている場合は、その LOD レベルで描画されます。
    //! SetDrawSingleSubmeshEnabled() で描画サブメッシュが指定されている場合、そのサブメッシュのみが描画されます。
    //!
    void Draw(nn::gfx::CommandBuffer* pCommandBuffer, int passIndex, int viewIndex, int bufferIndex) const NN_NOEXCEPT;

    //@}

private:
    static const int InvalidIndex = -1;
    static const int InvalidLod = -1;

    const nn::g3d::ModelObj*                m_pModelObj;
    const nn::g3d::ShapeObj*                m_pShapeObj;
    RenderShape**                           m_pRenderShapePtr;
    UniformBlock**                          m_pUniformBlockPtr;
    ShaderStorageBlock**                     m_pShaderStorageBlockPtr;
    const nn::g3d::ViewVolume**             m_pViewVolumePtr;
    nn::g3d::ICalculateLodLevelFunctor**    m_pCalculateLodLevelFunctorPtr;
    Sampler                                 m_DummySampler;
    int                                     m_DrawLodLevel;
    int                                     m_DrawSingleSubmeshIndex;
    bool                                    m_IsInitialized;
    int                                     m_ShapeIndex;
    int                                     m_PassCount;
};


//--------------------------------------------------------------------------------------------------
//! @brief モデルの描画クラスです。
class RenderModelObj
{
    NN_DISALLOW_COPY(RenderModelObj);
public:
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    RenderModelObj() NN_NOEXCEPT
        : m_pModelObj(NULL)
        , m_pRenderUnitObj(NULL)
        , m_pRenderModel(NULL)
        , m_DrawSingleShapeIndex(InvalidIndex)
        , m_IsInitialized(false)
    {
    }

    //! @brief RenderModelObj を初期化します。
    //!
    //! @param[in] pModelObj nn::g3d::ModelObj へのポインター。
    //! @param[in] pRenderModel nns::g3d::RenderModel へのポインター。
    //!
    //! @pre
    //! - RenderModelObj が未初期化である。
    //!
    //! @post
    //! - RenderModelObj が初期化されている。
    //!
    void Initialize(nn::g3d::ModelObj* pModelObj, RenderModel* pRenderModel) NN_NOEXCEPT;

    //! @brief RenderModelObj を破棄します。
    //!
    //! @pre
    //! - RenderModelObj が初期化されている。
    //!
    //! @post
    //! - RenderModelObj が未初期化である。
    //!
    void Finalize() NN_NOEXCEPT;

    //! @brief シェーダーを更新します。
    //!
    //! @details
    //! 設定されたシェーダーキーを元に適切なシェーダーを選択します。
    //!
    void UpdateShader(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //! @brief シェーダーを更新します。
    //!
    //! @details
    //! 設定されたシェーダーキーを元に適切なシェーダーを選択します。
    //!
    void UpdateShader(int passIndex, nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //! @brief ダミーサンプラーを設定します。
    //!
    //! @param[in] textureDescriptorSlot テクスチャーのデスクリプタースロット。
    //! @param[in] samplerDescriptorSlot サンプラーのデスクリプタースロット。
    //!
    //! @details
    //! テクスチャーが割り当てられていないサンプラーが存在する場合に、代替として使用するサンプラーを設定します。
    //!
    void BindDummySampler(const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT(textureDescriptorSlot.IsValid());
        NN_ASSERT(samplerDescriptorSlot.IsValid());

        int shapeCount = m_pModelObj->GetShapeCount();
        for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
        {
            m_pRenderUnitObj[shapeIndex].BindDummySampler(textureDescriptorSlot, samplerDescriptorSlot);
        }
    }

    //! @brief ユニフォームブロックを設定します。
    //!
    //! @param[in] id ユニフォームブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] bufferCount pBufferPtr の要素数。
    //!
    //! @details
    //! 複数バッファーを設定することにより、GPU 処理中に GPU が参照していないバッファーを更新出来るようになります。
    //! count には pBufferPtr 配列の要素数を指定します。
    //! 配下にある全てのシェイプの全ての描画パスにユニフォームブロックを設定します。
    //!
    void BindUniformBlock(const char* id, const nn::gfx::Buffer** pBufferPtr, int bufferCount) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_NOT_NULL(id);
        NN_ASSERT_NOT_NULL(pBufferPtr);

        for (int shapeIndex = 0, shapeCount = m_pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            m_pRenderUnitObj[shapeIndex].BindUniformBlock(id, pBufferPtr, bufferCount);
        }
    }

    //! @brief ユニフォームブロックを設定します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id ユニフォームブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] bufferCount pBufferPtr の要素数。
    //!
    //! @details
    //! 複数バッファーを設定することにより、GPU 処理中に GPU が参照していないバッファーを更新出来るようになります。
    //! count には pBufferPtr 配列の要素数を指定します。
    //!
    void BindUniformBlock(int passIndex, const char* id, const nn::gfx::Buffer** pBufferPtr, int bufferCount) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(passIndex, 0, m_pRenderModel->GetPassCount());
        NN_ASSERT_NOT_NULL(id);
        NN_ASSERT_NOT_NULL(pBufferPtr);

        for (int shapeIndex = 0, shapeCount = m_pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            m_pRenderUnitObj[shapeIndex].BindUniformBlock(passIndex, id, pBufferPtr, bufferCount);
        }
    }

    //! @brief シェーダーストレージブロックを設定します。
    //!
    //! @param[in] id シェーダーストレージブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] bufferCount pBufferPtr の要素数。
    //!
    //! @details
    //! シェーダーストレージブロック内に可変長配列が定義されていない場合にのみ使用してください。
    //! 可変長配列が定義されている場合は、サイズを指定できる BindShaderStorageBlock() を使用してください。
    //! 複数バッファーを設定することにより、GPU 処理中に GPU が参照していないバッファーを更新出来るようになります。
    //! count には pBufferPtr 配列の要素数を指定します。
    //! 配下にある全てのシェイプの全ての描画パスにシェーダーストレージブロックを設定します。
    //!
    void BindShaderStorageBlock(const char* id, const nn::gfx::Buffer** pBufferPtr, int bufferCount) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_NOT_NULL(id);
        NN_ASSERT_NOT_NULL(pBufferPtr);

        for (int shapeIndex = 0, shapeCount = m_pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            m_pRenderUnitObj[shapeIndex].BindShaderStorageBlock(id, pBufferPtr, bufferCount);
        }
    }

    //! @brief シェーダーストレージブロックを設定します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id シェーダーストレージブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] bufferCount pBufferPtr の要素数。
    //!
    //! @details
    //! シェーダーストレージブロック内に可変長配列が定義されていない場合にのみ使用してください。
    //! 可変長配列が定義されている場合は、サイズを指定できる BindShaderStorageBlock() を使用してください。
    //! 複数バッファーを設定することにより、GPU 処理中に GPU が参照していないバッファーを更新出来るようになります。
    //! count には pBufferPtr 配列の要素数を指定します。
    //!
    void BindShaderStorageBlock(int passIndex, const char* id, const nn::gfx::Buffer** pBufferPtr, int bufferCount) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(passIndex, 0, m_pRenderModel->GetPassCount());
        NN_ASSERT_NOT_NULL(id);
        NN_ASSERT_NOT_NULL(pBufferPtr);

        for (int shapeIndex = 0, shapeCount = m_pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            m_pRenderUnitObj[shapeIndex].BindShaderStorageBlock(passIndex, id, pBufferPtr, bufferCount);
        }
    }

    //! @brief シェーダーストレージブロックを設定します。
    //!
    //! @param[in] id シェーダーストレージブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] size 1 バッファーあたりのサイズ。
    //! @param[in] bufferCount pBufferPtr の要素数。
    //!
    //! @details
    //! シェーダーストレージブロック内に可変長配列が定義されている場合は、サイズを指定するために本関数を使用してください。
    //! 複数バッファーを設定することにより、GPU 処理中に GPU が参照していないバッファーを更新出来るようになります。
    //! count には pBufferPtr 配列の要素数を指定します。
    //! 配下にある全てのシェイプの全ての描画パスにシェーダーストレージブロックを設定します。
    //!
    void BindShaderStorageBlock(const char* id, const nn::gfx::Buffer** pBufferPtr, size_t size, int bufferCount) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_NOT_NULL(id);
        NN_ASSERT_NOT_NULL(pBufferPtr);

        for (int shapeIndex = 0, shapeCount = m_pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            m_pRenderUnitObj[shapeIndex].BindShaderStorageBlock(id, pBufferPtr, size, bufferCount);
        }
    }

    //! @brief シェーダーストレージブロックを設定します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id シェーダーストレージブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] size 1 バッファーあたりのサイズ。
    //! @param[in] bufferCount pBufferPtr の要素数。
    //!
    //! @details
    //! シェーダーストレージブロック内に可変長配列が定義されている場合は、サイズを指定するために本関数を使用してください。
    //! 複数バッファーを設定することにより、GPU 処理中に GPU が参照していないバッファーを更新出来るようになります。
    //! count には pBufferPtr 配列の要素数を指定します。
    //!
    void BindShaderStorageBlock(int passIndex, const char* id, const nn::gfx::Buffer** pBufferPtr, size_t size, int bufferCount) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(passIndex, 0, m_pRenderModel->GetPassCount());
        NN_ASSERT_NOT_NULL(id);
        NN_ASSERT_NOT_NULL(pBufferPtr);

        for (int shapeIndex = 0, shapeCount = m_pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            m_pRenderUnitObj[shapeIndex].BindShaderStorageBlock(passIndex, id, pBufferPtr, size, bufferCount);
        }
    }

    //! @brief サンプラーを設定します。
    //!
    //! @param[in] id サンプラーの識別子。
    //! @param[in] samplerDescriptorSlot サンプラーのデスクリプタースロット。
    //! @param[in] textureDescriptorSlot テクスチャーのデスクリプタースロット。
    //!
    //! @details
    //! 配下にある全てのシェイプの全ての描画パスにユニフォームブロックを設定します。
    //!
    void BindSampler(const char* id, const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int shapeIndex = 0, shapeCount = m_pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            m_pRenderUnitObj[shapeIndex].BindSampler(id, textureDescriptorSlot, samplerDescriptorSlot);
        }
    }

    //! @brief サンプラーを設定します。
    //!
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] id サンプラーの識別子。
    //! @param[in] samplerDescriptorSlot サンプラーのデスクリプタースロット。
    //! @param[in] textureDescriptorSlot テクスチャーのデスクリプタースロット。
    //!
    void BindSampler(int passIndex, const char* id, const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(passIndex, 0, m_pRenderModel->GetPassCount());
        NN_ASSERT_NOT_NULL(id);
        NN_ASSERT(textureDescriptorSlot.IsValid());
        NN_ASSERT(samplerDescriptorSlot.IsValid());

        for (int shapeIndex = 0, shapeCount = m_pModelObj->GetShapeCount(); shapeIndex < shapeCount; ++shapeIndex)
        {
            m_pRenderUnitObj[shapeIndex].BindSampler(passIndex, id, textureDescriptorSlot, samplerDescriptorSlot);
        }
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief nn::g3d::ModelObj を取得します。
    //!
    //! @return nn::g3d::ModelObj へのポインター。
    //!
    nn::g3d::ModelObj* GetModelObj() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_pModelObj;
    }

    //! @brief nn::g3d::ModelObj を取得します。
    //!
    //! @return nn::g3d::ModelObj へのポインター。
    //!
    const nn::g3d::ModelObj* GetModelObj() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_pModelObj;
    }

    //! @brief nns::g3d::RenderUnitObj を取得します。
    //!
    //! @return nns::g3d::RenderUnitObj へのポインター。
    //!
    const nns::g3d::RenderUnitObj* GetRenderUnitObj(int shapeIndex) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(shapeIndex, 0, m_pModelObj->GetShapeCount());
        return &m_pRenderUnitObj[shapeIndex];
    }

    //! @brief nns::g3d::RenderUnitObj を取得します。
    //!
    //! @return nns::g3d::RenderUnitObj へのポインター。
    //!
    nns::g3d::RenderUnitObj* GetRenderUnitObj(int shapeIndex) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(shapeIndex, 0, m_pModelObj->GetShapeCount());
        return &m_pRenderUnitObj[shapeIndex];
    }

    //! @brief nns::g3d::RenderModel を取得します。
    //!
    //! @return nns::g3d::RenderModel へのポインター。
    //!
    const nns::g3d::RenderModel* GetRenderModel() const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_pRenderModel;
    }

    //! @brief nns::g3d::RenderModel を取得します。
    //!
    //! @return nns::g3d::RenderModel へのポインター。
    //!
    nns::g3d::RenderModel* GetRenderModel() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        return m_pRenderModel;
    }

    //! @brief 常に指定した LOD レベルで描画される機能を有効にします。
    //!
    //! @param[in] lodLevel LOD レベル
    //!
    void SetDrawLodEnabled(int lodLevel) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        int shapeCount = m_pModelObj->GetShapeCount();
        for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
        {
            m_pRenderUnitObj[shapeIndex].SetDrawLodEnabled(lodLevel);
        }
    }

    //! @brief 常に指定した LOD レベルで描画される機能を無効にします。
    void SetDrawLodDisabled() NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        int shapeCount = m_pModelObj->GetShapeCount();
        for (int shapeIndex = 0; shapeIndex < shapeCount; ++shapeIndex)
        {
            m_pRenderUnitObj[shapeIndex].SetDrawLodDisabled();
        }
    }

    //! @brief LODレベル判定に使用する関数オブジェクトを設定します。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] pCalculateLodFunctor nn::g3d::ICalculateLodLevelFunctor へのポインター。
    //!
    void SetCalculateLodLevelFunctor(int viewIndex, nn::g3d::ICalculateLodLevelFunctor* pCalculateLodFunctor) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int index = 0; index < m_pModelObj->GetShapeCount(); ++index)
        {
            m_pRenderUnitObj[index].SetCalculateLodLevelFunctor(viewIndex, pCalculateLodFunctor);
        }
    }

    //! @brief ビューボリュームを設定します。
    //!
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] pViewVolume nn::g3d::ViewVolume へのポインター。
    //!
    void SetViewVolume(int viewIndex, nn::g3d::ViewVolume* pViewVolume) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        for (int index = 0; index < m_pModelObj->GetShapeCount(); ++index)
        {
            m_pRenderUnitObj[index].SetViewVolume(viewIndex, pViewVolume);
        }
    }

    //! @brief 指定した 1 つのシェイプのみを描画する機能を有効にします。
    //!
    //! @param[in] shapeIndex シェイプインデックス。
    //!
    void SetDrawSingleShapeEnabled(int shapeIndex) NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(shapeIndex, 0, m_pModelObj->GetShapeCount());
        m_DrawSingleShapeIndex = shapeIndex;
    }

    //! @brief 指定した 1 つのシェイプのみを描画する機能を無効にします。
    void SetDrawSingleShapeDisabled() NN_NOEXCEPT
    {
        m_DrawSingleShapeIndex = InvalidIndex;
    }

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief モデルを描画します。
    //!
    //! @param[in] pCommandBuffer nn::gfx::CommandBuffer へのポインター。
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] bufferIndex バッファーインデックス。
    //! @param[in] pDrawCullingCallback 描画するか判定を行う関数へのポインター。
    //!
    //! @details
    //! モデルを描画します。passIndex は RenderModel 生成時に指定した範囲内である必要があります。
    //! viewIndex はモデルが持つビュー数の範囲内である必要があります。
    //! bufferIndex はスケルトン、シェイプ、マテリアルおよび nns::g3d::RenderUnitObj::BindUniformBlock() で設定したユニフォームブロックすべてで有効なインデックスである必要があります。
    //! 具体的には nns::g3d::RenderUnitObj::BindUniformBlock() で 1 つのバッファーしか設定していない状態で、bufferIndex に 1 を指定することは不正です。
    //! SetViewVolume() によって nn::g3d::ViewVolume が設定されている場合、ビューフラスタムカリングが行われます。
    //! SetCalculateLodLevelFunctor() によって nn::g3d::ICalculateLodLevelFunctor が設定されている場合は、
    //! 設定された関数オブジェクトを呼び出し、描画 lod レベルが決定されます。設定されていない場合は、lod レベル 0 のモデルが描画されます。
    //! SetDrawLodEnabled() で描画 LOD レベルが指定されている場合は、その LOD レベルで描画されます。
    //! SetDrawShapeEnabled() で描画シェイプが指定されている場合は、そのシェイプのみが描画されます。
    //!
    void Draw(nn::gfx::CommandBuffer* pCommandBuffer, int passIndex, int viewIndex, int bufferIndex, DrawCullingCallback pDrawCullingCallback) const NN_NOEXCEPT;

    //! @brief モデルを描画します。
    //!
    //! @param[in] pCommandBuffer nn::gfx::CommandBuffer へのポインター。
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] bufferIndex バッファーインデックス。
    //!
    //! @details
    //! モデルを描画します。passIndex は RenderModel 生成時に指定した範囲内である必要があります。
    //! viewIndex はモデルが持つビュー数の範囲内である必要があります。
    //! bufferIndex はスケルトン、シェイプ、マテリアルおよび nns::g3d::RenderUnitObj::BindUniformBlock() で設定したユニフォームブロックすべてで有効なインデックスである必要があります。
    //! 具体的には nns::g3d::RenderUnitObj::BindUniformBlock() で 1 つのバッファーしか設定していない状態で、bufferIndex に 1 を指定することは不正です。
    //! SetViewVolume() によって nn::g3d::ViewVolume が設定されている場合、ビューフラスタムカリングが行われます。
    //! SetCalculateLodLevelFunctor() によって nn::g3d::ICalculateLodLevelFunctor が設定されている場合は、
    //! 設定された関数オブジェクトを呼び出し、描画 lod レベルが決定されます。設定されていない場合は、lod レベル 0 のモデルが描画されます。
    //! SetDrawLodEnabled() で描画 LOD レベルが指定されている場合は、その LOD レベルで描画されます。
    //! SetDrawShapeEnabled() で描画シェイプが指定されている場合は、そのシェイプのみが描画されます。
    //!
    void Draw(nn::gfx::CommandBuffer* pCommandBuffer, int passIndex, int viewIndex, int bufferIndex) const NN_NOEXCEPT;

    //! @brief モデルを描画します。
    //!
    //! @param[in] pCommandBuffer nn::gfx::CommandBuffer へのポインター。
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] bufferIndex バッファーインデックス。
    //! @param[in] pDrawCullingCallback 描画するか判定を行う関数へのポインター。
    //!
    //! @details
    //! モデルを描画します。viewIndex はモデルが持つビュー数の範囲内である必要があります。
    //! bufferIndex はスケルトン、シェイプ、マテリアルおよび nns::g3d::RenderUnitObj::BindUniformBlock() で設定したユニフォームブロックすべてで有効なインデックスである必要があります。
    //! 具体的には nns::g3d::RenderUnitObj::BindUniformBlock() で 1 つのバッファーしか設定していない状態で、bufferIndex に 1 を指定することは不正です。
    //! SetViewVolume() によって nn::g3d::ViewVolume が設定されている場合、ビューフラスタムカリングが行われます。
    //! SetCalculateLodLevelFunctor() によって nn::g3d::ICalculateLodLevelFunctor が設定されている場合は、
    //! 設定された関数オブジェクトを呼び出し、描画 lod レベルが決定されます。設定されていない場合は、lod レベル 0 のモデルが描画されます。
    //! SetDrawLodEnabled() で描画 LOD レベルが指定されている場合は、その LOD レベルで描画されます。
    //! SetDrawShapeEnabled() で描画シェイプが指定されている場合は、そのシェイプのみが描画されます。
    //!
    void Draw(nn::gfx::CommandBuffer* pCommandBuffer, int viewIndex, int bufferIndex, DrawCullingCallback pDrawCullingCallback) const NN_NOEXCEPT;

    //! @brief モデルを描画します。
    //!
    //! @param[in] pCommandBuffer nn::gfx::CommandBuffer へのポインター。
    //! @param[in] viewIndex ビューインデックス。
    //! @param[in] bufferIndex バッファーインデックス。
    //!
    //! @details
    //! モデルを描画します。viewIndex はモデルが持つビュー数の範囲内である必要があります。
    //! bufferIndex はスケルトン、シェイプ、マテリアルおよび nns::g3d::RenderUnitObj::BindUniformBlock() で設定したユニフォームブロックすべてで有効なインデックスである必要があります。
    //! 具体的には nns::g3d::RenderUnitObj::BindUniformBlock() で 1 つのバッファーしか設定していない状態で、bufferIndex に 1 を指定することは不正です。
    //! SetViewVolume() によって nn::g3d::ViewVolume が設定されている場合、ビューフラスタムカリングが行われます。
    //! SetCalculateLodLevelFunctor() によって nn::g3d::ICalculateLodLevelFunctor が設定されている場合は、
    //! 設定された関数オブジェクトを呼び出し、描画 lod レベルが決定されます。設定されていない場合は、lod レベル 0 のモデルが描画されます。
    //! SetDrawLodEnabled() で描画 LOD レベルが指定されている場合は、その LOD レベルで描画されます。
    //! SetDrawShapeEnabled() で描画シェイプが指定されている場合は、そのシェイプのみが描画されます。
    //!
    void Draw(nn::gfx::CommandBuffer* pCommandBuffer, int viewIndex, int bufferIndex) const NN_NOEXCEPT;

    //! @brief 初期化済みであるかを取得します。
    //!
    //! @return 初期化済みである場合は true 、未初期化の場合は false を返します。
    //!
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    //@}

private:
    static const int InvalidIndex = -1;

    nn::g3d::ModelObj*  m_pModelObj;
    RenderUnitObj*      m_pRenderUnitObj;
    RenderModel*        m_pRenderModel;
    int                 m_DrawSingleShapeIndex;
    bool                m_IsInitialized;
};

}}
//--------------------------------------------------------------------------------------------------
