﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/g3d.h>
#include <nns/g3d/g3d_System.h>
#include <nns/g3d/g3d_RenderModel.h>

namespace nns { namespace g3d {

//--------------------------------------------------------------------------------------------------
//! @brief マテリアルのデプスステンシルステート、ブレンドステート、ラスタライザーステート、シェーディングモデルを管理するクラスです。
class RenderMaterial
{
    NN_DISALLOW_COPY(RenderMaterial);

public:
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    RenderMaterial() NN_NOEXCEPT
        : m_pResMaterial(NULL)
        , m_pResShadingModel(NULL)
        , m_pShadingModelObj(NULL)
        , m_pBlendState(NULL)
        , m_pDepthStencilState(NULL)
        , m_pRasterizerState(NULL)
        , m_pModelSamplerIndex(NULL)
        , m_IsMaterialShaderAssigned(false)
        , m_IsInitialized(false)
    {
    }

    //! @brief RenderMaterial を初期化します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] pResMaterial 対象のマテリアルへのポインター。
    //! @param[in] pResShaderArchive nn::g3d::ResShaderArchive へのポインター。
    //!
    //! @pre
    //! - RenderMaterial が未初期化である。
    //!
    //! @post
    //! - RenderMaterial が初期化されている。
    //!
    //! @return 初期化が成功した場合は true、失敗した場合は false を返します。
    //!
    //! @details
    //! pResShaderArchive から pResMaterial に関連付けられたシェーディングモデルを見つけ、初期化を行います。
    //! 適切なシェーディングモデルがシェーダーアーカイブの中に見つからない場合、初期化は行われません。
    //!
    //! 初期化後は、マテリアル描画時に以下のステートが適用されるようになっています。
    //!
    //! ブレンドステートは nn::gfx::BlendStateInfo::SetDefault() を行った後、 nn::gfx::BlendTargetStateInfo::SetDefault() を行った nn::gfx::BlendTargetStateInfo を 1 つ
    //! 指定し、初期化された nn::gfx::BlendState オブジェクト。
    //!
    //! デプスステンシルステートは nn::gfx::DepthStencilStateInfo::SetDefault() を行った後、以下を設定し、
    //! 初期化された nn::gfx::DepthStencilState オブジェクト。
    //!
    //! - SetDepthTestEnabled(true)
    //! - SetDepthWriteEnabled(true)
    //!
    //! ラスタライザーステートは nn::gfx::RasterizerStateInfo::SetDefault() 後、以下を設定し、初期化された nn::gfx::RasterizerState オブジェクト。
    //!
    //! - SetScissorEnabled(true)
    //!
    //! これらは SetBlendState()、SetDepthStencilState()、SetRasterizerState() で描画時に適用するステートを変更可能です。
    //!
    bool Initialize(nn::gfx::Device* pDevice, nn::g3d::ResMaterial* pResMaterial, nn::g3d::ResShaderArchive* pResShaderArchive) NN_NOEXCEPT;

    //! @brief RenderMaterial を初期化します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] pResMaterial 対象のマテリアルへのポインター。
    //! @param[in] pResShadingModel nn::g3d::ResShadingModel へのポインター。
    //!
    //! @pre
    //! - RenderMaterial が未初期化である。
    //!
    //! @post
    //! - RenderMaterial が初期化されている。
    //!
    //! @return 初期化が成功した場合は true、失敗した場合は false を返します。
    //!
    //! @details
    //! 初期化後は、マテリアル描画時に以下のステートが適用されるようになっています。
    //!
    //! ブレンドステートは nn::gfx::BlendStateInfo::SetDefault() を行った後、
    //! nn::gfx::BlendTargetStateInfo::SetDefault() を行った nn::gfx::BlendTargetStateInfo を 1 つ
    //! 指定し、初期化された nn::gfx::BlendState オブジェクト。
    //!
    //! デプスステンシルステートは nn::gfx::DepthStencilStateInfo::SetDefault() を行った後、以下をを設定し、
    //! 初期化された nn::gfx::DepthStencilState オブジェクト。
    //!
    //! - SetDepthTestEnabled(true)
    //! - SetDepthWriteEnabled(true)
    //!
    //! ラスタライザーステートは nn::gfx::RasterizerStateInfo::SetDefault() 後、以下を設定し、
    //! 初期化された nn::gfx::RasterizerState オブジェクト。
    //!
    //! - SetScissorEnabled(true)
    //!
    //! これらは SetBlendState()、SetDepthStencilState()、SetRasterizerState() で描画時に適用するステートを変更可能です。
    //!
    bool Initialize(nn::gfx::Device* pDevice, nn::g3d::ResMaterial* pResMaterial, nn::g3d::ResShadingModel* pResShadingModel) NN_NOEXCEPT;

    //! @brief RenderMaterial を破棄します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //!
    //! @pre
    //! - RenderMaterial が初期化されている。
    //!
    //! @post
    //! - RenderMaterial が未初期化である。
    //!
    void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief マテリアルを取得します。
    //!
    //! @return const nn::g3d::ResMaterial へのポインター。
    //!
    const nn::g3d::ResMaterial* GetResMaterial() const NN_NOEXCEPT
    {
        return m_pResMaterial;
    }

    //! @brief nn::g3d::ResShadingModel を取得します。
    //!
    //! @return nn::g3d::ResShadingModel へのポインター。
    //!
    const nn::g3d::ResShadingModel* GetResShadingModel() const NN_NOEXCEPT
    {
        return m_pResShadingModel;
    }

    //! @brief nn::g3d::ShadingModelObj を取得します。
    //!
    //! @return nn::g3d::ShadingModelObj へのポインター。
    //!
    nn::g3d::ShadingModelObj* GetShadingModelObj() NN_NOEXCEPT
    {
        return m_pShadingModelObj;
    }

    //! @brief nn::g3d::ShadingModelObj を取得します。
    //!
    //! @return nn::g3d::ShadingModelObj へのポインター。
    //!
    const nn::g3d::ShadingModelObj* GetShadingModelObj() const NN_NOEXCEPT
    {
        return m_pShadingModelObj;

    }
    //! @brief シェーディングモデルのサンプラーインデックスをモデルのサンプラーインデックスに変換します。
    //!
    //! @param[in] index シェーディングモデルのサンプラーインデックス。
    //!
    //! @return モデルのサンプラーインデックス。
    //!
    int GetModelSamplerIndex(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_pResShadingModel->GetSamplerCount());
        return m_pModelSamplerIndex[index];
    }

    //! @brief コマンドバッファーにブレンドステート、デプスステンシルステート、ラスタライザーステートを設定します。
    //!
    //! @param[out] pCommandBuffer nn::gfx::CommandBuffer へのポインター。
    //!
    void SetRenderStateTo(nn::gfx::CommandBuffer* pCommandBuffer) const NN_NOEXCEPT;

    //! @brief ブレンドステートを取得します。
    //!
    //! @return nn::gfx::BlendState へのポインター。
    //!
    const nn::gfx::BlendState* GetBlendState() const NN_NOEXCEPT
    {
        return m_pBlendState;
    }

    //! @brief ブレンドステートを設定します。
    //!
    //! @param[in] pBlendState nn::gfx::BlendState へのポインター。
    //!
    void SetBlendState(const nn::gfx::BlendState* pBlendState) NN_NOEXCEPT
    {
        NN_ASSERT(pBlendState == NULL || nn::gfx::IsInitialized(*pBlendState));
        m_pBlendState = pBlendState;
    }

    //! @brief デプスステンシルステートを取得します。
    //!
    //! @return nn::gfx::DepthStencilState へのポインター。
    //!
    const nn::gfx::DepthStencilState* GetDepthStencilState() const NN_NOEXCEPT
    {
        return m_pDepthStencilState;
    }

    //! @brief デプスステンシルステートを設定します。
    //!
    //! @param[in] pDepthStencilState nn::gfx::DepthStencilState へのポインター。
    //!
    void SetDepthStencilState(const nn::gfx::DepthStencilState* pDepthStencilState) NN_NOEXCEPT
    {
        NN_ASSERT(pDepthStencilState == NULL || nn::gfx::IsInitialized(*pDepthStencilState));
        m_pDepthStencilState = pDepthStencilState;
    }

    //! @brief ラスタライザーステートを取得します。
    //!
    //! @return nn::gfx::RasterizerState へのポインター。
    //!
    const nn::gfx::RasterizerState* GetRasterizerState() const NN_NOEXCEPT
    {
        return m_pRasterizerState;
    }

    //! @brief ラスタライザーステートを設定します。
    //!
    //! @param[in] pRasterizerState nn::gfx::RasterizerState へのポインター。
    //!
    void SetRasterizerState(const nn::gfx::RasterizerState* pRasterizerState) NN_NOEXCEPT
    {
        NN_ASSERT(pRasterizerState == NULL || nn::gfx::IsInitialized(*pRasterizerState));
        m_pRasterizerState = pRasterizerState;
    }

    //! @brief マテリアルシェーダーが割り当てられているかどうかを取得します。
    //!
    //! @return マテリアルシェーダーが割り当てられている場合は true、割り当てられていない場合は false を返します。
    //!
    bool IsMaterialShaderAssigned() const NN_NOEXCEPT
    {
        return m_IsMaterialShaderAssigned;
    }

    //! @brief 初期化済みかどうかを取得します。
    //!
    //! @return 初期化済みの場合は true、未初期化の場合は false を返します。
    //!
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    //@}

    //! @briefprivate
    static void InitializeDefaultRenderState(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //! @briefprivate
    static void FinalizeDefaultRenderState(nn::gfx::Device* pDevice) NN_NOEXCEPT;
private:
    nn::g3d::ResMaterial*             m_pResMaterial;
    nn::g3d::ResShadingModel*         m_pResShadingModel;
    nn::g3d::ShadingModelObj*         m_pShadingModelObj;
    const nn::gfx::BlendState*        m_pBlendState;
    const nn::gfx::DepthStencilState* m_pDepthStencilState;
    const nn::gfx::RasterizerState*   m_pRasterizerState;
    int*                              m_pModelSamplerIndex; // シェーダのサンプラーインデックスからモデルのサンプラーインデックスへの変換配列
    bool                              m_IsMaterialShaderAssigned;
    bool                              m_IsInitialized;

    static nn::gfx::BlendState        g_DefaultBlendState;
    static nn::gfx::DepthStencilState g_DefaultDepthStencilState;
    static nn::gfx::RasterizerState   g_DefaultRasterizerState;
};

//--------------------------------------------------------------------------------------------------

//! @brief ユニフォームブロックを管理するクラスです。
class UniformBlock
{
    NN_DISALLOW_COPY(UniformBlock);
public:

    //! @brief ユニフォームブロックの識別子の最大文字数です。
    static const size_t     MaxIdSize = 32;

    //----------------------------------------
    //! @name 構築/破棄
    // @{

    //! @brief コンストラクタです。
    //!
    //! @param[in] pWorkMemory nn::gfx::Buffer を管理するためのワークメモリへのポインター。
    //! @param[in] count 管理するバッファの要素数。
    //!
    //! @pre
    //! - pWorkBuffer != NULL
    //! - count > 0
    //!
    UniformBlock(void* pWorkMemory, int count) NN_NOEXCEPT
        : m_IsInitialized(false)
        , m_Size(0)
    {
        NN_ASSERT_NOT_NULL(pWorkMemory);
        NN_ASSERT_GREATER(count, 0);

        m_pBufferPtr = reinterpret_cast<const nn::gfx::Buffer**>(pWorkMemory);
        m_Count = count;
        m_Id[0] = '\0';
    }

    //! @brief 管理領域を初期化します。
    //!
    //! @param[in] id    ユニフォームブロックの識別子。
    //! @param[in] pBufferPtr nn::gfx::Buffer のポインター配列へのポインター。
    //! @param[in] size     バッファーのサイズ
    //! @param[in] count    pBufferPtr の要素数
    //!
    //! @pre
    //! - id != NULL
    //! - pBufferPtr != NULL
    //! - 0 <= count < m_Count
    //!
    //! @post
    //! - IsInitialized() == true
    //!
    void Initialize(const char* id, const nn::gfx::Buffer** pBufferPtr, size_t size, int count) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(id);
        NN_ASSERT_NOT_NULL(pBufferPtr);
        NN_ASSERT(0 < count && count <= m_Count);

        strncpy(m_Id, id, sizeof(m_Id));
        for (int bufferIndex = 0; bufferIndex < count; ++bufferIndex)
        {
            m_pBufferPtr[bufferIndex] = pBufferPtr[bufferIndex];
        }
        m_Size = size;
        m_IsInitialized = true;
    }

    //! @brief 管理領域を破棄します。
    //!
    //! @post
    //! - IsInitialized() == false
    //!
    void Finalize() NN_NOEXCEPT
    {
        m_Id[0] = '\0';
        m_Size = 0;
        m_IsInitialized = false;
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief ユニフォームブロックの識別子を取得します。
    //!
    //! @return  ユニフォームブロックの識別子。
    //!
    const char* GetId() const NN_NOEXCEPT
    {
        return m_Id;
    }

    //! @brief バッファーを取得します。
    //!
    //! @return const nn::gfx::Buffer へのポインター。
    //!
    //! @pre
    //! - 0 <= count < m_Count
    //!
    const nn::gfx::Buffer* GetBuffer(int index) const NN_NOEXCEPT
    {
        NN_ASSERT_RANGE(index, 0, m_Count);
        return m_pBufferPtr[index];
    }

    //! @brief バッファーのサイズを取得します。
    //!
    //! @return バッファーのサイズを返します。
    //!
    size_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;;
    }

    //! @brief バッファーの要素数を取得します。
    //!
    //! @return バッファーの要素数を返します。
    //!
    int GetCount() const NN_NOEXCEPT
    {
        return m_Count;
    }

    //! @brief 初期化済みかを取得します。
    //!
    //! @return 初期化済みの場合は true、未初期化の場合は false を返します。
    //!
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    //@}

private:
    char                    m_Id[MaxIdSize];
    const nn::gfx::Buffer** m_pBufferPtr;
    bool                    m_IsInitialized;
    size_t                  m_Size;
    int                     m_Count;
};

//! @brief シェーダーストレージブロックを管理するクラスです。
typedef UniformBlock ShaderStorageBlock;

//--------------------------------------------------------------------------------------------------

//! @brief サンプラーを管理するクラスです。
class Sampler
{
    NN_DISALLOW_COPY(Sampler);
public:

    //! @brief サンプラーの識別子の最大文字数です。
    static const size_t MaxIdSize = 32;

    //----------------------------------------
    //! @name 構築/破棄
    // @{

    //! @brief コンストラクタです。
    Sampler() NN_NOEXCEPT
    {
        Finalize();
    }

    //! @brief サンプラーの管理領域を初期化します。
    //!
    //! @param[in] id                    サンプラーの識別子
    //! @param[in] samplerDescriptorSlot    サンプラーのデスクリプタースロット。
    //! @param[in] textureDescriptorSlot    テクスチャーのデスクリプタースロット。
    //!
    //! @pre
    //! - id != NULL
    //! - samplerDescriptorSlot.IsValid()
    //! - textureDescriptorSlot.IsValid()
    //!
    //! @post
    //! - IsInitialized() == true
    //!
    void Initialize(const char* id, nn::gfx::DescriptorSlot samplerDescriptorSlot, nn::gfx::DescriptorSlot textureDescriptorSlot) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(id);
        NN_ASSERT(samplerDescriptorSlot.IsValid());
        NN_ASSERT(textureDescriptorSlot.IsValid());

        strncpy(m_Id, id, sizeof(m_Id));
        m_SamplerDescriptorSlot = samplerDescriptorSlot;
        m_TextureDescriptorSlot = textureDescriptorSlot;
        m_IsInitialized = true;
    }

    //! @brief サンプラーの管理領域を破棄します。
    //!
    //! @post
    //! - IsInitialized() == false
    //!
    void Finalize() NN_NOEXCEPT
    {
        m_Id[0] = '\0';
        nn::gfx::DescriptorSlot invalidDescriptorSlot;
        invalidDescriptorSlot.Invalidate();
        m_SamplerDescriptorSlot = invalidDescriptorSlot;
        m_TextureDescriptorSlot = invalidDescriptorSlot;
        m_IsInitialized = false;
    }

    //@}

     //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief サンプラーの識別子を取得します。
    //!
    //! @return  ユニフォームブロックの識別子。
    //!
    const char* GetId() const NN_NOEXCEPT
    {
        return m_Id;
    }

    //! @brief サンプラーのデスクリプタースロットを取得します。
    //!
    //! @return サンプラーのデスクリプタースロット
    //!
    nn::gfx::DescriptorSlot GetSamplerDescriptorSlot() const NN_NOEXCEPT
    {
        return m_SamplerDescriptorSlot;
    }

    //! @brief サンプラーのデスクリプタースロットを設定します。
    //!
    //! @param[in] descriptorSlot サンプラーのデスクリプタースロット
    //!
    void SetSamplerDescriptorSlot(const nn::gfx::DescriptorSlot& descriptorSlot) NN_NOEXCEPT
    {
        m_SamplerDescriptorSlot = descriptorSlot;
    }

    //! @brief テクスチャーのデスクリプタースロットを取得します。
    //!
    //! @return テクスチャーのデスクリプタースロット
    //!
    nn::gfx::DescriptorSlot GetTextureDescriptorSlot() const NN_NOEXCEPT
    {
        return m_TextureDescriptorSlot;
    }

    //! @brief テクスチャーのデスクリプタースロットを取得します。
    //!
    //! @param[in] descriptorSlot テクスチャーのデスクリプタースロット
    //!
    void SetTextureDescriptorSlot(const nn::gfx::DescriptorSlot& descriptorSlot) NN_NOEXCEPT
    {
        m_TextureDescriptorSlot = descriptorSlot;
    }

    //! @brief 初期化済みかを取得します。
    //!
    //! @return 初期化済みの場合は true、未初期化の場合は false を返します。
    //!
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    //@}

private:
    char                    m_Id[MaxIdSize];
    nn::gfx::DescriptorSlot m_SamplerDescriptorSlot;
    nn::gfx::DescriptorSlot m_TextureDescriptorSlot;
    bool                    m_IsInitialized;
};

//--------------------------------------------------------------------------------------------------

class RenderUnitObj;

//! @brief シェイプの描画設定クラスです。
class RenderShape
{
    NN_DISALLOW_COPY(RenderShape);
public:
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    RenderShape() NN_NOEXCEPT
        : m_pResShape(NULL)
        , m_pRenderMaterial(NULL)
        , m_pShaderSelector(NULL)
        , m_pUniformBlockList(NULL)
        , m_pShaderStorageBlockList(NULL)
        , m_pSamplerList(NULL)
        , m_IsInitialized(false)
    {
    }

    //! @brief RenderShape を初期化します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] pResShape 対象のシェイプへのポインター。
    //! @param[in] pRenderMaterial シェイプが参照するマテリアルの描画情報を管理する RenderMaterial へのポインター。
    //!
    //! @pre
    //! - RenderShape が未初期化である。
    //!
    //! @post
    //! - RenderShape が初期化されている。
    //!
    //! @return 初期化が成功した場合は true、失敗した場合は false を返します。
    //!
    //! @details
    //! 初期化後はシェイプ描画時に以下の頂点ステートが適用されるようになっています。
    //!
    //! pResShadingModel の持つ頂点属性情報と pResShapep が参照するマテリアルが持つ頂点属性情報を調べ、マテリアルが対応する頂点属性情報を持つ場合に、それを頂点属性
    //! として採用し、初期化された nn::gfx::VertexState オブジェクト。
    //!
    //! マテリアルが対応する頂点属性情報を持っていない場合には、頂点属性を比較し、一番最初に一致した頂点属性を割り当てます。
    //!
    bool Initialize(nn::gfx::Device* pDevice, nn::g3d::ResShape* pResShape, RenderMaterial* pRenderMaterial) NN_NOEXCEPT;

    //! @brief RenderShape を破棄します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //!
    //! @pre
    //! - RenderShape が初期化されている。
    //!
    //! @post
    //! - RenderShape が未初期化である。
    //!
    void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief シェイプを取得します。
    //!
    //! @return const nn::g3d::ResShape へのポインター。
    //!
    const nn::g3d::ResShape* GetResShape() const NN_NOEXCEPT
    {
        return m_pResShape;
    }

    //! @brief RenderMaterial を取得します。
    //!
    //! @return const nns::g3d::RenderMaterial へのポインター。
    //!
    nns::g3d::RenderMaterial* GetRenderMaterial() NN_NOEXCEPT
    {
        return m_pRenderMaterial;
    }

    //! @brief RenderMaterial を取得します。
    //!
    //! @return const nns::g3d::RenderMaterial へのポインター。
    //!
    const nns::g3d::RenderMaterial* GetRenderMaterial() const NN_NOEXCEPT
    {
        return m_pRenderMaterial;
    }

    //! @brief コマンドバッファーに描画の設定を行います。
    //!
    //! @param[out] pCommandBuffer  nn::gfx::CommandBuffer へのポインター。
    //! @param[in]  pRenderUnitObj  描画対象の RenderUnitObj へのポインター。
    //! @param[in]  viewIndex       ビューインデックス。
    //! @param[in]  bufferIndex     バッファーインデックス。
    //!
    //! @details
    //! シェイプを描画するための設定をコマンドバッファーに設定します。
    //! 具体的には以下のステート・オブジェクトが設定されます。
    //! - VertexBuffer
    //! - VertexState
    //! - BlendState
    //! - RasterizerState
    //! - DepthStencilState
    //! - ResShaderProgram
    //! - Sampler
    //! - Uniform Block
    //! - Shader Storage Block
    //!
    void SetRenderStateTo(nn::gfx::CommandBuffer* pCommandBuffer, const nns::g3d::RenderUnitObj* pRenderUnitObj, int viewIndex, int bufferIndex) const NN_NOEXCEPT;

    //! @brief 頂点ステートを取得します。
    //!
    //! @return nn::gfx::VertexState へのポインター。
    //!
    const nn::gfx::VertexState* GetVertexState() const NN_NOEXCEPT
    {
        return &m_VertexState;
    }

    //! @brief 頂点ステートを取得します。
    //!
    //! @return nn::gfx::VertexState へのポインター。
    //!
    nn::gfx::VertexState* GetVertexState() NN_NOEXCEPT
    {
        return &m_VertexState;
    }

    //! @brief 初期化済みかを取得します。
    //!
    //! @return 初期化済みの場合は true、未初期化の場合は false を返します。
    //!
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    //! @brief シェイプ描画に使用される nn::g3d::ShaderSelector を取得します。
    //!
    //! @return nn::g3d::ShaderSelector へのポインター。
    //!
    const nn::g3d::ShaderSelector* GetShaderSelector() const NN_NOEXCEPT
    {
        return m_pShaderSelector;
    }

    //! @brief シェイプ描画に使用される nn::g3d::ShaderSelector を取得します。
    //!
    //! @return nn::g3d::ShaderSelector へのポインター。
    //!
    nn::g3d::ShaderSelector* GetShaderSelector() NN_NOEXCEPT
    {
        return m_pShaderSelector;
    }

    //! @brief シェーダーを更新します。
    //!
    //! @details
    //! 設定されたシェーダーキーを元に適切なシェーダーを選択します。
    //!
    void UpdateShader(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //! @brief ユニフォームブロックリストを設定します。
    //!
    //! @param[in] pUniformBlockList nns::g3d::UniformBlock の配列へのポインター。
    //! @param[in] uniformBlockCount pUniformBlockList の要素数。
    //!
    void RegisterUniformBlockList(const UniformBlock* pUniformBlockList, int uniformBlockCount) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_NOT_NULL(pUniformBlockList);
        NN_ASSERT_EQUAL(uniformBlockCount, m_pRenderMaterial->GetResShadingModel()->GetUniformBlockCount());
        NN_UNUSED(uniformBlockCount);
        m_pUniformBlockList = pUniformBlockList;
    }

    //! @brief ユニフォームブロックリストの設定を解除します。
    //!
    //! @return 設定を解除した nns::g3d::UniformBlock の配列へのポインター。
    void UnregisterUniformBlockList() NN_NOEXCEPT
    {
        m_pUniformBlockList = NULL;
    }

    //! @brief シェーダーストレージブロックリストを設定します。
    //!
    //! @param[in] pShaderStorageBlockList nns::g3d::ShaderStorageBlock の配列へのポインター。
    //! @param[in] shaderStorageBlockCount pShaderStorageBlockList の要素数。
    //!
    void RegisterShaderStorageBlockList(const ShaderStorageBlock* pShaderStorageBlockList, int shaderStorageBlockCount) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_NOT_NULL(pShaderStorageBlockList);
        NN_ASSERT_EQUAL(shaderStorageBlockCount, m_pRenderMaterial->GetResShadingModel()->GetShaderStorageBlockCount());
        NN_UNUSED(shaderStorageBlockCount);
        m_pShaderStorageBlockList = pShaderStorageBlockList;
    }

    //! @brief シェーダーストレージブロックリストの設定を解除します。
    //!
    //! @return 設定を解除した nns::g3d::ShaderStorageBlock の配列へのポインター。
    void UnregisterShaderStorageBlockList() NN_NOEXCEPT
    {
        m_pShaderStorageBlockList = NULL;
    }

    //! @brief サンプラーを設定します。
    //!
    //! @param[in] id サンプラーの識別子。
    //! @param[in] samplerIndex 設定するサンプラーのインデックス。
    //! @param[in] textureDescriptorSlot テクスチャーのデスクリプタースロット。
    //! @param[in] samplerDescriptorSlot サンプラーのデスクリプタースロット。
    //!
    void InitializeSampler(const char* id, int samplerIndex, const nn::gfx::DescriptorSlot& textureDescriptorSlot, const nn::gfx::DescriptorSlot& samplerDescriptorSlot) NN_NOEXCEPT;

    //! @brief サンプラーを取得します。
    //!
    //! @param[in] id サンプラーの識別子
    //!
    nns::g3d::Sampler* FindSampler(const char* id) NN_NOEXCEPT;

    //! @brief ユニフォームブロックを破棄します。
    //!
    //! @param[in] id サンプラーの識別子
    //!
    void FinalizeSampler(const char* id) NN_NOEXCEPT;

    //@}
private:
    enum MemoryBlock
    {
        MemoryBlock_AttribStateInfo,
        MemoryBlock_BufferStateInfo,
        MemoryBlock_Max,
    };

    void InititalizeVertexState(nn::gfx::Device* pDevice, nn::gfx::VertexState* pVertexState) NN_NOEXCEPT;
    void FinalizeVertexState(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    void InitializeSamplerList() NN_NOEXCEPT;
    void FinalizeSamplerList() NN_NOEXCEPT;

    NN_FORCEINLINE void LoadTextureAndSampler(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::g3d::ResShaderProgram* pResShaderProgram,
        const nns::g3d::RenderUnitObj* pRenderUnitObj
    ) const NN_NOEXCEPT;

    NN_FORCEINLINE void LoadTextureAndSampler(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::gfx::DescriptorSlot& textureDescriptorSlot,
        const nn::gfx::DescriptorSlot& samplerDescriptorSlot,
        const nn::g3d::ResShaderProgram* pResShaderProgram,
        int samplerIndex
    ) const NN_NOEXCEPT;

    NN_FORCEINLINE void LoadUniformBlock(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::g3d::ResShaderProgram* pResShaderProgram,
        int viewIndex,
        int bufferIndex,
        const nns::g3d::RenderUnitObj* pRenderUnitObj
    ) const NN_NOEXCEPT;

    NN_FORCEINLINE void LoadUniformBlock(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::gfx::Buffer* pBuffer,
        const nn::g3d::ResShaderProgram* pResShaderProgram,
        int blockIndex,
        size_t size
    ) const NN_NOEXCEPT;

    NN_FORCEINLINE void LoadShaderStorageBlock(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::g3d::ResShaderProgram* pResShaderProgram,
        int bufferIndex,
        const nns::g3d::RenderUnitObj* pRenderUnitObj
    ) const NN_NOEXCEPT;

    NN_FORCEINLINE void LoadShaderStorageBlock(
        nn::gfx::CommandBuffer* pCommandBuffer,
        const nn::gfx::Buffer* pBuffer,
        const nn::g3d::ResShaderProgram* pResShaderProgram,
        int blockIndex,
        size_t size
    ) const NN_NOEXCEPT;

private:
    nn::g3d::ResShape*                  m_pResShape;
    RenderMaterial*                     m_pRenderMaterial;
    nn::g3d::ShaderSelector*            m_pShaderSelector;
    const nns::g3d::UniformBlock*       m_pUniformBlockList;
    const nns::g3d::ShaderStorageBlock* m_pShaderStorageBlockList;
    nns::g3d::Sampler*                  m_pSamplerList;
    nn::gfx::VertexState                m_VertexState;
    bool                                m_IsInitialized;
};

//--------------------------------------------------------------------------------------------------

//! @brief モデルの頂点ステート、デプスステンシルステート、ブレンドステート、ラスタライザーステート、シェーディングモデルを管理するクラスです。
class RenderModel
{
     NN_DISALLOW_COPY(RenderModel);
public:
    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    RenderModel() NN_NOEXCEPT
        : m_pResModel(NULL)
        , m_pRenderMaterialPtr(NULL)
        , m_pRenderShapePtr(NULL)
        , m_PassCount(0)
        , m_IsInitialized(false)
    {
    }

    //! @brief RenderModel を初期化します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] pResModel 対象のモデルへのポインター。
    //! @param[in] passCount 描画パスの数
    //!
    //! @pre
    //! - !IsInitialized()
    //! - pResModel != NULL
    //! - passCount > 0
    //!
    //! @post
    //! - IsInitialized()
    //!
    void Initialize(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel, int passCount) NN_NOEXCEPT;

    //! @brief RenderModel を初期化します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] pResModel 対象のモデルへのポインター。
    //!
    //! @pre
    //! - !IsInitialized()
    //! - pResModel != NULL
    //!
    //! @post
    //! - IsInitialized()
    //!
    //! @details
    //! この関数は Initialize(pDevice, pResModel, 1) を呼ぶことに相当します。
    //!
    void Initialize(nn::gfx::Device* pDevice, nn::g3d::ResModel* pResModel) NN_NOEXCEPT;

    //! @brief RenderModel を破棄します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //!
    //! @pre
    //! - pDevice != NULL
    //! - nn::gfx::IsInitialized(*pDevice)
    //! - IsInitialized()
    //!
    //! @post
    //! - !IsInitialized()
    //!
    void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //! @brief 描画パスを構築します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] pResShaderArchive モデルのシェーダー割り当てに設定されているシェーディングモデルを含んだ nn::g3d::ResShaderArchive へのポインター。
    //!
    //! @pre
    //! - IsInitialized()
    //! - pDevice != NULL
    //! - nn::gfx::IsInitialized(*pDevice)
    //! - 0 <= passIndex < m_PassCount
    //! - pResShaderArchive != NULL
    //!
    //! @return 設定に成功した場合は true、失敗した場合は false を返します。
    //!
    //! @details
    //! pResModel 配下の nn::g3d::ResShape および nn::g3d::ResMaterial に対して、 nns::g3d::RenderShape、 nns::g3d::RenderMaterial を作成し、初期化を行います。
    //! nns::g3d::RenderMaterial の初期化には、 nns::g3d::RenderMaterial::Initialize() の nn::g3d::ResShaderArchive を引数にとるものが使用されます。
    //!
    bool CreateDrawPass(nn::gfx::Device* pDevice, int passIndex, nn::g3d::ResShaderArchive* pResShaderArchive) NN_NOEXCEPT;

    //! @brief 描画パスを構築します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] pResShaderArchive モデルのシェーダー割り当てに設定されているシェーディングモデルを含んだ nn::g3d::ResShaderArchive へのポインター。
    //!
    //! @return 設定に成功した場合は true、失敗した場合は false を返します。
    //!
    //! @details
    //! この関数は CreateDrawPass(pDevice, 0, pResShaderArchive) を呼ぶことに相当します。
    //!
    bool CreateDrawPass(nn::gfx::Device* pDevice, nn::g3d::ResShaderArchive* pResShaderArchive) NN_NOEXCEPT;

    //! @brief 描画パスを構築します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] pResShadingModel モデルのシェーダー割り当てに設定されているシェーディングモデルを含んだ nn::g3d::ResShaderArchive へのポインター。
    //!
    //! @pre
    //! - IsInitialized()
    //! - pDevice != NULL
    //! - nn::gfx::IsInitialized(*pDevice)
    //! - 0 <= passIndex < m_PassCount
    //! - pResShadingModel != NULL
    //!
    //! @return 設定に成功した場合は true、失敗した場合は false を返します。
    //!
    //! @details
    //! pResModel 配下の nn::g3d::ResShape および nn::g3d::ResMaterial に対して、 nns::g3d::RenderShape、 nns::g3d::RenderMaterial を作成し、初期化を行います。
    //! nns::g3d::RenderMaterial の初期化には、 nns::g3d::RenderMaterial::Initialize() の nn::g3d::ResShadingModel を引数にとるものが使用されます。
    //!
    bool CreateDrawPass(nn::gfx::Device* pDevice, int passIndex, nn::g3d::ResShadingModel* pResShadingModel) NN_NOEXCEPT;

    //! @brief 描画パスを構築します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] pResShadingModel モデルのシェーダー割り当てに設定されているシェーディングモデルを含んだ nn::g3d::ResShaderArchive へのポインター。
    //!
    //! @return 設定に成功した場合は true、失敗した場合は false を返します。
    //!
    //! @details
    //! この関数は CreateDrawPass(pDevice, 0, pResShadingModel) を呼ぶことに相当します。
    //!
    bool CreateDrawPass(nn::gfx::Device* pDevice, nn::g3d::ResShadingModel* pResShadingModel) NN_NOEXCEPT;

    //! @brief 描画パスを構築します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] passIndex 描画パスのインデックス。
    //! @param[in] pResShaderArchiveArray モデルと組み合わせて使用するシェーディングモデルを含んだ nn::g3d::ResShaderArchive へのポインターの配列。
    //! @param[in] resShaderArchiveCount nn::g3d::ResShaderArchive へのポインターの配列の要素数。
    //!
    //! @pre
    //! - IsInitialized()
    //! - pDevice != NULL
    //! - nn::gfx::IsInitialized(*pDevice)
    //! - 0 <= passIndex < m_PassCount
    //! - pResShaderArchiveArray != NULL
    //! - resShaderArchiveCount == m_pResModel->GetMaterialCount()
    //!
    //! @return 設定に成功した場合は true、失敗した場合は false を返します。
    //!
    //! @details
    //! pResModel 配下の nn::g3d::ResShape および nn::g3d::ResMaterial に対して、 nns::g3d::RenderShape、 nns::g3d::RenderMaterial を作成し、初期化を行います。
    //! nns::g3d::RenderMaterial の初期化には、 nns::g3d::RenderMaterial::Initialize() の nn::g3d::ResShaderArchive を引数にとるものが使用され、
    //! インデックス 0 の nns::g3d::RenderMaterial には配列のインデックス 0 の要素の nn::g3d::ResShaderArchive が使用されます。
    //!
    bool CreateDrawPass(nn::gfx::Device* pDevice, int passIndex, nn::g3d::ResShaderArchive** pResShaderArchiveArray, int resShaderArchiveCount) NN_NOEXCEPT;

    //! @brief 描画パスを構築します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] pResShaderArchiveArray モデルと組み合わせて使用するシェーディングモデルを含んだ nn::g3d::ResShaderArchive へのポインターの配列。
    //! @param[in] resShaderArchiveCount nn::g3d::ResShaderArchive へのポインターの配列の要素数。
    //!
    //! @return 設定に成功した場合は true、失敗した場合は false を返します。
    //!
    //! @details
    //! この関数は CreateDrawPass(pDevice, 0, pResShaderArchiveArray, resShaderArchiveCount) を呼ぶことに相当します。
    //!
    bool CreateDrawPass(nn::gfx::Device* pDevice, nn::g3d::ResShaderArchive** pResShaderArchiveArray, int resShaderArchiveCount) NN_NOEXCEPT;

    //! @brief 描画パスを破棄します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //! @param[in] passIndex 描画パスのインデックス。
    //!
    void DestroyDrawPass(nn::gfx::Device* pDevice, int passIndex) NN_NOEXCEPT;

    //! @brief 描画パスを破棄します。
    //!
    //! @param[in] pDevice デバイスへのポインター。
    //!
    //! @details
    //! この関数は DestroyDrawPass(pDevice, 0) を呼ぶことに相当します。
    //!
    void DestroyDrawPass(nn::gfx::Device* pDevice) NN_NOEXCEPT;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief モデルを取得します。
    //!
    //! @return const nn::g3d::ResModel へのポインター。
    //!
    const nn::g3d::ResModel* GetResModel() const NN_NOEXCEPT
    {
        return m_pResModel;
    }

    //! @brief モデルを取得します。
    //!
    //! @return const nn::g3d::ResModel へのポインター。
    //!
    nn::g3d::ResModel* GetResModel() NN_NOEXCEPT
    {
        return m_pResModel;
    }

    //! @brief RenderMaterial を取得します。
    //!
    //! @param[in] passIndex        描画パスのインデックス。
    //! @param[in] materialIndex    マテリアルのインデックス。
    //!
    //! @return nns::g3d::RenderMaterial へのポインター。
    //!
    RenderMaterial* GetRenderMaterial(int passIndex,int materialIndex) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
        NN_ASSERT_RANGE(materialIndex, 0, m_pResModel->GetMaterialCount());
        return &m_pRenderMaterialPtr[passIndex][materialIndex];
    }

    //! @brief RenderMaterial を取得します。
    //!
    //! @param[in] materialIndex    マテリアルのインデックス。
    //!
    //! @return nns::g3d::RenderMaterial へのポインター。
    //!
    //! @details
    //! この関数は GetRenderMaterial(0, materialIndex) を呼ぶことに相当します。
    //!
    RenderMaterial* GetRenderMaterial(int materialIndex) NN_NOEXCEPT
    {
        return GetRenderMaterial(0, materialIndex);
    }

    //! @brief RenderMaterial を取得します。
    //!
    //! @param[in] passIndex        描画パスのインデックス。
    //! @param[in] materialIndex    マテリアルのインデックス。
    //!
    //! @return nns::g3d::RenderMaterial へのポインター。
    //!
    const RenderMaterial* GetRenderMaterial(int passIndex, int materialIndex) const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
        NN_ASSERT_RANGE(materialIndex, 0, m_pResModel->GetMaterialCount());
        return &m_pRenderMaterialPtr[passIndex][materialIndex];
    }

    //! @brief RenderMaterial を取得します。
    //!
    //! @param[in] materialIndex    マテリアルのインデックス。
    //!
    //! @return nns::g3d::RenderMaterial へのポインター。
    //!
    //! @details
    //! この関数は GetRenderMaterial(0, materialIndex) を呼ぶことに相当します。
    //!
    const RenderMaterial* GetRenderMaterial(int materialIndex) const NN_NOEXCEPT
    {
        return GetRenderMaterial(0, materialIndex);
    }

    //! @brief RenderShape を取得します。
    //!
    //! @param[in] passIndex        描画パスのインデックス。
    //! @param[in] shapeIndex       シェイプのインデックス。
    //!
    //! @return nn::g3d::RenderShape へのポインター。
    //!
    RenderShape* GetRenderShape(int passIndex, int shapeIndex) NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
        NN_ASSERT_RANGE(shapeIndex, 0, m_pResModel->GetShapeCount());
        return &m_pRenderShapePtr[passIndex][shapeIndex];
    }

    //! @brief RenderShape を取得します。
    //!
    //! @param[in] passIndex        描画パスのインデックス。
    //! @param[in] shapeIndex       シェイプのインデックス。
    //!
    //! @return nn::g3d::RenderShape へのポインター。
    //!
    const RenderShape* GetRenderShape(int passIndex, int shapeIndex) const NN_NOEXCEPT
    {
        NN_ASSERT(IsInitialized());
        NN_ASSERT_RANGE(passIndex, 0, m_PassCount);
        NN_ASSERT_RANGE(shapeIndex, 0, m_pResModel->GetShapeCount());
        return &m_pRenderShapePtr[passIndex][shapeIndex];
    }

    //! @brief 描画パスの数を取得します。
    //!
    //! @return 描画パスの数を返します。
    //!
    int GetPassCount() const NN_NOEXCEPT
    {
        return m_PassCount;
    }

    //! @brief 初期化済みかを取得します。
    //!
    //! @return 初期化済みの場合は true、未初期化の場合は false を返します。
    //!
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

private:
    nn::g3d::ResModel*  m_pResModel;
    RenderMaterial**    m_pRenderMaterialPtr;
    RenderShape**       m_pRenderShapePtr;
    int                 m_PassCount;
    bool                m_IsInitialized;
};

}}
//--------------------------------------------------------------------------------------------------
