﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/hid/hid_DebugPad.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_Npad.h>
#if defined(NN_BUILD_CONFIG_SPEC_NX)
#include <nn/hid/hid_NpadJoyDual.h>
#endif

#include <nns/console/console_InputEventListener.h>
#include <nns/console/detail/console_Hid.h>

namespace nns { namespace console {

// InputManager : パッドのボタン入力や、 USB Keyboard のキー入力を取得する
//  - 対応する入力
//    - パッド : 最初に検出されたデバイスから独占的に入力を取得し、切断時は別の有効なデバイスに切り替わる
//      - DebugPad
//      - Npad No1 (FullKey, JoyDual)
//      - Npad Handheld
//    - USB Keyboard
template <int InputIntervalFirst, int InputInterval>
class InputManager final
{
private:
    detail::PadContext m_PadContext;
    detail::KeyboardContext m_KeyboardContext;

public:
    InputManager() NN_NOEXCEPT
        : m_PadContext(InputIntervalFirst, InputInterval)
        , m_KeyboardContext(InputIntervalFirst, InputInterval)
    {
    }

    void Initialize() NN_NOEXCEPT
    {
        nn::hid::InitializeDebugPad();
        nn::hid::InitializeKeyboard();
        nn::hid::InitializeNpad();

#if defined(NN_BUILD_CONFIG_SPEC_NX)
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask | nn::hid::NpadStyleJoyDual::Mask);
#else
        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
#endif
        const nn::hid::NpadIdType npadIds[] =
        {
            nn::hid::NpadId::No1,
            nn::hid::NpadId::Handheld,
        };
        nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));
    }

    void SetPadEventListener(PadEventListenerBase* pEventListener) NN_NOEXCEPT
    {
        m_PadContext.SetEventListener(pEventListener);
    }

    void SetKeyboardEventListener(KeyboardEventListenerBase* pEventListener) NN_NOEXCEPT
    {
        m_KeyboardContext.SetEventListener(pEventListener);
    }

    void UnsetEventListeners() NN_NOEXCEPT
    {
        m_PadContext.UnsetEventListener();
        m_KeyboardContext.UnsetEventListener();
    }

    void Update() NN_NOEXCEPT
    {
        m_PadContext.Update();
        m_KeyboardContext.Update();
    }
};

typedef InputManager<16, 8> DefaultInputManager;

} // ~namespace nns::console
}
