﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include <cassert>
#include <limits>

class LoudnessUtilTest;

struct LoudnessValues
{
    uint64_t index;
    float momentary;
    float shortTerm;
    float integrated;
};

struct FilterCoeff
{
    float a1;
    float a2;
    float b0;
    float b1;
    float b2;
};

struct BiquadFilterState
{
    float context[2];
};

namespace {

// TODO: 長時間録音のための対処
// TODO: true peak の計測
// TODO: 32kHz の coeff の算出。独自計算で算出することは許可されている。
//       (Tests have shown that the performance of the algorithm is not sensitive to small variations in these coefficients.)
// TODO: Range LU への対応
// TODO: sampleFormat 指定
// TODO: 最大チャンネル数の可変化

const FilterCoeff ShelvingFilterCoeff48kHz{ -1.69065929318241f, 0.73248077421585f, 1.53512485958697f, -2.69169618940638f, 1.19839281085285f };
const FilterCoeff HighPassFilterCoeff48kHz{ -1.99004745483398f, 0.99007225036621f, 1.0f,              -2.0f,              1.0f              };

const int ChannelCountMax = 6;
const float ChannelWeightings[ChannelCountMax] = { 1.0f, 1.0f, 1.0f, 0.0f, 1.41f, 1.41f }; // LFE は計算から除外される。

const int StepInterval = 100; // [ms]
const int MomentaryInterval = 400; // [ms]
const int ShortTermInterval = 3000; // [ms]
const int MomentaryValueBufferCount = MomentaryInterval / StepInterval;
const int ShortTermValueBufferCount = ShortTermInterval / StepInterval;
const int StepHistoryCount = 10; // StepInterval * 10 = 1 [sec]
const float AbsoluteThreshold = -70.f;
const float RelativeThreshold = -10.f;
const int DefaultGatedBlockBufferDuration = 60 * 60 * 10; // 60 min * 60 sec * 10 steps (1 step = 100msec)

template<typename T, int count>
class ValueBuffer
{
private:
    T m_Buffer[count];
    int m_CurPos;
    int m_PushedCount;
    bool m_Filled;

public:
    ValueBuffer()
        : m_Filled{ false }
        , m_CurPos{ 0 }
        , m_PushedCount{ 0 }
    {
        for (auto& t : m_Buffer)
        {
            t = 0;
        }
    }

    // Push a value to buffer.
    // Oldest value is dropped if the buffer is full.
    void PushValue(T val)
    {
        m_Buffer[m_CurPos++] = val;
        if (m_CurPos >= count)
        {
            m_Filled = true;
        }
        m_PushedCount = (m_PushedCount + 1) > count ? count : m_PushedCount + 1;
        m_CurPos %= count;
    }

    bool Filled() const
    {
        return m_Filled;
    }

    T Sum() const
    {
        T s = 0;
        for (const auto& t : m_Buffer)
        {
            s += t;
        }
        return s;
    }

    int Length() const
    {
        return count;
    }

    int PushedCount() const
    {
        return m_PushedCount;
    }

    void Reset()
    {
        m_Filled = false;
        m_CurPos = 0;
        m_PushedCount = 0;
        for (auto& t : m_Buffer)
        {
            t = 0;
        }
    }

    T operator[](int index)
    {
        return m_Buffer[index];
    }
};

}

class LoudnessUtil
{
    friend LoudnessUtilTest;

private:
    struct Step
    {
        Step* next;
        uint64_t index;
        float square[ChannelCountMax];

        void Reset();
    };

private:
    int m_SampleRate;
    int m_ChannelCount;
    int m_SampleCountPerAudioFrame;
    int m_StepSampleCount;
    int m_GatingSampleCount;
    uint64_t m_CalcIndex;

    static const int StepCount = StepHistoryCount;
    Step m_Steps[StepCount];

    Step* freeStepList;
    Step* lastFreeStep;

    Step* filledStepList;
    Step* lastFilledStep;

    Step* fillingStep;
    int m_StepFilledCount;

    const FilterCoeff* m_ShelvCoeff;
    const FilterCoeff* m_HPFCoeff;
    BiquadFilterState m_ShelvState[ChannelCountMax];
    BiquadFilterState m_HighPassState[ChannelCountMax];

    ValueBuffer<float, MomentaryValueBufferCount> m_MomentaryPower;
    ValueBuffer<float, ShortTermValueBufferCount> m_ShortTermPower;
    ValueBuffer<float, DefaultGatedBlockBufferDuration> m_AbsGatedPower;
    ValueBuffer<float, DefaultGatedBlockBufferDuration> m_AbsGatedLoudness;

    float m_Integrated;
    uint64_t m_IntegratedCount;

protected:
    Step* PopStep(Step** list, Step** lastStep);
    void PushStep(Step** list, Step** lastStep, Step* step);
    float BiquadFilter(const FilterCoeff& coeff, BiquadFilterState& state, float sample);

    inline float CalcLKFS(float val)
    {
        return (val > 0.f) ?
            -0.691f + 10.f * log10f(val) :
            -1 * std::numeric_limits<float>::infinity();
    }

    template<typename T, int count>
    inline float CalcLKFS(const ValueBuffer<T, count>& buffer)
    {
        return (buffer.Filled()) ?
            CalcLKFS(buffer.Sum() / (m_StepSampleCount * buffer.Length())) :
            -1 * std::numeric_limits<float>::infinity();
    }

    void ResetValueBuffer();

protected:
    Step* GetFreeStep();
    Step* GetFilledStep();
    void PushFreeStep(Step* step);
    void PushFilledStep(Step* step);

    float KFilitering(float sample, int channel);
    int FillSteps(const int16_t* samples, int sampleCount);

    void UpdateValueBuffer(const Step* pStep);
    bool Calc(LoudnessValues* pOutValues);

    static int GetStepCount();

public:
    LoudnessUtil(int sampleRate, int channelCount, int sampleCount, int sampleFormat = 0 /* TODO: define format */);

    void Reset();
    void Update(const void* buffer, size_t bufferSize);
    int GetValues(LoudnessValues* pOutValues, int valueCount);
};
