﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include <cafe.h>

#include <nw/demo/demo_Main.h>
#include <nw/snd.h>
#include "common.fsid"

namespace
{
const char CONTENT_DIR[] = "/vol/content";
const char SOUND_ARC_PATH[] = "snd/common/common.bfsar";
const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

PADStatus s_Pad[PAD_MAX_CONTROLLERS]; // internal use only
u32 s_PadButton = 0;

FSClient* s_pFsClient;

nw::snd::FsSoundArchive     s_SoundArchive;
nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
nw::snd::SoundDataManager   s_SoundDataManager;
nw::snd::SoundHeap          s_SoundHeap;
nw::snd::SoundHandle        s_SoundHandle;

void* s_pMemoryForSoundSystem;
void* s_pMemoryForFsSoundArchive;
void* s_pMemoryForInfoBlock;
void* s_pMemoryForSoundDataManager;
void* s_pMemoryForSoundArchivePlayer;
void* s_pMemoryForSoundHeap;
void* s_pMemoryForStreamBuffer;

OSRendezvous    s_Rendezvous;

void OpenArchiveAndLoadData()
{
    // サウンドアーカイブの初期化
    {
        size_t size = s_SoundArchive.GetRequiredMemSize();
        s_pMemoryForFsSoundArchive = MEMAllocFromDefaultHeap(size);


        if ( ! s_SoundArchive.Open(
                s_pFsClient, SOUND_ARC_PATH, s_pMemoryForFsSoundArchive, size) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
    }

    // INFO ブロックのロード
    {
        size_t infoBlockSize = s_SoundArchive.GetHeaderSize();
        s_pMemoryForInfoBlock = MEMAllocFromDefaultHeapEx( infoBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
        if ( ! s_SoundArchive.LoadHeader( s_pMemoryForInfoBlock, infoBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load infoBlock(%s)", SOUND_ARC_PATH );
        }
    }

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = s_SoundDataManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundDataManager = MEMAllocFromDefaultHeapEx( setupSize, 4 );
        s_SoundDataManager.Initialize( &s_SoundArchive, s_pMemoryForSoundDataManager, setupSize );
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = s_SoundArchivePlayer.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundArchivePlayer = MEMAllocFromDefaultHeapEx( setupSize, 32 );
        size_t setupStrmBufferSize =
            s_SoundArchivePlayer.GetRequiredStreamBufferSize( &s_SoundArchive );
        s_pMemoryForStreamBuffer = MEMAllocFromDefaultHeapEx( setupStrmBufferSize, 8 );
        bool result = s_SoundArchivePlayer.Initialize(
                &s_SoundArchive,
                &s_SoundDataManager,
                s_pMemoryForSoundArchivePlayer, setupSize,
                s_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    // サウンドヒープの構築
    {
        s_pMemoryForSoundHeap = MEMAllocFromDefaultHeap( SOUND_HEAP_SIZE );
        bool result = s_SoundHeap.Create( s_pMemoryForSoundHeap, SOUND_HEAP_SIZE );
        NW_ASSERT( result );
    }

    if ( ! s_SoundDataManager.LoadData( SEQ_MARIOKART, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SEQ_MARIOKART) failed." );
    }
    if ( ! s_SoundDataManager.LoadData( SE_YOSHI, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
    }
}

int InitializeSoundFunc(int intArg, void *ptrArg)
{
    // AX 初期化
    {
        AXInit();
        AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_DSP );
    }

    // NW4F サウンドシステムの初期化
    {
        nw::snd::SoundSystem::SoundSystemParam param;
        size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
        s_pMemoryForSoundSystem = MEMAllocFromDefaultHeapEx( workMemSize, 4 );

        nw::snd::SoundSystem::Initialize(
                param,
                reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
                workMemSize );
    }

    OSWaitRendezvous(&s_Rendezvous, OS_WAIT_CORE_MAIN);

    return 0;
}

int FinalizeSoundFunc(int intArg, void *ptrArg)
{
    AXQuit();

    OSWaitRendezvous(&s_Rendezvous, OS_WAIT_CORE_MAIN);

    return 0;
}

} // namespace

int
NwDemoMain(int argc, char **argv)
{
    u32 gx2InitAttribs[] =
    {
        GX2_INIT_ATTRIB_ARGC,    (u32)argc,  // number of args
        GX2_INIT_ATTRIB_ARGV,    (u32)argv,  // command-line args
        GX2_INIT_ATTRIB_NULL                 // terminates the list
    };
    GX2Init(gx2InitAttribs);

    PADInit();

    // FS 初期化
    {
        FSInit();
        s_pFsClient = reinterpret_cast<FSClient*>(MEMAllocFromDefaultHeap(sizeof(FSClient)));
        NW_ASSERT_NOT_NULL(s_pFsClient);
        std::memset(s_pFsClient, 0x00, sizeof(FSClient));

        FSAddClient(s_pFsClient, FS_RET_NO_ERROR);
    }

    // ディレクトリ移動
    {
        FSCmdBlock* block = reinterpret_cast<FSCmdBlock*>(MEMAllocFromDefaultHeap(sizeof(FSCmdBlock)));
        NW_ASSERT_NOT_NULL(block);
        FSInitCmdBlock(block);

        FSStatus status = FSChangeDir(s_pFsClient, block, CONTENT_DIR, FS_RET_NO_ERROR);
        NW_ASSERT(status == FS_STATUS_OK);

        MEMFreeToDefaultHeap(block);
    }

    // サウンドの初期化 (別コアで呼び出します)
    {
        OSInitRendezvous(&s_Rendezvous);

        OSThread* thread = OSGetDefaultThread(2); // coreId
        OSRunThread(
            thread,
            InitializeSoundFunc,
            0,
            NULL);

        OSWaitRendezvous(&s_Rendezvous, OS_WAIT_CORE_SUB2);
    }

    OpenArchiveAndLoadData();

    NW_LOG("----------------------------------------\n");
    NW_LOG("NW4F Demo/snd/subCore\n");
    NW_LOG("----------------------------------------\n");
    NW_LOG("[A]     Start SEQ  (SEQ_MARIOKART)\n");
    NW_LOG("[X]     Start WSD  (SE_YOSHI)\n");
    NW_LOG("[Y]     Start STRM (STRM_MARIOKART)\n");
    NW_LOG("[B]     Stop Sound\n");
    NW_LOG("[HOME]  Exit Application\n");
    NW_LOG("----------------------------------------\n");

    while( true ) {
        GX2WaitForVsync();

        PADRead( s_Pad );

        u32 connectedBits = 0;
        for (int chan = 0; chan < PAD_MAX_CONTROLLERS; ++chan)
        {
            u32 padBit = PAD_CHAN0_BIT >> chan;

            switch (s_Pad[chan].err)
            {
              case PAD_ERR_NONE:
              case PAD_ERR_TRANSFER:
                connectedBits |= padBit;
                break;
              case PAD_ERR_NO_CONTROLLER:
                connectedBits &= ~padBit;
                break;
              case PAD_ERR_NOT_READY:
              default:
                break;
            }
        }

        PADClampCircle( s_Pad );

        u32 button = s_Pad[0].button;
        u32 trig = ( s_PadButton ^ button ) & button;
        s_PadButton = button;

        if ( trig & PAD_BUTTON_A ) {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SEQ_MARIOKART ).IsSuccess();
            NW_LOG("[SEQ] SEQ_MARIOKART ... (%d)\n", result);
        }
        if ( trig & PAD_BUTTON_X ) {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SE_YOSHI ).IsSuccess();
            NW_LOG("[WSD] SE_YOSHI ... (%d)\n", result);
        }
        if ( trig & PAD_BUTTON_Y ) {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, STRM_MARIOKART ).IsSuccess();
            NW_LOG("[STRM] STRM_MARIOKART ... (%d)\n", result);
        }
        if ( trig & PAD_BUTTON_B ) {
            s_SoundHandle.Stop( 0 );
        }
        if ( trig & PAD_BUTTON_START ) {
            break;
        }

        s_SoundArchivePlayer.Update();
    }

    s_SoundArchivePlayer.Finalize();
    s_SoundDataManager.Finalize();
    s_SoundArchive.Close();
    s_SoundHeap.Destroy();
    nw::snd::SoundSystem::Finalize();

    MEMFreeToDefaultHeap( s_pMemoryForSoundHeap );
    MEMFreeToDefaultHeap( s_pMemoryForSoundArchivePlayer );
    MEMFreeToDefaultHeap( s_pMemoryForStreamBuffer );
    MEMFreeToDefaultHeap( s_pMemoryForSoundDataManager );
    MEMFreeToDefaultHeap( s_pMemoryForInfoBlock );
    MEMFreeToDefaultHeap( s_pMemoryForFsSoundArchive );
    MEMFreeToDefaultHeap( s_pMemoryForSoundSystem );

    // サウンドの破棄 (初期化を呼び出しコアで呼び出します)
    {
        OSInitRendezvous(&s_Rendezvous);

        OSThread* thread = OSGetDefaultThread(2); // coreId
        OSRunThread(
            thread,
            FinalizeSoundFunc,
            0,
            NULL);

        OSWaitRendezvous(&s_Rendezvous, OS_WAIT_CORE_SUB2);
    }

    FSDelClient(s_pFsClient, FS_RET_NO_ERROR);
    MEMFreeToDefaultHeap(s_pFsClient);
    s_pFsClient = NULL;
    FSShutdown();
    GX2Shutdown();

    return 0;
}
