﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include <nw/snd.h>
#include <nw/snd/util/sndutil_FileReader.h>
#include <nw/demo.h>

#if defined(NW_PLATFORM_CAFE)
#include <cafe.h>
#elif defined( NW_USE_NINTENDO_SDK )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#include <nn/fs.h>
#else
#include <winext/cafe.h>
#include "Shlwapi.h"
#endif

namespace
{
#if defined(NW_USE_NINTENDO_SDK)
const char MOUNT_NAME[] = "content";
const char SOUND_ARC_PATH[] = "content:snd/common/common.bfsar";
const char EXTERNAL_FILE_ROOT[] = "content:snd/common/";
#else
const char CONTENT_DIR[] = "/vol/content";
const char SOUND_ARC_PATH[] = "snd/common/common.bfsar";
const char EXTERNAL_FILE_ROOT[] = "snd/common/";
#endif

nw::snd::MemorySoundArchive s_SoundArchive;
nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
nw::snd::SoundDataManager   s_SoundDataManager;
nw::snd::SoundHandle        s_SoundHandle;

void* s_pMemoryForSoundSystem;
void* s_pMemoryForMemorySoundArchive;
void* s_pMemoryForSoundDataManager;
void* s_pMemoryForSoundArchivePlayer;
void* s_pMemoryForStreamBuffer;
void* s_pMemoryForFsCommandBlockBufferPool;

#if defined(NW_PLATFORM_CAFE)
FSClient* s_pFsClient;
#endif

#if defined(NW_USE_NINTENDO_SDK)
void* Allocate(size_t size)
{
    nw::demo::DefaultAllocator allocator;
    return allocator.Alloc(size);
}

void Deallocate(void* memory, size_t size)
{
    NW_UNUSED_VARIABLE(size);

    nw::demo::DefaultAllocator allocator;
    allocator.Free(memory);
}
#endif

void InitializeFs()
{
    nw::demo::DefaultAllocator allocator;

#if defined(NW_PLATFORM_CAFE)
    // FS 初期化
    {
        FSInit();
        s_pFsClient = reinterpret_cast<FSClient*>(allocator.Alloc(sizeof(FSClient)));
        std::memset(s_pFsClient, 0x00, sizeof(FSClient));
        NW_ASSERT_NOT_NULL(s_pFsClient);

        FSAddClient(s_pFsClient, FS_RET_NO_ERROR);
    }

    // ディレクトリ移動
    {
        FSCmdBlock* block = reinterpret_cast<FSCmdBlock*>(allocator.Alloc(sizeof(FSCmdBlock)));
        NW_ASSERT_NOT_NULL(block);
        FSInitCmdBlock(block);

        FSStatus status = FSChangeDir(s_pFsClient, block, CONTENT_DIR, FS_RET_NO_ERROR);
        NW_ASSERT(status == FS_STATUS_OK);

        allocator.Free(block);
    }
#elif defined( NW_USE_NINTENDO_SDK )
    // nn::fs の初期化
    nn::fs::SetAllocator(Allocate, Deallocate);
    // CONTENT_DIRのマウント
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            nn::fs::MountHost(MOUNT_NAME, buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return;
        }
    }
#else
    // ディレクトリ移動
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            NW_LOG("buf(%s)\n", buf);
            SetCurrentDirectoryA(buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return;
        }
    }
#endif
}

void FinalizeFs()
{
#if defined( NW_PLATFORM_CAFE )
    nw::demo::DefaultAllocator allocator;
    FSDelClient(s_pFsClient, FS_RET_NO_ERROR);
    allocator.Free(s_pFsClient);
    s_pFsClient = NULL;
    FSShutdown();
#endif

#if defined ( NW_USE_NINTENDO_SDK )
    nn::fs::Unmount(MOUNT_NAME);
#endif
}

void InitializeNwSound()
{
    nw::demo::DefaultAllocator allocator;

    // サウンドシステムの初期化
    nw::snd::SoundSystem::SoundSystemParam param;
    size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
    s_pMemoryForSoundSystem = allocator.Alloc( workMemSize, 4 );

    nw::snd::SoundSystem::Initialize(
            param,
            reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
            workMemSize );

    // メモリーサウンドアーカイブの初期化
    {
        // サウンドアーカイブファイルをメモリに展開
        nw::snd::util::FileReader fileReader;

        nw::snd::util::FileReader::Result result = fileReader.Initialize(
#if defined(NW_PLATFORM_CAFE)
            s_pFsClient
#endif
        );
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

        result = fileReader.Open(SOUND_ARC_PATH);
        NW_ASSERTMSG( result == nw::snd::util::FileReader::RESULT_SUCCESS,
            "Failed to open file(%s) status(%d)\n", SOUND_ARC_PATH, result );

        u32 size = fileReader.GetSize();
        s_pMemoryForMemorySoundArchive = allocator.Alloc(size, 64);
        s32 readSize = 0;
        fileReader.Read(s_pMemoryForMemorySoundArchive, size, &readSize);
        NW_ASSERT(size == static_cast<u32>(readSize));

        result = fileReader.Close();
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);
        result = fileReader.Finalize();
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

        // MemorySoundArchive インスタンスの初期化
        s_SoundArchive.Initialize( s_pMemoryForMemorySoundArchive );
    }

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = s_SoundDataManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundDataManager = allocator.Alloc( setupSize, 4 );
        s_SoundDataManager.Initialize( &s_SoundArchive, s_pMemoryForSoundDataManager, setupSize );
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = s_SoundArchivePlayer.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundArchivePlayer = allocator.Alloc( setupSize, 32 );
        size_t setupStrmBufferSize =
            s_SoundArchivePlayer.GetRequiredStreamBufferSize( &s_SoundArchive );
        s_pMemoryForStreamBuffer = allocator.Alloc( setupStrmBufferSize, 8 );
        bool result = s_SoundArchivePlayer.Initialize(
                &s_SoundArchive,
                &s_SoundDataManager,
                s_pMemoryForSoundArchivePlayer, setupSize,
                s_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    // MemorySoundArchiveでストリームサウンドを使用するための処理です。
    // ストリームサウンドを使用しない場合はこの処理は不要です。
    {
        // 外部参照ファイルであるストリームサウンドバイナリを使用するために、
        // 外部参照ファイルのパスの基準となるディレクトリを設定します。
        // サウンドアーカイブバイナリ内部には相対パスでストリームサウンドバイナリのパスが埋め込まれているため、
        // 基本的にはサウンドアーカイブバイナリが存在しているディレクトリ
        // を設定することでストリームサウンドを再生できるようになります。
        const char* pathDir = EXTERNAL_FILE_ROOT;

#ifdef NW_PLATFORM_CAFE
        // サウンドアーカイブがFSを使用できるようにするためにFS関連オブジェクトを設定します。
        size_t poolSize = s_SoundArchive.GetRequiredFsCommandBlockPoolMemSize();
        s_pMemoryForFsCommandBlockBufferPool = allocator.Alloc( poolSize );
        s_SoundArchive.CreateFsCommandBlockBufferPool(s_pMemoryForFsCommandBlockBufferPool, poolSize);
        s_SoundArchive.SetFsClient(s_pFsClient);

        bool isFullPath = false;
        if ( pathDir[0] == '/' || pathDir[0] == '\\' )
        {
            isFullPath = true;
        }

        if ( isFullPath )
        {
            s_SoundArchive.SetExternalFileRoot( pathDir );
        }
        else
        {
            static const int FILE_PATH_MAX  = FS_MAX_FULLPATH_SIZE;
            char currentDir[FILE_PATH_MAX];

            // フルパスでカレントディレクトリを取得
            FSCmdBlock* block = reinterpret_cast<FSCmdBlock*>(allocator.Alloc(sizeof(FSCmdBlock)));
            NW_ASSERT_NOT_NULL(block);
            FSInitCmdBlock(block);

            FSStatus status = FSGetCwd(s_pFsClient, block, currentDir, FILE_PATH_MAX, FS_RET_NO_ERROR);
            NW_ASSERT(status == FS_STATUS_OK);
            allocator.Free(block);

            // currentDir <= currentDir + pathDir
            nw::ut::strncat( currentDir, FILE_PATH_MAX, pathDir, std::strlen(pathDir) );
            s_SoundArchive.SetExternalFileRoot( currentDir );
        }
#elif defined( NW_USE_NINTENDO_SDK )
        s_SoundArchive.SetExternalFileRoot( pathDir );
#else
        if ( ! PathIsRelativeA(pathDir) )
        {
            s_SoundArchive.SetExternalFileRoot( pathDir );
        }
        else
        {
            static const int FILE_PATH_MAX = 256;
            char currentDir[FILE_PATH_MAX];

            // フルパスでカレントディレクトリを取得
            DWORD result = GetCurrentDirectoryA(FILE_PATH_MAX, currentDir);
            NW_ASSERT( result != 0 && result < FILE_PATH_MAX );
            if (currentDir[result - 1] != '\\')
            {
                NW_ASSERT(result + 1 <= FILE_PATH_MAX);
                currentDir[result + 1] = '\0';
                currentDir[result] = '\\';
            }

            // currentDir <= currentDir + pathDir
            nw::ut::strncat( currentDir, FILE_PATH_MAX, pathDir, std::strlen(pathDir) );
            s_SoundArchive.SetExternalFileRoot( currentDir );
        }
#endif
    }
}

} // namespace

int
NwDemoMain(int argc, char **argv)
{
#if defined( NW_PLATFORM_CAFE )
    AXInit();
    AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_DSP );

#else
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);
#endif

    nw::demo::DefaultAllocator allocator;
    nw::demo::DemoSystem::CreateArg demoSystemArg;
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    demoSystemArg.waitVBlank = 0; // DemoSystem の SwapBuffer は呼ばないため、VBlank 待ちはしない。
#endif
    demoSystemArg.allocator = &allocator;
    nw::demo::DemoSystem* pDemo =
        new( allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) )
        nw::demo::DemoSystem( demoSystemArg );
    pDemo->Initialize();
    pDemo->InitializeGraphicsSystem();

    InitializeFs();
    InitializeNwSound();


    NW_LOG("----------------------------------------\n");
    NW_LOG("NW4F Demo/snd/memorySoundArchive\n");
    NW_LOG("----------------------------------------\n");
    NW_LOG("[A]     Start SEQ  (SEQ_MARIOKART)\n");
    NW_LOG("[X]     Start WSD  (SE_YOSHI)\n");
    NW_LOG("[Y]     Start STRM (STRM_MARIOKART)\n");
    NW_LOG("[B]     Stop Sound\n");
#if defined( NW_PLATFORM_CAFE )
    NW_LOG("[HOME]  Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_LOG("[S]     Exit Application\n");
#endif
    NW_LOG("----------------------------------------\n");

    while ( !pDemo->IsExiting() )
    {
#if defined( NW_PLATFORM_CAFE )
        GX2WaitForVsync();
#else
        pDemo->WaitForVBlank();
#endif

        pDemo->UpdatePad();
        nw::demo::Pad* pad =pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) ) {
            s_SoundHandle.Stop( 0 );
            bool result =
                s_SoundArchivePlayer.StartSound( &s_SoundHandle, "SEQ_MARIOKART" ).IsSuccess();
            NW_LOG("[SEQ] StartSound(SEQ_MARIOKART) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) ) {
            s_SoundHandle.Stop( 0 );
            bool result =
                s_SoundArchivePlayer.StartSound( &s_SoundHandle, "SE_YOSHI" ).IsSuccess();
            NW_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) ) {
            s_SoundHandle.Stop( 0 );
            bool result =
                s_SoundArchivePlayer.StartSound( &s_SoundHandle, "STRM_MARIOKART" ).IsSuccess();
            NW_LOG("[STRM] StartSound(STRM_MARIOKART) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) ) {
            s_SoundHandle.Stop( 0 );
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) ) {
            break;
        }

        s_SoundArchivePlayer.Update();
    }

    s_SoundArchivePlayer.Finalize();
#ifdef NW_PLATFORM_CAFE
    // 再生中に終了した場合には、SoundArchivePlayer::Finalize で使用されるため、
    // SoundArchivePlayer::Finalize() よりも後にで破棄処理を行います。
    s_SoundArchive.DestroyFsCommandBlockBufferPool();
#endif
    s_SoundDataManager.Finalize();
    s_SoundArchive.Finalize();
    nw::snd::SoundSystem::Finalize();

#ifdef NW_PLATFORM_CAFE
    allocator.Free( s_pMemoryForFsCommandBlockBufferPool );
#endif
    allocator.Free( s_pMemoryForSoundArchivePlayer );
    allocator.Free( s_pMemoryForStreamBuffer );
    allocator.Free( s_pMemoryForSoundDataManager );
    allocator.Free( s_pMemoryForMemorySoundArchive );
    allocator.Free( s_pMemoryForSoundSystem );

    FinalizeFs();

#if defined( NW_PLATFORM_CAFE )
    AXQuit();
#endif

    pDemo->FinalizeGraphicsSystem();
    pDemo->Finalize();
    allocator.Free( pDemo );

    return 0;
}
