﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include <nw/snd.h>
#include <nw/demo.h>
#include "common.fsid"

#if defined( NW_PLATFORM_CAFE )
#include <cafe.h>
#else
#include <winext/cafe.h>
#include <Windows.h>
#endif

namespace
{
const char CONTENT_DIR[] = "/vol/content";
const char SOUND_ARC_PATH[] = "snd/common/common.bfsar";
const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

enum StreamRegion {
    STREAM_REGION_INTRO = 0,
    STREAM_REGION_A = 1
};
u32 s_RegionList[] = {
    STREAM_REGION_INTRO,
    STREAM_REGION_A,
    STREAM_REGION_A
};
u32 s_RegionListIndex = 0;

nw::snd::StreamRegionCallbackResult StreamRegionCallback(
    nw::snd::StreamRegionCallbackParam* param, void* /*arg*/ )
{
    if ( s_RegionListIndex >= sizeof(s_RegionList)/sizeof(s_RegionList[0]) ) {
        NW_LOG("[%s] s_RegionListIndex(%d) => FINISH\n", __FUNCTION__, s_RegionListIndex);
        return nw::snd::STREAM_REGION_CALLBACK_FINISH;
    }

    param->regionNo = s_RegionList[s_RegionListIndex];
    NW_LOG("[%s] s_RegionListIndex(%d) => CONTINUE\n", __FUNCTION__, s_RegionListIndex);
    s_RegionListIndex++;
    return nw::snd::STREAM_REGION_CALLBACK_CONTINUE;
}

nw::snd::FsSoundArchive     s_SoundArchive;
nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
nw::snd::SoundDataManager   s_SoundDataManager;
nw::snd::SoundHeap          s_SoundHeap;
nw::snd::SoundHandle        s_SoundHandle;
nw::snd::SoundHandle        s_SoundHandleHold;

void* s_pMemoryForSoundSystem;
void* s_pMemoryForInfoBlock;
void* s_pMemoryForSoundDataManager;
void* s_pMemoryForSoundArchivePlayer;
void* s_pMemoryForSoundHeap;
void* s_pMemoryForStreamBuffer;

#if defined(NW_PLATFORM_CAFE)
FSClient* s_pFsClient;
void* s_pMemoryForFsSoundArchive;
#endif

void InitializeNwSound()
{
    // サウンドシステムの初期化
    nw::snd::SoundSystem::SoundSystemParam param;
    size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
    nw::demo::DefaultAllocator allocator;
    s_pMemoryForSoundSystem = allocator.Alloc( workMemSize, 4 );

    nw::snd::SoundSystem::Initialize(
            param,
            reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
            workMemSize );

    // サウンドアーカイブの初期化
    {
#if defined(NW_PLATFORM_CAFE)
        size_t size = s_SoundArchive.GetRequiredMemSize();
        s_pMemoryForFsSoundArchive = MEMAllocFromDefaultHeap(size);
        if ( ! s_SoundArchive.Open(
                    s_pFsClient, SOUND_ARC_PATH, s_pMemoryForFsSoundArchive, size) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
#else
        if ( ! s_SoundArchive.Open(SOUND_ARC_PATH) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
#endif
    }

    // INFO ブロックのロード
    {
        size_t infoBlockSize = s_SoundArchive.GetHeaderSize();
        s_pMemoryForInfoBlock = allocator.Alloc( infoBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
        if ( ! s_SoundArchive.LoadHeader( s_pMemoryForInfoBlock, infoBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load infoBlock(%s)", SOUND_ARC_PATH );
        }
    }

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = s_SoundDataManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundDataManager = allocator.Alloc( setupSize, 4 );
        s_SoundDataManager.Initialize( &s_SoundArchive, s_pMemoryForSoundDataManager, setupSize );
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = s_SoundArchivePlayer.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundArchivePlayer = allocator.Alloc( setupSize, 32 );
        size_t setupStrmBufferSize =
            s_SoundArchivePlayer.GetRequiredStreamBufferSize( &s_SoundArchive );
        s_pMemoryForStreamBuffer = allocator.Alloc( setupStrmBufferSize, 8 );
        bool result = s_SoundArchivePlayer.Initialize(
                &s_SoundArchive,
                &s_SoundDataManager,
                s_pMemoryForSoundArchivePlayer, setupSize,
                s_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    // サウンドヒープの構築
    {
        s_pMemoryForSoundHeap = allocator.Alloc( SOUND_HEAP_SIZE );
        bool result = s_SoundHeap.Create( s_pMemoryForSoundHeap, SOUND_HEAP_SIZE );
        NW_ASSERT( result );
    }
}

} // namespace

int
NwDemoMain(int argc, char **argv)
{
#if defined( NW_PLATFORM_CAFE )
    AXInit();
#else
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);
#endif

    nw::demo::DefaultAllocator allocator;
    nw::demo::DemoSystem::CreateArg demoSystemArg;
    demoSystemArg.allocator = &allocator;
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    demoSystemArg.waitVBlank = 0; // DemoSystem の SwapBuffer は呼ばないため、VBlank 待ちはしない。
#endif
    nw::demo::DemoSystem* pDemo =
        new( allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) )
        nw::demo::DemoSystem( demoSystemArg );
    pDemo->Initialize();
    pDemo->InitializeGraphicsSystem();

#if defined(NW_PLATFORM_CAFE)
    // FS 初期化
    {
        FSInit();
        s_pFsClient = reinterpret_cast<FSClient*>(MEMAllocFromDefaultHeap(sizeof(FSClient)));
        NW_ASSERT_NOT_NULL(s_pFsClient);
        std::memset(s_pFsClient, 0x00, sizeof(FSClient));

        FSAddClient(s_pFsClient, FS_RET_NO_ERROR);
    }

    // ディレクトリ移動
    {
        FSCmdBlock* block = reinterpret_cast<FSCmdBlock*>(MEMAllocFromDefaultHeap(sizeof(FSCmdBlock)));
        NW_ASSERT_NOT_NULL(block);
        FSInitCmdBlock(block);

        FSStatus status = FSChangeDir(s_pFsClient, block, CONTENT_DIR, FS_RET_NO_ERROR);
        NW_ASSERT(status == FS_STATUS_OK);

        MEMFreeToDefaultHeap(block);
    }
#else
    // ディレクトリ移動
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if (GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0)
        {
            NW_LOG("buf(%s)\n", buf);
            SetCurrentDirectoryA(buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return false;
        }
    }
#endif

    InitializeNwSound();

    NW_LOG("----------------------------------------\n");
    NW_LOG("NW4F Demo/snd/streamJump\n");
    NW_LOG("----------------------------------------\n");
    NW_LOG("[A]     StartSound STRM_REGION_JUMP\n");
    NW_LOG("[B]     Stop Sound\n");
#if defined( NW_PLATFORM_CAFE )
    NW_LOG("[HOME]  Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_LOG("[S]     Exit Application\n");
#endif
    NW_LOG("---------------------------------------\n");

    while ( !pDemo->IsExiting() )
    {
#if defined( NW_PLATFORM_CAFE )
        GX2WaitForVsync();
#else
        pDemo->WaitForVBlank();
#endif

        pDemo->UpdatePad();
        nw::demo::Pad* pad = pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) ) {
            s_SoundHandle.Stop( 0 );
            s_RegionListIndex = 0;
            nw::snd::SoundStartable::StartInfo info;
            info.enableFlag |= nw::snd::SoundStartable::StartInfo::ENABLE_STRM_SOUND_INFO;
            info.streamSoundInfo.regionCallback = StreamRegionCallback;
            info.streamSoundInfo.regionCallbackArg = NULL;
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, STRM_REGION_JUMP, &info ).IsSuccess();
            NW_LOG("StartSound(STRM_REGION_JUMP) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) ) {
            s_SoundHandle.Stop( 0 );
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) ) {
            break;
        }

        s_SoundArchivePlayer.Update();
    }

    s_SoundArchivePlayer.Finalize();
    s_SoundDataManager.Finalize();
    s_SoundArchive.Close();
    s_SoundHeap.Destroy();
    nw::snd::SoundSystem::Finalize();

    allocator.Free( s_pMemoryForSoundHeap );
    allocator.Free( s_pMemoryForSoundArchivePlayer );
    allocator.Free( s_pMemoryForStreamBuffer );
    allocator.Free( s_pMemoryForSoundDataManager );
    allocator.Free( s_pMemoryForInfoBlock );
    allocator.Free( s_pMemoryForSoundSystem );

#if defined( NW_PLATFORM_CAFE )
    allocator.Free( s_pMemoryForFsSoundArchive );
    FSDelClient(s_pFsClient, FS_RET_NO_ERROR);
    MEMFreeToDefaultHeap(s_pFsClient);
    s_pFsClient = NULL;
    FSShutdown();

    AXQuit();
#endif

    pDemo->FinalizeGraphicsSystem();
    pDemo->Finalize();
    allocator.Free( pDemo );

    return 0;
}
