﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include "graphics/SoundDemoUtility.h"
#include "graphics/ModelUtility.h"

#include <nw/snd.h>
#include <nw/snd/util/sndutil_FileReader.h>
#include "animSound.fsid"
#if defined( NW_PLATFORM_CAFE )
#include <cafe.h>
#else
#include <winext/cafe.h>
#include <Windows.h>
#endif

#if defined(NW_PLATFORM_CAFE)
    static const int BUFFER_MIN_ALIGNMENT = 64;
#else
    static const int BUFFER_MIN_ALIGNMENT = 32;
#endif

//------------------------------------------------------------------------------
namespace {
// アロケータ
#define HEAP_SIZE ( 32 * 1024 * 1024 )

// サウンドデモフレームワーク
SoundDemoSystem g_SoundDemo;

// グラフィックスモデル
SimpleModel g_Model;
void*       g_ModelResource = NULL;

// ファイルシステム
const char CONTENT_DIR[] = "/vol/content";

// サウンド
const char SOUND_ARC_PATH[] = "snd/animSound/animSound.bfsar";
const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

nw::snd::FsSoundArchive     s_SoundArchive;
nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
nw::snd::SoundDataManager   s_SoundDataManager;
nw::snd::SoundHeap          s_SoundHeap;
nw::snd::SoundHandle        s_SoundHandle;
nw::snd::SoundHandle        s_SoundHandleHold;

void* s_pMemoryForSoundSystem;
void* s_pMemoryForInfoBlock;
void* s_pMemoryForSoundDataManager;
void* s_pMemoryForSoundArchivePlayer;
void* s_pMemoryForSoundHeap;
void* s_pMemoryForStreamBuffer;

#if defined(NW_PLATFORM_CAFE)
FSClient* s_pFsClient;
void* s_pMemoryForFsSoundArchive;
#endif

nw::snd::Sound3DManager s_Sound3DManager;
nw::snd::Sound3DListener s_Sound3DListener;
nw::snd::Sound3DActor s_Sound3DActor;
void* s_pMemoryFor3dManager;

const char* ANIMSOUND_BIN_PATH =  "snd/animSound/human_run.bfasd";
nw::snd::AnimSound          s_AnimSound( s_Sound3DActor );
void* s_pMemoryForStringBlock;
void* s_pMemoryForAnimSound;

}
//==============================================================================
// 初期化、終了処理
//==============================================================================

//------------------------------------------------------------------------------
// ファイルシステム
bool
InitializeFs()
{
#if defined(NW_PLATFORM_CAFE)
    // FS 初期化
    {
        FSInit();
        s_pFsClient = reinterpret_cast<FSClient*>(MEMAllocFromDefaultHeap(sizeof(FSClient)));
        NW_ASSERT_NOT_NULL(s_pFsClient);
        std::memset(s_pFsClient, 0x00, sizeof(FSClient));

        FSAddClient(s_pFsClient, FS_RET_NO_ERROR);
    }

    // ディレクトリ移動
    {
        FSCmdBlock* block = reinterpret_cast<FSCmdBlock*>(MEMAllocFromDefaultHeap(sizeof(FSCmdBlock)));
        NW_ASSERT_NOT_NULL(block);
        FSInitCmdBlock(block);

        FSStatus status = FSChangeDir(s_pFsClient, block, CONTENT_DIR, FS_RET_NO_ERROR);
        NW_ASSERT(status == FS_STATUS_OK);

        MEMFreeToDefaultHeap(block);
    }
#else
    // ディレクトリ移動
    {
        static const int cFileNameLen = 512;
        char buf[cFileNameLen];
        if ( GetEnvironmentVariableA("CAFE_CONTENT_DIR", buf, cFileNameLen) > 0 )
        {
            NW_LOG("buf(%s)\n", buf);
            SetCurrentDirectoryA(buf);
        }
        else
        {
            NW_LOG("cannot read CAFE_CONTENT_DIR\n");
            return false;
        }
    }
#endif

    return true;
}

void
FinalizeFs()
{
#if defined( NW_PLATFORM_CAFE )
    FSDelClient(s_pFsClient, FS_RET_NO_ERROR);
    MEMFreeToDefaultHeap(s_pFsClient);
    s_pFsClient = NULL;
    FSShutdown();
#endif
}

//------------------------------------------------------------------------------
// グラフィックス
void
InitializeGraphcis(nw::ut::IAllocator& allocator)
{
    InitializeG3dShaderResource(&allocator);

    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();

    // モデル
    {
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path        = "g3d/demo/human.bfres";
        loadArg.allocator   = &allocator;
        loadArg.alignment   = 8 * 1024;
        g_ModelResource = fileSystem->Load( loadArg );
        NW_ASSERT( loadArg.readSize > 0 );

        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        {
            g_Model.Initialize( g_ModelResource, &allocator, "human_run" );
        }
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
    }
}

void
FinalizeGraphics(nw::ut::IAllocator& allocator)
{
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        g_Model.Finalize(&allocator);
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    FinalizeG3dShaderResource(&allocator);
}

//------------------------------------------------------------------------------
// サウンド
void
InitializeSdkSound()
{
#if defined( NW_PLATFORM_CAFE )
    AXInit();
    AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_DSP );
#endif
}

void
FinalizeSdkSound()
{
#if defined( NW_PLATFORM_CAFE )
    AXQuit();
#endif
}

void
InitializeNwSound(nw::ut::IAllocator& allocator)
{
    // サウンドシステムの初期化
    {
        nw::snd::SoundSystem::SoundSystemParam param;
        size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
        s_pMemoryForSoundSystem = allocator.Alloc( workMemSize, 4 );

        nw::snd::SoundSystem::Initialize(
            param,
            reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
            workMemSize );
    }

    // サウンドアーカイブの初期化
    {
#if defined(NW_PLATFORM_CAFE)
        size_t size = s_SoundArchive.GetRequiredMemSize();
        s_pMemoryForFsSoundArchive = MEMAllocFromDefaultHeap(size);
        if ( ! s_SoundArchive.Open(
                    s_pFsClient, SOUND_ARC_PATH, s_pMemoryForFsSoundArchive, size) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
#else
        if ( ! s_SoundArchive.Open(SOUND_ARC_PATH) )
        {
            NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", SOUND_ARC_PATH );
        }
#endif
    }

    // INFO ブロックのロード
    {
        size_t infoBlockSize = s_SoundArchive.GetHeaderSize();
        s_pMemoryForInfoBlock = allocator.Alloc( infoBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
        if ( ! s_SoundArchive.LoadHeader( s_pMemoryForInfoBlock, infoBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load infoBlock(%s)", SOUND_ARC_PATH );
        }
    }

    // サウンドデータマネージャーの初期化
    {
        size_t setupSize = s_SoundDataManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundDataManager = allocator.Alloc( setupSize, 4 );
        s_SoundDataManager.Initialize( &s_SoundArchive, s_pMemoryForSoundDataManager, setupSize );
    }

    // サウンドアーカイブプレイヤーの初期化
    {
        size_t setupSize = s_SoundArchivePlayer.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryForSoundArchivePlayer = allocator.Alloc( setupSize, 32 );
        size_t setupStrmBufferSize =
            s_SoundArchivePlayer.GetRequiredStreamBufferSize( &s_SoundArchive );
        s_pMemoryForStreamBuffer = allocator.Alloc( setupStrmBufferSize, 8 );
        bool result = s_SoundArchivePlayer.Initialize(
                &s_SoundArchive,
                &s_SoundDataManager,
                s_pMemoryForSoundArchivePlayer, setupSize,
                s_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    // サウンドヒープの構築
    {
        s_pMemoryForSoundHeap = allocator.Alloc( SOUND_HEAP_SIZE );
        bool result = s_SoundHeap.Create( s_pMemoryForSoundHeap, SOUND_HEAP_SIZE );
        NW_ASSERT( result );
    }

    if ( ! s_SoundDataManager.LoadData( WSD_FOOTSTEP0, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(WSD_FOOTSTEP0) failed." );
    }
    if ( ! s_SoundDataManager.LoadData( WSD_FOOTSTEP1, &s_SoundHeap ) )
    {
        NW_ASSERTMSG( false, "LoadData(WSD_FOOTSTEP1) failed." );
    }

    // STRING ブロックのロード
    {
        u32 stringBlockSize = s_SoundArchive.GetLabelStringDataSize();
        s_pMemoryForStringBlock = allocator.Alloc( stringBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
        if ( ! s_SoundArchive.LoadLabelStringData( s_pMemoryForStringBlock, stringBlockSize ) )
        {
            NW_ASSERTMSG( 0, "cannot load stringBlock(%s)", SOUND_ARC_PATH );
        }
    }

    // 3D サウンドマネージャーの初期化
    {
        size_t setupSize = s_Sound3DManager.GetRequiredMemSize( &s_SoundArchive );
        s_pMemoryFor3dManager = allocator.Alloc( setupSize );
        s_Sound3DManager.Initialize( &s_SoundArchive, s_pMemoryFor3dManager, setupSize );
        s_Sound3DManager.SetMaxPriorityReduction( 32 );
        s_Sound3DManager.SetSonicVelocity( 340.0f / 60 );
    }

    // 3D サウンドリスナーの初期化
    {
        s_Sound3DManager.AddListener( &s_Sound3DListener );
        s_Sound3DListener.SetMaxVolumeDistance( 5.0f );
        s_Sound3DListener.SetUnitDistance( 5.0f );
        s_Sound3DListener.SetInteriorSize( 5.0f );
    }

    // 3D サウンドアクターの初期化
    {
        s_Sound3DActor.Initialize( s_SoundArchivePlayer, s_Sound3DManager );
        s_Sound3DActor.SetPosition( nw::math::VEC3::Zero() );
    }

    // AnimSound の初期化
    {
        nw::snd::util::FileReader fileReader;

        nw::snd::util::FileReader::Result result = fileReader.Initialize(
#if defined(NW_PLATFORM_CAFE)
            s_pFsClient
#endif
        );
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

        result = fileReader.Open(ANIMSOUND_BIN_PATH);
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

        u32 size = fileReader.GetSize();
        s_pMemoryForAnimSound = allocator.Alloc(size, BUFFER_MIN_ALIGNMENT);
        s32 readSize = 0;
        fileReader.Read(s_pMemoryForAnimSound, size, &readSize);
        NW_ASSERT(size == static_cast<u32>(readSize));

        result = fileReader.Close();
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);
        result = fileReader.Finalize();
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

        // アニメサウンドファイルの内容確認
        nw::snd::AnimSound::Dump( s_pMemoryForAnimSound );

        // AnimSound インスタンスの初期化
        s_AnimSound.Initialize( s_pMemoryForAnimSound );
        s_AnimSound.ConvertSoundId( s_SoundArchive );
    }

    s_Sound3DListener.SetMatrix( nw::math::MTX34::Identity() );
}

void
FinalizeNwSound(nw::ut::IAllocator& allocator)
{
    s_AnimSound.Finalize();
    s_SoundArchivePlayer.Finalize();
    s_SoundDataManager.Finalize();
    s_SoundArchive.Close();
    s_SoundHeap.Destroy();
    nw::snd::SoundSystem::Finalize();

    allocator.Free( s_pMemoryForAnimSound );
    allocator.Free( s_pMemoryForStringBlock );
    allocator.Free( s_pMemoryFor3dManager );
    allocator.Free( s_pMemoryForSoundHeap );
    allocator.Free( s_pMemoryForStreamBuffer );
    allocator.Free( s_pMemoryForSoundArchivePlayer );
    allocator.Free( s_pMemoryForSoundDataManager );
    allocator.Free( s_pMemoryForInfoBlock );
#if defined(NW_PLATFORM_CAFE)
    allocator.Free( s_pMemoryForFsSoundArchive );
#endif
    allocator.Free( s_pMemoryForSoundSystem );
}

//---------------------------------------------------------------------------
void PrintUsage()
{
    NW_LOG("----------------------------------------\n");
    NW_LOG("NW4F Demo/snd/animSound\n");
    NW_LOG("---------------------------------------\n");
    NW_LOG("[X] Reset Camera Position\n");
    NW_LOG("[LEFT/RIGHT] Move Camera Position.z\n");
    NW_LOG("[UP/DOWN]    Move Camera Position.x\n");
    NW_LOG("---------------------------------------\n");
}

//---------------------------------------------------------------------------
void
DrawModel()
{
    g_SoundDemo.UpdateViewBlock(g_Model);
    nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_BLOCK);
    g_Model.Draw();
    nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_REGISTER);
}

//---------------------------------------------------------------------------
void
ProcScene()
{
    // フレームワークの更新
    g_SoundDemo.ProcFrame();

    // モデルの更新
    g_Model.Calc();

    // サウンドの更新
    if (!g_Model.IsEndFrame())
    {
        s_AnimSound.UpdateFrame( g_Model.GetAnimationFrame() );
    }
    s_Sound3DListener.SetMatrix( g_SoundDemo.GetCameraMatrix() );
    s_SoundArchivePlayer.Update();
}

//---------------------------------------------------------------------------
void
DrawScene()
{
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
#if defined(USE_DRC)
        g_SoundDemo.GetDemoSystem().BeginDrawTV();
        {
#endif
            g_SoundDemo.GetDemoSystem().ClearFrameBuffers();

            // モデルの描画処理
            DrawModel();
#if defined(USE_DRC)
        }
        g_SoundDemo.GetDemoSystem().EndDrawTV();

        // DRC 描画
        g_SoundDemo.GetDemoSystem().BeginDrawDRC();
        {
            g_SoundDemo.GetDemoSystem().ClearFrameBuffers();
            f32 width  = static_cast<f32>( g_SoundDemo.GetDemoSystem().GetDRCWidth() );
            f32 height = static_cast<f32>( g_SoundDemo.GetDemoSystem().GetDRCHeight() );
            g_SoundDemo.GetDemoSystem().SetViewport( 0.f, 0.f, width, height, 0.f, 1.f );
            g_SoundDemo.GetDemoSystem().SetScissor( 0.f, 0.f, width, height );

            // モデルの描画処理
            DrawModel();
        }
        g_SoundDemo.GetDemoSystem().EndDrawDRC();
#endif
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
}

//---------------------------------------------------------------------------
//! @brief        main関数です。
//---------------------------------------------------------------------------
int
NwDemoMain(int /*argc*/, char** /*argv*/)
{
    // アロケーターの初期化
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    void* addr = malloc( HEAP_SIZE );
#else
    void* addr = MEMAllocFromDefaultHeap( HEAP_SIZE );
#endif
    nw::ut::MemoryAllocator allocator;
    allocator.Initialize( addr, HEAP_SIZE );

    if (!InitializeFs())
    {
        return -1;
    }

    // デモ初期化処理
    g_SoundDemo.Initialize(allocator);

    InitializeGraphcis(allocator);
    InitializeSdkSound();
    InitializeNwSound(allocator);

    PrintUsage();

    // メインループ
    while ( !g_SoundDemo.GetDemoSystem().IsExiting() )
    {
        // 更新処理
        ProcScene();

        // 描画処理
        DrawScene();

        // バッファのスワップ
        g_SoundDemo.GetDemoSystem().SwapBuffer();

        // フレームの終了処理
        g_SoundDemo.GetDemoSystem().EndFrame();

        //  VBlank 待ち
        g_SoundDemo.GetDemoSystem().WaitForVBlank();
    }

    FinalizeNwSound(allocator);
    FinalizeSdkSound();
    FinalizeGraphics(allocator);

    // デモ終了処理
    g_SoundDemo.Finalize();

    FinalizeFs();

    // アロケータ終了処理
    allocator.Finalize();
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    free( addr );
#else
    MEMFreeToDefaultHeap( addr );
#endif

    return 0;
}
