﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef MODEL_UTILITY_H_
#define MODEL_UTILITY_H_

#include <nw/g3d.h>
#include <nw/ut.h>
#include <nw/dev.h>


class SimpleShader;


//------------------------------------------------------------------------------
// ※ グラフィックスに関する処理は、g3dのデモを参考にしてください。
//------------------------------------------------------------------------------


//------------------------------------------------------------------------------
//　　　シェーダリソースを初期化処理を行います。
//------------------------------------------------------------------------------
void InitializeG3dShaderResource( nw::ut::IAllocator* allocator );


//------------------------------------------------------------------------------
//　　　シェーダリソースを取得します。
//------------------------------------------------------------------------------
void GetG3dShaderResource( nw::g3d::GfxShader::SetupArg* setupArg );


//------------------------------------------------------------------------------
//　　　シェーダリソースの終了処理を行います。
//------------------------------------------------------------------------------
void FinalizeG3dShaderResource( nw::ut::IAllocator* allocator );


//------------------------------------------------------------------------------
//　　　モデルの生成を行います。
//------------------------------------------------------------------------------
nw::g3d::ModelObj* BuildModel( nw::ut::IAllocator* allocator, nw::g3d::res::ResModel* rMdl );


//------------------------------------------------------------------------------
//　　　モデルの破棄を行います。
//------------------------------------------------------------------------------
void DestroyModel( nw::ut::IAllocator* allocator, nw::g3d::ModelObj* mdl );


//------------------------------------------------------------------------------
//      スケルタルアニメの生成を行います。
//------------------------------------------------------------------------------
nw::g3d::SkeletalAnimObj* BuildSklAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResSkeletalAnim* rSklAnim, int maxSklCount );


//------------------------------------------------------------------------------
//　    パラメータアニメの生成を行います。
//------------------------------------------------------------------------------
nw::g3d::ShaderParamAnimObj* BuildParamAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResShaderParamAnim* rParamAnim, int maxMatCount );


//------------------------------------------------------------------------------
//　    テクスチャアニメの生成を行います。
//------------------------------------------------------------------------------
nw::g3d::TexPatternAnimObj* BuildTexAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResTexPatternAnim* rTexAnim, int maxMatCount );


//------------------------------------------------------------------------------
//　    ビジビリティアニメの生成を行います。
//------------------------------------------------------------------------------
nw::g3d::VisibilityAnimObj* BuildVisAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResVisibilityAnim* rVisAnim, int maxCount );


//------------------------------------------------------------------------------
//　    シェイプアニメの生成を行います。
//------------------------------------------------------------------------------
nw::g3d::ShapeAnimObj* BuildShapeAnim( nw::ut::IAllocator* allocator, nw::g3d::res::ResShapeAnim* rShapeAnim, int maxShapeCount );


//------------------------------------------------------------------------------
//　    アニメの破棄を行います。
//------------------------------------------------------------------------------
void DestroyModelAnim( nw::ut::IAllocator* allocator, nw::g3d::ModelAnimObj* anim );


//------------------------------------------------------------------------------
//　    フェッチシェーダの生成を行います。
//------------------------------------------------------------------------------
int BuildFetchShader( nw::ut::IAllocator* allocator, const nw::g3d::ModelObj* mdl, nw::g3d::GfxFetchShader** pFetchShaderArray );


//------------------------------------------------------------------------------
//　    フェッチシェーダの破棄を行います。
//------------------------------------------------------------------------------
void DestroyFetchShader( nw::ut::IAllocator* allocator, nw::g3d::GfxFetchShader** pFetchShaderArray, int numShader );


//------------------------------------------------------------------------------
//　    モデルにシェーダを紐づけします。
//------------------------------------------------------------------------------
int BindShader( nw::ut::IAllocator* allocator, const SimpleShader& shader, nw::g3d::ModelObj* model, const char** blockId );


//------------------------------------------------------------------------------
//　    シェーダの破棄を行います。
//------------------------------------------------------------------------------
void DestroyShader( nw::ut::IAllocator* allocator, SimpleShader& shader );




//------------------------------------------------------------------------------
//　　　シンプルシェーダクラス
//------------------------------------------------------------------------------
class SimpleShader : public nw::g3d::GfxShader
{
public:
    enum UniformBlockType
    {
        MATERIAL_BLOCK,
        SHAPE_BLOCK,
        SKELETON_BLOCK,
        VIEW_BLOCK,
        USER_SHAPE_BLOCK
    };

    struct ViewBlock
    {
        nw::g3d::Mtx44 projMtx;
        nw::g3d::Mtx34 cameraMtx;
    };

    enum
    {
        // シェーダで使用する各項目の最大数を設定します。
        // 必要数を事前に決定しておくことでロケーションを記録するメモリを節約することも可能です。
        NUM_SHADER_ATTRIB   = 8,
        NUM_SHADER_SAMPLER  = 8,
        NUM_SHADER_BLOCK    = 5,

        // モデルに使用するテクスチャのアライメントの最大値を設定します。
        // リソースファイルごとのアライメントを事前に取得しておくことで、
        // 必要なメモリのアライメントを最小限に抑えてメモリを節約することも可能です。
        TEXTURE_ALIGNMENT   = 8192
    };

    struct Path
    {
        const char* gsh;
        const char* vertexShader;
        const char* geometryShader;
        const char* fragmentShader;
    };

    struct Location
    {
        s8 vertex;
        s8 geometry;
        s8 fragment;
        u8 reserved;
    };


    //---------------------------------------------------------------------------
    //! @brief      サンプラ―ロケーションの初期化を行います。
    //---------------------------------------------------------------------------
    void InitSamplerLocations(int numSampler, const char* name[]);

    //---------------------------------------------------------------------------
    //! @brief      ブロックロケーションの初期化を行います。
    //---------------------------------------------------------------------------
    void InitBlockLocations(int numBlock, const char* name[]);

    //---------------------------------------------------------------------------
    //! @brief      ビューブロックの初期化を行います。
    //---------------------------------------------------------------------------
    void InitViewBlock( nw::ut::IAllocator* allocator );

    //---------------------------------------------------------------------------
    //! @brief      ビューブロックの終了処理を行います。
    //---------------------------------------------------------------------------
    void FinalizeViewBlock( nw::ut::IAllocator* allocator );

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    int GetSamplerCount() const { return mNumSampler; }

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    int GetBlockCount() const { return mNumBlock; }

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    nw::g3d::GfxBuffer& GetViewBlock() { return mViewBlock; };

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    const Location& GetSamplerLocation(int index) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, mNumSampler);
        return mSamplerLocation[index];
    }

    //---------------------------------------------------------------------------
    //! @brief
    //---------------------------------------------------------------------------
    const Location& GetBlockLocation(int index) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, mNumBlock);
        return mBlockLocation[index];
    }

private:
    u8                  mNumSampler;
    u8                  mNumBlock;
    u16                 reserved;
    Location            mSamplerLocation[NUM_SHADER_SAMPLER];
    Location            mBlockLocation[NUM_SHADER_BLOCK];
    nw::g3d::GfxBuffer  mViewBlock;
};




//------------------------------------------------------------------------------
//　　　シンプルモデルクラス
//------------------------------------------------------------------------------
class SimpleModel
{
public:
    enum
    {
        MODEL_ANIM_MAX = 8
    };

    //---------------------------------------------------------------------------
    //! @brief     コンストラクタ
    //---------------------------------------------------------------------------
    SimpleModel();


    //---------------------------------------------------------------------------
    //! @brief     デストラクタ
    //---------------------------------------------------------------------------
    ~SimpleModel();


    //---------------------------------------------------------------------------
    //! @brief     初期化処理を行います。
    //---------------------------------------------------------------------------
    bool Initialize( void* modelResource, nw::ut::IAllocator* allocator, const char* animName );


    //---------------------------------------------------------------------------
    //! @brief     終了処理を行います。
    //---------------------------------------------------------------------------
    void Finalize( nw::ut::IAllocator* allocator );


    //---------------------------------------------------------------------------
    //! @brief     計算処理を行います。
    //---------------------------------------------------------------------------
    void Calc();


    //---------------------------------------------------------------------------
    //! @brief     ブロックの更新処理を行います。
    //---------------------------------------------------------------------------
    void CalcBlock( const nw::g3d::math::Mtx34* viewMtx, const nw::g3d::math::Mtx44* projMtx );


    //---------------------------------------------------------------------------
    //! @brief     描画処理を行います。
    //---------------------------------------------------------------------------
    void Draw();


    //---------------------------------------------------------------------------
    //! @brief     描画マトリクスを設定します。
    //---------------------------------------------------------------------------
    void SetDrawMatrix( nw::g3d::Mtx34& drawMatrix );

    //---------------------------------------------------------------------------
    //! @brief     現在のフレームが最終フレームであるかどうかを調べます。
    //---------------------------------------------------------------------------
    bool IsEndFrame() const;

    //---------------------------------------------------------------------------
    //! @brief     現在のフレームを取得します。
    //---------------------------------------------------------------------------
    f32 GetAnimationFrame() const;



    //---------------------------------------------------------------------------
    //! @brief        モデル名を取得します。
    //---------------------------------------------------------------------------
    const char* GetModelName() const
    {
        return mModelObj->GetResource()->GetName();
    }


    //---------------------------------------------------------------------------
    //! @brief        ボーン数を取得します。
    //---------------------------------------------------------------------------
    s32 GetBoneNum() const
    {
        nw::g3d::SkeletonObj* skeleton = mModelObj->GetSkeleton();
        if ( !skeleton ) return 0;
        return skeleton->GetBoneCount();
    }


    //---------------------------------------------------------------------------
    //! @brief        ボーン名を取得します。
    //---------------------------------------------------------------------------
    const char* GetBoneName( u32 idx ) const
    {
        nw::g3d::SkeletonObj* skeleton = mModelObj->GetSkeleton();
        if ( !skeleton ) return NULL;
        nw::g3d::ResBone* bone = skeleton->GetBone(idx);
        if ( !bone ) return NULL;

        return bone->GetName();
    }


    //---------------------------------------------------------------------------
    //! @brief        ボーンのワールドマトリックスを取得します。
    //---------------------------------------------------------------------------
    const nw::g3d::math::Mtx34* GetBoneWorldMatrix(int index)
    {
        nw::g3d::SkeletonObj* skeleton = mModelObj->GetSkeleton();
        int boneCount = skeleton->GetBoneCount();

        if (boneCount <= index)
            return NULL;

        return &skeleton->GetWorldMtxArray()[index];
    }



private:
    nw::g3d::ModelObj*          mModelObj;
    nw::g3d::ModelAnimObj*      mModelAnimObj;

    int                         mNumFechShader;
    nw::g3d::GfxFetchShader*    mFetchShaderArray;

    int                         mNumShapeBlock;
    nw::g3d::GfxBuffer*         mShapeBlockArray;

    SimpleShader                mShader;

    nw::g3d::Mtx34              mDrawMatrix;
};


#endif // ifndef MODEL_UTILITY_H_
