﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nw/demo.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "finalMix";

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    const s32 WAVE_AMP_MAX = 8192;

    class MyFinalMixCallback : public nw::snd::FinalMixCallback
    {
    public:
        void OnFinalMix(nw::snd::OutputDevice device, const nw::snd::FinalMixData* data)
        {
            NW_ASSERT_NOT_NULL(data);
            if (device == nw::snd::OUTPUT_DEVICE_MAIN)
            {
                OnMainOut(data);
            }
            else
            {
                // 現時点ではDRC2台に対応していません。
                // 今後対応された場合には、2台ともこの処理を行うようになるため、
                // どちらのDRCからも矩形波の音が再生されるようになります。
                OnDrcOut(data);
            }
        }

    private:
        // TV出力に対して、1 サウンドフレームを 1 周期とした約 333 Hzのサイン波を合成します。
        void OnMainOut(const nw::snd::FinalMixData* data)
        {
            s32 sampleCount = data->sampleCount;
            u32 step = static_cast<u32>(0x100000000 / sampleCount);
            for (s32 i = 0; i < sampleCount; ++i)
            {
                f32 factor = nw::math::SinIdx(i * step);

                if (data->frontLeft != NULL)
                {
                    data->frontLeft[i] += static_cast<s32>(WAVE_AMP_MAX * factor);
                }
                if (data->frontRight != NULL)
                {
                    data->frontRight[i] += static_cast<s32>(WAVE_AMP_MAX * factor);
                }
            }
        }

        // DRC出力に対して、1 サウンドフレームを1 周期とした約 333 Hzの矩形波を合成します。
        void OnDrcOut(const nw::snd::FinalMixData* data)
        {
            s32 sampleCount = data->sampleCount;
            for (s32 i = 0; i < sampleCount; ++i)
            {
                f32 factor = 0.0f;
                if (i > sampleCount * 0.5f)
                {
                    factor = 0.5f;
                }

                if (data->frontLeft != NULL)
                {
                    data->frontLeft[i] += static_cast<s32>(WAVE_AMP_MAX * factor);
                }
                if (data->frontRight != NULL)
                {
                    data->frontRight[i] += static_cast<s32>(WAVE_AMP_MAX * factor);
                }
            }
        }
    };

    MyFinalMixCallback s_FinalMixCallback;
    bool s_IsRegistered = false;

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A]     Register/Unregister FinalMixCallback\n");
#if defined( NW_PLATFORM_CAFE )
        NW_LOG("[HOME]  Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_LOG("[S]     Exit Application\n");
#endif
        NW_LOG("---------------------------------------\n");
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::Pad* pad = pDemo->GetPad();

        // Register/Unregister FinalMixCallback
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            if (!s_IsRegistered)
            {
                NW_LOG("Register FinalMixCallback\n");
                nw::snd::SoundSystem::AppendFinalMixCallback(&s_FinalMixCallback);
                s_IsRegistered = true;
            }
            else
            {
                NW_LOG("Unregister FinalMixCallback\n");
                nw::snd::SoundSystem::EraseFinalMixCallback(&s_FinalMixCallback);
                s_IsRegistered = false;
            }
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) ) {
            return false;
        }

        return true;
    }
}

namespace snddemo
{

    void FinalMixDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        snddemo::InitializeSoundSystem(allocator);

        PrintUsage();

        s_IsRegistered = false;

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        if (s_IsRegistered)
        {
            nw::snd::SoundSystem::EraseFinalMixCallback(&s_FinalMixCallback);
        }

        snddemo::FinalizeSoundSystem(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();
    }

}
