﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nw/demo.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "filter";
    const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::FsSoundArchive     s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;
    nw::snd::SoundHeap          s_SoundHeap;

    nw::snd::SoundHandle s_SoundHandle;

    int s_FilterType;
    f32 s_LpfValue;
    f32 s_BqfValue;
    const f32 FILTER_DELTA = 0.1f;
    const f32 FILTER_VALUE_EPSILON = 0.5f * FILTER_DELTA;
    const f32 LPF_VALUE_MIN = -1.0f;
    const f32 LPF_VALUE_MAX = 0.0f;
    const f32 BQF_VALUE_MIN = 0.0f;
    const f32 BQF_VALUE_MAX = 1.0f;

    enum FilterType
    {
        FILTER_TYPE_NONE,
        FILTER_TYPE_HANDLE_LPF,            // 単極フィルタ
        FILTER_TYPE_HANDLE_BIQUAD_LPF,     // 以下、双極フィルタ
        FILTER_TYPE_HANDLE_BIQUAD_HPF,
        FILTER_TYPE_HANDLE_BIQUAD_BPF_512,
        FILTER_TYPE_HANDLE_BIQUAD_BPF_1024,
        FILTER_TYPE_HANDLE_BIQUAD_BPF_2048,
        FILTER_TYPE_PLAYER_LPF,            // 単極フィルタ
        FILTER_TYPE_PLAYER_BIQUAD_LPF,     // 以下、双極フィルタ
        FILTER_TYPE_PLAYER_BIQUAD_HPF,
        FILTER_TYPE_PLAYER_BIQUAD_BPF_512,
        FILTER_TYPE_PLAYER_BIQUAD_BPF_1024,
        FILTER_TYPE_PLAYER_BIQUAD_BPF_2048,
        FILTER_TYPE_MAX = FILTER_TYPE_PLAYER_BIQUAD_BPF_2048
    };

    const char* GetFilterTypeString( int type )
    {
        static const char* FILTER_TYPE_STRING[] =
        {
            "NONE",
            "HANDLE_LPF",
            "HANDLE_BIQUAD_LPF",
            "HANDLE_BIQUAD_HPF",
            "HANDLE_BIQUAD_BPF_512",
            "HANDLE_BIQUAD_BPF_1024",
            "HANDLE_BIQUAD_BPF_2048",
            "PLAYER_LPF",
            "PLAYER_BIQUAD_LPF",
            "PLAYER_BIQUAD_HPF",
            "PLAYER_BIQUAD_BPF_512",
            "PLAYER_BIQUAD_BPF_1024",
            "PLAYER_BIQUAD_BPF_2048",
        };
        if ( type > FILTER_TYPE_MAX || type < 0 )
        {
            return "INVALID";
        }
        return FILTER_TYPE_STRING[ type ];
    }

    void SetFilter(nw::snd::SoundArchivePlayer& archivePlayer)
    {
        nw::snd::SoundPlayer& player = archivePlayer.GetSoundPlayer(PLAYER_DEFAULT);

        switch ( s_FilterType )
        {
        case FILTER_TYPE_NONE:
            // ハンドルもプレイヤーもデフォルト値
            s_SoundHandle.SetLpfFreq( LPF_VALUE_MAX );
            s_SoundHandle.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
            player.SetLpfFreq( LPF_VALUE_MAX );
            player.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
            break;
        case FILTER_TYPE_HANDLE_LPF:
            // ハンドルの LPF のみ設定
            s_SoundHandle.SetLpfFreq( s_LpfValue );
            // 他はデフォルト値
            s_SoundHandle.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
            player.SetLpfFreq( LPF_VALUE_MAX );
            player.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
            break;
        case FILTER_TYPE_HANDLE_BIQUAD_LPF:
        case FILTER_TYPE_HANDLE_BIQUAD_HPF:
        case FILTER_TYPE_HANDLE_BIQUAD_BPF_512:
        case FILTER_TYPE_HANDLE_BIQUAD_BPF_1024:
        case FILTER_TYPE_HANDLE_BIQUAD_BPF_2048:
            // ハンドルの Biquad Filter のみ設定
            s_SoundHandle.SetBiquadFilter( s_FilterType - FILTER_TYPE_HANDLE_LPF, s_BqfValue );
            // 他はデフォルト値
            s_SoundHandle.SetLpfFreq( LPF_VALUE_MAX );
            player.SetLpfFreq( LPF_VALUE_MAX );
            player.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
            break;
        case FILTER_TYPE_PLAYER_LPF:
            // プレイヤーの LPF のみ設定
            player.SetLpfFreq( s_LpfValue );
            // 他はデフォルト値
            s_SoundHandle.SetLpfFreq( LPF_VALUE_MAX );
            s_SoundHandle.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
            player.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
            break;
        case FILTER_TYPE_PLAYER_BIQUAD_LPF:
        case FILTER_TYPE_PLAYER_BIQUAD_HPF:
        case FILTER_TYPE_PLAYER_BIQUAD_BPF_512:
        case FILTER_TYPE_PLAYER_BIQUAD_BPF_1024:
        case FILTER_TYPE_PLAYER_BIQUAD_BPF_2048:
            // プレイヤーの Biquad Filter のみ設定
            player.SetBiquadFilter( s_FilterType - FILTER_TYPE_PLAYER_LPF, s_BqfValue );
            // 他はデフォルト値
            s_SoundHandle.SetLpfFreq( LPF_VALUE_MAX );
            s_SoundHandle.SetBiquadFilter( nw::snd::BIQUAD_FILTER_TYPE_INHERIT, BQF_VALUE_MIN );
            player.SetLpfFreq( LPF_VALUE_MAX );
            break;
        }

        NW_LOG("filter( %-22s )   ", GetFilterTypeString( s_FilterType ));
        if ( s_FilterType == FILTER_TYPE_NONE )
        {
            NW_LOG("\n");
        }
        else if ( s_FilterType == FILTER_TYPE_HANDLE_LPF ||
                  s_FilterType == FILTER_TYPE_PLAYER_LPF )
        {
            NW_LOG("value(% .1f)\n", s_LpfValue );
        }
        else
        {
            NW_LOG("value(% .1f)\n", s_BqfValue );
        }
    }

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(snddemo::GetCommmonSoundArchivePath(), soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);
        snddemo::InitializeFsSoundArchive(s_SoundArchive, soundArchivePath, allocator);
        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundArchivePlayer(s_SoundArchivePlayer, s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundHeap(s_SoundHeap, SOUND_HEAP_SIZE, allocator);

        // データロード
        if ( ! s_SoundDataManager.LoadData( SEQ_MARIOKART, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SEQ_MARIOKART) failed." );
        }
        if ( ! s_SoundDataManager.LoadData( SE_YOSHI, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
        }
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        snddemo::FinalizeSoundArchivePlayer(s_SoundArchivePlayer, allocator);
        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
        snddemo::FinalizeFsSoundArchive(s_SoundArchive, allocator);
        snddemo::FinalizeSoundHeap(s_SoundHeap, allocator);
        snddemo::FinalizeSoundSystem(allocator);
    }

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A]          Start SEQ  (SEQ_MARIOKART)\n");
        NW_LOG("[X]          Start WSD  (SE_YOSHI)\n");
        NW_LOG("[Y]          Start STRM (STRM_MARIOKART)\n");
        NW_LOG("[B]          Stop Sound\n");
        NW_LOG("[LEFT/RIGHT] Change FilterType\n");
        NW_LOG("[UP/DOWN]    Change FilterValue\n");
#if defined( NW_PLATFORM_CAFE )
        NW_LOG("[HOME]       Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
        // TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_LOG("[S]          Exit Application\n");
#endif
        NW_LOG("----------------------------------------\n");
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::Pad* pad = pDemo->GetPad();

        // サウンドの再生
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SEQ_MARIOKART )
                .IsSuccess();
            SetFilter(s_SoundArchivePlayer);
            NW_LOG("[SEQ] SEQ_MARIOKART ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SE_YOSHI )
                .IsSuccess();
            SetFilter(s_SoundArchivePlayer);
            NW_LOG("[WSD] SE_YOSHI ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, STRM_MARIOKART )
                .IsSuccess();
            SetFilter(s_SoundArchivePlayer);
            NW_LOG("[STRM] STRM_MARIOKART ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_SoundHandle.Stop( 0 );
        }

        // フィルタの調整
        if ( pad->IsTrig( nw::demo::Pad::MASK_LEFT ) )
        {
            s_FilterType -= 1;
            if ( s_FilterType < 0 )
            {
                s_FilterType = FILTER_TYPE_MAX;
            }
            SetFilter(s_SoundArchivePlayer);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_RIGHT ) )
        {
            s_FilterType += 1;
            if ( s_FilterType > FILTER_TYPE_MAX )
            {
                s_FilterType = 0;
            }
            SetFilter(s_SoundArchivePlayer);
        }
        if ( s_FilterType == FILTER_TYPE_NONE )
        {
            // do nothing
        }
        else if ( s_FilterType == FILTER_TYPE_HANDLE_LPF ||
                  s_FilterType == FILTER_TYPE_PLAYER_LPF )
        {
            if ( pad->IsTrig( nw::demo::Pad::MASK_UP ) )
            {
                s_LpfValue += FILTER_DELTA;
                if ( s_LpfValue > LPF_VALUE_MAX - FILTER_VALUE_EPSILON )
                {
                    s_LpfValue = LPF_VALUE_MAX;
                }
                SetFilter(s_SoundArchivePlayer);
            }
            if ( pad->IsTrig( nw::demo::Pad::MASK_DOWN ) )
            {
                s_LpfValue -= FILTER_DELTA;
                if ( s_LpfValue < LPF_VALUE_MIN + FILTER_VALUE_EPSILON )
                {
                    s_LpfValue = LPF_VALUE_MIN;
                }
                SetFilter(s_SoundArchivePlayer);
            }
        }
        else    // biquad filter
        {
            if ( pad->IsTrig( nw::demo::Pad::MASK_UP ) )
            {
                s_BqfValue += FILTER_DELTA;
                if ( s_BqfValue > BQF_VALUE_MAX - FILTER_VALUE_EPSILON )
                {
                    s_BqfValue = BQF_VALUE_MAX;
                }
                SetFilter(s_SoundArchivePlayer);
            }
            if ( pad->IsTrig( nw::demo::Pad::MASK_DOWN ) )
            {
                s_BqfValue -= FILTER_DELTA;
                if ( s_BqfValue < BQF_VALUE_MIN + FILTER_VALUE_EPSILON )
                {
                    s_BqfValue = BQF_VALUE_MIN;
                }
                SetFilter(s_SoundArchivePlayer);
            }
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) ) {
            return false;
        }

        s_SoundArchivePlayer.Update();

        return true;
    }
}

namespace snddemo
{

    void FilterDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);


        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        PrintUsage();

        s_FilterType = 0;
        s_BqfValue = 0.8f;
        s_LpfValue = -0.4f;
        SetFilter(s_SoundArchivePlayer);    // 初期状態表示のため

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();
    }

}
