﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/demo.h>
#include <nw/dev.h>
#include <nw/gfnd.h>
#include <nw/math.h>
#include <nw/ut.h>

#include <nw/lyt.h>
#include <nw/font.h>

#include <nw/dev/dev_Profile.h>

#if defined(NW_PLATFORM_WIN32)
#include <nn/os/os_Tick.h>
#endif

#include <nn/nn_Log.h>

#include "SimpleAllocator.h"

// ----------------------------------------------------------------------------
// 定数
// ----------------------------------------------------------------------------
const size_t NormalHeapMemorySize = 1024 * 1024 * 64;
const size_t LoggingHeapMemorySize = 1024 * 1024 * 64;
const uint32_t TestDataCount = 128;
const int FrameNum = 3;
const int MaxPath = 160;
#if defined(NW_PLATFORM_CAFE)
const char* const DemoFontFileName = "nintendo_NTLG-DB_002.bffnt";
#else
const char* const DemoFontFileName = "nintendo_NTLG-DB_002_Nw4f.bffnt";
#endif

// ----------------------------------------------------------------------------
// 変数宣言
// ----------------------------------------------------------------------------
SimpleAllocator g_NormalAllocator;
LoggingAllocator g_LoggingAllocator;
void* g_NormalAllocatorBuffer;
void* g_LoggingAllocatorBuffer;

nw::demo::DemoSystem* g_pDemo = NULL;

nw::lyt::Layout** g_pLayout;
nw::lyt::ArcResourceAccessor* g_pArcResourceAccessor;
void* g_pLayoutArchiveBinary;
nw::font::ResFont* g_pFont;
void* g_pFontBinary;
nw::lyt::GraphicsResource g_GraphicsResource;
nw::lyt::DrawInfo g_DrawInfo;
nw::lyt::GroupAnimator* g_pAnimator;
nw::gfnd::GraphicsContext g_GraphicsContext;
#if defined(NW_PLATFORM_CAFE)
nw::lyt::ShaderSetupHelper g_ShaderSetupHelper;
#endif

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
class ProfilerAllocator : public nw::ut::IAllocator
{
public:

    virtual void* Alloc(size_t size, u32 alignment)
    {
        return g_NormalAllocator.Alloc(size, alignment);
    }

    //---------------------------------------------------------------------------
    //! @brief メモリを解放します。
    //! @param[in] memory 解放するメモリのアドレスです。
    //---------------------------------------------------------------------------
    virtual void  Free(void* memory)
    {
        g_NormalAllocator.Free(memory);
    }
protected:
private:
};

ProfilerAllocator   g_ProfAllocator;

// ----------------------------------------------------------------------------
// 関数定義
// ----------------------------------------------------------------------------

// 各種初期化処理
void Initialize(const char* layoutName)
{
    // アロケータの初期化
    {
        g_NormalAllocatorBuffer = new char[NormalHeapMemorySize];
        g_NormalAllocator.Initialize(g_NormalAllocatorBuffer, static_cast<u32>(NormalHeapMemorySize));
        g_LoggingAllocatorBuffer = new char[LoggingHeapMemorySize];
        g_LoggingAllocator.Initialize(g_LoggingAllocatorBuffer, static_cast<u32>(LoggingHeapMemorySize));
    }

    NW_INITIALIZE_PROFILE(128, &g_ProfAllocator);

    // デモシステムの作成
    {
        nw::demo::DemoSystem::CreateArg arg;
        arg.allocator = &g_NormalAllocator;
        arg.waitVBlank = 1;
        arg.width = 1280;
        arg.height = 720;
        arg.drawMeter = false;
        arg.fontPath = DemoFontFileName;
        arg.clearColor.r = 0;
        arg.clearColor.g = 0;
        arg.clearColor.b = 128;
        arg.clearColor.a = 255;
        arg.isSRGBWrite = false;
#if defined(NW_PLATFORM_CAFE)
        arg.fontShaderPath = "font_BuildinShader.gsh";
        arg.primitiveRendererShaderPath = "dev_PrimitiveRenderer.gsh";
#else
        arg.primitiveRendererInitialize = true;
#endif
        arg.fileDeviceManagerInitialize = true;
        g_pDemo = new(g_NormalAllocator.Alloc(sizeof(nw::demo::DemoSystem), 1)) nw::demo::DemoSystem(arg);
        g_pDemo->Initialize();
        g_pDemo->InitializeGraphicsSystem();
    }

    // レイアウトの初期化
    nw::lyt::Initialize(&g_LoggingAllocator);

    // リソースアクセサの初期化
    g_pArcResourceAccessor = new(g_LoggingAllocator.AllocateMemory(sizeof(nw::lyt::ArcResourceAccessor), 1)) nw::lyt::ArcResourceAccessor();

    // レイアウトアーカイブの読み込み
    {
        char arcName[MaxPath];
        sprintf(arcName, "%s.arc", layoutName);
        nw::dev::FileDevice::LoadArg arg;
        arg.path = arcName;
        arg.allocator = &g_NormalAllocator;
        arg.alignment = nw::lyt::ARCHIVE_RESOURCE_ALIGNMENT;
        g_pLayoutArchiveBinary = nw::dev::FileDeviceManager::GetInstance()->Load(arg);
        NW_ASSERT(g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, "."));
    }

    // フォントの読み込み
    {
        const char* fontName = "sample.bffnt";
        g_pFont = new(g_LoggingAllocator.AllocateMemory(sizeof(nw::font::ResFont), 1)) nw::font::ResFont();
        nw::dev::FileDevice::LoadArg arg;
        arg.path = fontName;
        arg.allocator = &g_NormalAllocator;
        arg.alignment = nw::font::RESOURCE_ALIGNMENT;
        g_pFontBinary = nw::dev::FileDeviceManager::GetInstance()->Load(arg);
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        NW_ASSERT(g_pFont->SetResource(g_pFontBinary));
        g_pArcResourceAccessor->RegistFont(fontName, g_pFont);
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
    }

    // グラフィックスリソースの設定
#if defined(NW_PLATFORM_CAFE)
    {
        u8* pFontBinary = NULL;
        u8* pLayoutBinary = NULL;
        nw::ut::MemoryRange fontShaderBinary;
        nw::ut::MemoryRange lytShaderBinary;
        {
            nw::dev::FileDevice::LoadArg arg;
            arg.path = "font_BuildinShader.gsh";
            arg.allocator = &g_NormalAllocator;
            pFontBinary = nw::dev::FileDeviceManager::GetInstance()->Load(arg);
            fontShaderBinary = nw::ut::MakeMemoryRange(pFontBinary, arg.readSize);
        }
        {
            nw::dev::FileDevice::LoadArg arg;
            arg.path = "lyt_BuildinShader.gsh";
            arg.allocator = &g_NormalAllocator;
            pLayoutBinary = nw::dev::FileDeviceManager::GetInstance()->Load(arg);
            lytShaderBinary = nw::ut::MakeMemoryRange(pLayoutBinary, arg.readSize);
        }
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        g_GraphicsResource.Setup(512, fontShaderBinary);
        g_ShaderSetupHelper.Setup(&g_GraphicsResource, lytShaderBinary);
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
    }
#else
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    g_GraphicsResource.Setup(512);
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
#endif

    g_pLayout = static_cast<nw::lyt::Layout**>(g_LoggingAllocator.AllocateMemory(sizeof(nw::lyt::Layout*) * TestDataCount, 4));

    for (uint32_t i = 0; i < TestDataCount;++i)
    {
        // レイアウトの初期化
        g_pLayout[i] = new(g_LoggingAllocator.AllocateMemory(sizeof(nw::lyt::Layout), 1)) nw::lyt::Layout();
        {
            char bflytName[MaxPath];
            sprintf(bflytName, "%s.bflyt", layoutName);
            nw::gfnd::Graphics::GetInstance()->LockDrawContext();
            g_pLayout[i]->BuildWithName(bflytName, g_pArcResourceAccessor);
            nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
        }

        // 描画に使用する情報の設定
        {
            nw::math::MTX44 projection;
            nw::ut::Rect rect = g_pLayout[i]->GetLayoutRect();
            nw::math::MTX44Ortho(&projection, rect.left, rect.right, rect.bottom, rect.top, 0.0f, 300.0f);
            nw::math::MTX34 view;
            nw::math::VEC3 pos(0.0f, 0.0f, 1.0f);
            nw::math::VEC3 up(0.0f, 1.0f, 0.0f);
            nw::math::VEC3 target(0.0f, 0.0f, 0.0f);
            nw::math::MTX34LookAt(&view, &pos, &up, &target);

            g_DrawInfo.SetGraphicsResource(&g_GraphicsResource);
            g_DrawInfo.SetProjectionMtx(projection);
            g_DrawInfo.SetViewMtx(view);
        }

        // アニメーションの作成と再生
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        g_pAnimator = g_pLayout[i]->CreateGroupAnimator("loop");
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
        g_pAnimator->PlayAuto(1.0f);
    }
} // NOLINT (readability/fn_size)

// 各種解放処理
void Finalize()
{
    // レイアウトの解放
#if defined(NW_PLATFORM_CAFE)
    g_ShaderSetupHelper.Finalize();
#endif
    g_GraphicsResource.Finalize();
    for (uint32_t i = 0; i < TestDataCount;++i)
    {
        g_pLayout[i]->~Layout();
        g_LoggingAllocator.FreeMemory(g_pLayout[i]);
    }
    g_LoggingAllocator.FreeMemory(g_pLayout);
    g_pArcResourceAccessor->Detach();
    g_pFont->RemoveResource();
    g_pFont->~ResFont();
    g_LoggingAllocator.FreeMemory(g_pFont);
    g_NormalAllocator.Free(g_pFontBinary);
    g_NormalAllocator.Free(g_pLayoutArchiveBinary);
    g_pArcResourceAccessor->~ArcResourceAccessor();
    g_LoggingAllocator.FreeMemory(g_pArcResourceAccessor);

    // デモシステムの解放
    g_pDemo->FinalizeGraphicsSystem();
    g_pDemo->Finalize();
    g_NormalAllocator.Free(g_pDemo);

    // メモリ使用量の表示
    g_LoggingAllocator.PrintOccupiedMemory();

    // アロケータの解放
    g_LoggingAllocator.Finalize();
    delete[] g_LoggingAllocatorBuffer;
    g_NormalAllocator.Finalize();
    delete[] g_NormalAllocatorBuffer;
} // NOLINT (readability/fn_size)

// 一連処理の実行
void Run(const char* layoutName)
{
    NN_LOG("Begin: %s\n", layoutName);

    // 各種初期化処理
    Initialize(layoutName);

    // CPU 負荷の計測準備
    int64_t totalTime = 0;

    // 描画ループ
    for (int frame = 0; frame < FrameNum; frame++)
    {
        NW_START_PROFILE("LytTest");
        g_pDemo->BeginFrame();
        {
            // ロック
            nw::gfnd::Graphics* pGraphics = nw::gfnd::Graphics::GetInstance();
            pGraphics->LockDrawContext();

#if defined(NW_PLATFORM_CAFE)
            // GPU の read キャッシュを全域クリア
            GX2Invalidate(static_cast<GX2InvalidateType>(GX2_INVALIDATE_ATTRIB_BUFFER | GX2_INVALIDATE_TEXTURE | GX2_INVALIDATE_CONSTANT_BUFFER | GX2_INVALIDATE_SHADER), 0x00000000, 0xffffffff);
#endif
            // 画面のクリアと設定
            g_pDemo->ClearFrameBuffers();
            g_pDemo->SetViewport(0.0f, 0.0f, static_cast<f32>(g_pDemo->GetWidth()), static_cast<f32>(g_pDemo->GetHeight()), 0.0f, 1.0f);
            g_pDemo->SetScissor(0.0f, 0.0f, static_cast<f32>(g_pDemo->GetWidth()), static_cast<f32>(g_pDemo->GetHeight()));

            // キャッシュを汚す目的で CPU の L3 キャッシュサイズ分適当にメモリを読み書きする。
            char* pSrc = new char[10 * 1024 * 1024 + frame * 1024];
            char* pDest = new char[10 * 1024 * 1024 + frame * 1024];

            for(int write = 0; write < 10 * 1024 * 1024 + frame * 1024;++write)
            {
                pDest[write] = pSrc[write];
            }

            // CPU 負荷計測の前処理
            int64_t beforeTime;
#if defined(NW_PLATFORM_CAFE)
            beforeTime = static_cast<int64_t>(OSGetTick());
#else
            beforeTime = nn::os::GetSystemTick().GetInt64Value();
#endif

            g_GraphicsContext.SetDepthEnable(false, false);
            g_GraphicsContext.SetCullingMode(nw::gfnd::Graphics::CULLING_MODE_NONE);
            g_GraphicsContext.Apply();

            // レイアウトの描画
            for (uint32_t i = 0; i < TestDataCount;++i)
            {
                g_pLayout[i]->AnimateAndUpdateAnimFrame();
                g_pLayout[i]->CalculateMtx(g_DrawInfo);
                g_pLayout[i]->Draw(g_DrawInfo);
            }

            NW_STOP_PROFILE();
            NW_DUMP_PROFILE();
            NW_CLEAR_PROFILE();

            // CPU 負荷計測の後処理
            int64_t afterTime;
#if defined(NW_PLATFORM_CAFE)
            afterTime = static_cast<int64_t>(OSGetTick());
#else
            afterTime = nn::os::GetSystemTick().GetInt64Value();
#endif
            totalTime += afterTime - beforeTime;
            NN_LOG("CPU load(%d): %lld\n", frame, afterTime - beforeTime);
            delete[] pSrc;
            delete[] pDest;

            // アンロック
            pGraphics->UnlockDrawContext();

            // スワップ
            g_pDemo->SwapBuffer();
        }
        g_pDemo->EndFrame();
        // 垂直同期待ち
        g_pDemo->WaitForVBlank();
    }

    // CPU 負荷の表示
    NN_LOG("CPU load: %lld\n", totalTime / static_cast<int64_t>(FrameNum));

    // 各種解放処理
    Finalize();

    NW_FINALIZE_PROFILE(&g_ProfAllocator);

    NN_LOG("End: %s\n\n", layoutName);
} // NOLINT (readability/fn_size)

// メイン関数
int NwDemoMain(int argc, char** argv)
{
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);

    Run("simple");

    return 0;
}
