﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3ddemo_ModelUtility.h>
#include <nw/g3d/res/g3d_ResFile.h>
#include <nw/g3d/res/g3d_ResShader.h>
#include <nw/g3d/g3d_MaterialObj.h>
#include <nw/g3d/g3d_ShaderUtility.h>
#include <g3ddemo_DemoUtility.h>
#include <g3ddemo_UserModel.h>

namespace nw { namespace g3d { namespace demo {

void CreateShaderAssign(nw::g3d::ResModel* pResModel)
{
    NW_G3D_ASSERT_NOT_NULL(pResModel);

    int numMat = pResModel->GetMaterialCount();
    int numShape = pResModel->GetShapeCount();

    // ModelAssign と MaterialAssign、ShapeAssign を1つのバッファで確保します。
    size_t size = sizeof(ModelAssign) +
        sizeof(MaterialAssign) * numMat + sizeof(ShapeAssign) * numShape;
    void* ptr = AllocMem2(size);
    NW_G3D_ASSERT_NOT_NULL(ptr);

    // ResModel に対して ModelAssign を付加します。
    ModelAssign* pModelAssign = new(ptr) ModelAssign(pResModel);
    NW_G3D_ASSERT(pResModel->GetUserPtr() == NULL);
    pResModel->SetUserPtr(pModelAssign);
    ptr = nw::g3d::AddOffset(ptr, sizeof(*pModelAssign));

    // ResMaterial に対して MaterialAssign を付加します。
    // エディットライブラリが ResMaterial を差し替えても追従できるよう、
    // ResMaterial の UserPtr 経由で取得します。
    for (int idxMat = 0; idxMat < numMat; ++idxMat)
    {
        nw::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(idxMat);
        if (pResMaterial->GetUserPtr() == NULL)
        {
            MaterialAssign* pMatAssign = new(ptr) MaterialAssign(pResMaterial);
            pResMaterial->SetUserPtr(pMatAssign);
            ptr = nw::g3d::AddOffset(ptr, sizeof(*pMatAssign));
        }
    }

    // ResShape に対して ShapeAssign を付加します。
    for (int idxShape = 0; idxShape < numShape; ++idxShape)
    {
        nw::g3d::ResShape* pResShape = pResModel->GetShape(idxShape);

        nw::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(pResShape->GetMaterialIndex());
        if (pResShape->GetUserPtr() == NULL)
        {
            ShapeAssign* pShapeAssign = new(ptr) ShapeAssign(pResShape, pResMaterial);
            pResShape->SetUserPtr(pShapeAssign);
            ptr = nw::g3d::AddOffset(ptr, sizeof(*pShapeAssign));
        }
    }
}

void DestroyShaderAssign(nw::g3d::ResModel* pResModel)
{
    NW_G3D_ASSERT_NOT_NULL(pResModel);

    // MaterialAssign を取り外して破棄します。
    for (int idxMat = 0, numMat = pResModel->GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        nw::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(idxMat);
        if (MaterialAssign* pMatAssign = pResMaterial->GetUserPtr<MaterialAssign>())
        {
            pMatAssign->~MaterialAssign();
            pResMaterial->SetUserPtr(NULL);
        }
    }

    // ShapeAssign を取り外して破棄します。
    for (int idxShape = 0, numShape = pResModel->GetShapeCount(); idxShape < numShape; ++idxShape)
    {
        nw::g3d::ResShape* pResShape = pResModel->GetShape(idxShape);
        if (ShapeAssign* pShapeAssign = pResShape->GetUserPtr<ShapeAssign>())
        {
            pShapeAssign->~ShapeAssign();
            pResShape->SetUserPtr(NULL);
        }
    }

    // ModelAssign を取り外して破棄します。
    NW_G3D_ASSERT_NOT_NULL(pResModel->GetUserPtr());
    ModelAssign* pModelAssign = pResModel->GetUserPtr<ModelAssign>();
    pModelAssign->~ModelAssign();
    pResModel->SetUserPtr(NULL);

    // ModelAssign と MaterialAssign、ShapeAssign を1つのバッファで確保したので、
    // 解放も一括で行います。
    FreeMem2(pModelAssign);
}

//--------------------------------------------------------------------------------------------------

void CreateUserModel(nw::g3d::ModelObj* pModelObj)
{
    NW_G3D_ASSERT_NOT_NULL(pModelObj);

    int numMat = pModelObj->GetMaterialCount();
    int numShape = pModelObj->GetShapeCount();

    // UserModel と UserMaterial、UserShape を1つのバッファで確保します。
    size_t size = sizeof(UserModel) +
        sizeof(UserMaterial) * numMat + sizeof(UserShape) * numShape;
    void* ptr = AllocMem2(size);
    NW_G3D_ASSERT_NOT_NULL(ptr);

    // ModelObj に対して UserModel を付加します。
    UserModel* pUserModel = new(ptr) UserModel(pModelObj);
    NW_G3D_ASSERT(pModelObj->GetUserPtr() == NULL);
    pModelObj->SetUserPtr(pUserModel);
    ptr = nw::g3d::AddOffset(ptr, sizeof(*pUserModel));

    // MaterialObj に対して UserMaterial を付加します。
    for (int idxMat = 0; idxMat < numMat; ++idxMat)
    {
        nw::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(idxMat);
        if (pMaterialObj->GetUserPtr() == NULL)
        {
            UserMaterial* pUserMat = new(ptr) UserMaterial(pMaterialObj);
            pMaterialObj->SetUserPtr(pUserMat);
            ptr = nw::g3d::AddOffset(ptr, sizeof(*pUserMat));
        }
    }

    // ShapeObj に対して UserShape を付加します。
    for (int idxShape = 0; idxShape < numShape; ++idxShape)
    {
        nw::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(idxShape);
        if (pShapeObj->GetUserPtr() == NULL)
        {
            UserShape* pUserShape = new(ptr) UserShape(pShapeObj);
            pUserShape->Setup();
            pShapeObj->SetUserPtr(pUserShape);
            ptr = nw::g3d::AddOffset(ptr, sizeof(*pUserShape));
        }
    }
}

void DestroyUserModel(nw::g3d::ModelObj* pModelObj)
{
    NW_G3D_ASSERT_NOT_NULL(pModelObj);

    int numMat = pModelObj->GetMaterialCount();
    int numShape = pModelObj->GetShapeCount();

    // UserMaterial を取り外して破棄します。
    for (int idxMat = 0; idxMat < numMat; ++idxMat)
    {
        nw::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(idxMat);
        if (UserMaterial* pUserMat = pMaterialObj->GetUserPtr<UserMaterial>())
        {
            pUserMat->~UserMaterial();
            pMaterialObj->SetUserPtr(NULL);
        }
    }

    // UserShape を取り外して破棄します。
    for (int idxShape = 0; idxShape < numShape; ++idxShape)
    {
        nw::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(idxShape);
        if (UserShape* pUserShape = pShapeObj->GetUserPtr<UserShape>())
        {
            pUserShape->Cleanup();
            pUserShape->~UserShape();
            pShapeObj->SetUserPtr(NULL);
        }
    }

    // UserModel を取り外して破棄します。
    NW_G3D_ASSERT_NOT_NULL(pModelObj->GetUserPtr());
    UserModel* pUserModel = pModelObj->GetUserPtr<UserModel>();
    pUserModel->~UserModel();
    pModelObj->SetUserPtr(NULL);

    // UserModel と UserMaterial、UserShape を1つのバッファで確保したので、
    // 解放も一括で行います。
    FreeMem2(pUserModel);
}

//--------------------------------------------------------------------------------------------------

void CreateUserShadingModel(nw::g3d::ResShadingModel* pResShadingModel, nw::g3d::ModelObj* pEnvModel)
{
    size_t size = sizeof(UserShadingModel);
    void* ptr = AllocMem2(size);
    NW_G3D_ASSERT_NOT_NULL(ptr);

    UserShadingModel* pUserShadingModel = new(ptr) UserShadingModel(pResShadingModel, pEnvModel);
    NW_G3D_ASSERT(pResShadingModel->GetUserPtr() == NULL);

    pResShadingModel->SetUserPtr(pUserShadingModel);
}

void DestroyUserShadingModel(nw::g3d::ResShadingModel* pResShadingModel)
{
    NW_G3D_ASSERT_NOT_NULL(pResShadingModel);
    if (UserShadingModel* pUserShadingModel = pResShadingModel->GetUserPtr<UserShadingModel>())
    {
        pUserShadingModel->~UserShadingModel();
        pResShadingModel->SetUserPtr(NULL);

        FreeMem2(pUserShadingModel);
    }
}

//--------------------------------------------------------------------------------------------------

nw::g3d::ModelObj* CreateModelObj(const CreateModelObjArg& arg)
{
    nw::g3d::ModelObj* pModelObj = NULL;

    {
        size_t alignment = nw::g3d::ModelObj::BUFFER_ALIGNMENT;
        size_t bufSize = nw::g3d::ModelObj::CalcBufferSize(arg.initArg);
        size_t objSize = nw::g3d::Align(sizeof(*pModelObj), alignment);
        void* ptr = AllocMem2(objSize + bufSize, alignment);
        NW_G3D_ASSERT_NOT_NULL(ptr);

        pModelObj = new(ptr) nw::g3d::ModelObj();
        ptr = nw::g3d::AddOffset(ptr, objSize);

        bool success = pModelObj->Init(arg.initArg, ptr, bufSize);
        (void)success;
        NW_G3D_ASSERT(success);
    }

    {
        // Uniform Block のバッファは所定のアライメントが必要です。
        size_t size = pModelObj->CalcBlockBufferSize();
        void* ptr = AllocMem2(size, nw::g3d::ModelObj::BLOCK_BUFFER_ALIGNMENT);
        NW_G3D_ASSERT_NOT_NULL(ptr);

        bool success = pModelObj->SetupBlockBuffer(ptr, size);
        (void)success;
        NW_G3D_ASSERT(success);
    }

    return pModelObj;
}

void DestroyModelObj(nw::g3d::ModelObj* pModelObj)
{
    NW_G3D_ASSERT_NOT_NULL(pModelObj);

    pModelObj->CleanupBlockBuffer();
    if (void* ptr = pModelObj->GetBlockBufferPtr())
    {
        FreeMem2(ptr);
    }

    // BufferPtr は pModelObj のバッファの一部として確保したので解放不要。
    FreeMem2(pModelObj);
}

//--------------------------------------------------------------------------------------------------

nw::g3d::SkeletalAnimObj* CreateSkeletalAnimObj(const SkeletalAnimObj::InitArg& arg)
{
    nw::g3d::SkeletalAnimObj* pAnimObj = NULL;

    // オブジェクトと初期化用バッファを同じバッファからとるため、
    // オブジェクトのサイズをバッファのアライメントで切り上げています。
    size_t alignment = nw::g3d::SkeletalAnimObj::BUFFER_ALIGNMENT;
    size_t objSize = nw::g3d::Align(sizeof(*pAnimObj), alignment);
    size_t bufSize = nw::g3d::SkeletalAnimObj::CalcBufferSize(arg);
    void* ptr = AllocMem2(objSize + bufSize, alignment);
    NW_G3D_ASSERT_NOT_NULL(ptr);

    pAnimObj = new(ptr) nw::g3d::SkeletalAnimObj();
    ptr = nw::g3d::AddOffset(ptr, objSize);

    bool success = pAnimObj->Init(arg, ptr, bufSize);
    (void)success;
    NW_G3D_ASSERT(success);

    return pAnimObj;
}

nw::g3d::VisibilityAnimObj* CreateVisibilityAnimObj(const VisibilityAnimObj::InitArg& arg)
{
    nw::g3d::VisibilityAnimObj* pAnimObj = NULL;

    // オブジェクトと初期化用バッファを同じバッファからとるため、
    // オブジェクトのサイズをバッファのアライメントで切り上げています。
    size_t alignment = nw::g3d::VisibilityAnimObj::BUFFER_ALIGNMENT;
    size_t objSize = nw::g3d::Align(sizeof(*pAnimObj), alignment);
    size_t bufSize = nw::g3d::VisibilityAnimObj::CalcBufferSize(arg);
    void* ptr = AllocMem2(objSize + bufSize, alignment);
    NW_G3D_ASSERT_NOT_NULL(ptr);

    pAnimObj = new(ptr) nw::g3d::VisibilityAnimObj();
    ptr = nw::g3d::AddOffset(ptr, objSize);

    bool success = pAnimObj->Init(arg, ptr, bufSize);
    (void)success;
    NW_G3D_ASSERT(success);

    return pAnimObj;
}

nw::g3d::ShaderParamAnimObj* CreateShaderParamAnimObj(const ShaderParamAnimObj::InitArg& arg)
{
    nw::g3d::ShaderParamAnimObj* pAnimObj = NULL;

    // オブジェクトと初期化用バッファを同じバッファからとるため、
    // オブジェクトのサイズをバッファのアライメントで切り上げています。
    size_t alignment = nw::g3d::ShaderParamAnimObj::BUFFER_ALIGNMENT;
    size_t objSize = nw::g3d::Align(sizeof(*pAnimObj), alignment);
    size_t bufSize = nw::g3d::ShaderParamAnimObj::CalcBufferSize(arg);
    void* ptr = AllocMem2(objSize + bufSize, alignment);
    NW_G3D_ASSERT_NOT_NULL(ptr);

    pAnimObj = new(ptr) nw::g3d::ShaderParamAnimObj();
    ptr = nw::g3d::AddOffset(ptr, objSize);

    bool success = pAnimObj->Init(arg, ptr, bufSize);
    (void)success;
    NW_G3D_ASSERT(success);

    return pAnimObj;
}

nw::g3d::TexPatternAnimObj* CreateTexPatternAnimObj(const TexPatternAnimObj::InitArg& arg)
{
    nw::g3d::TexPatternAnimObj* pAnimObj = NULL;

    // オブジェクトと初期化用バッファを同じバッファからとるため、
    // オブジェクトのサイズをバッファのアライメントで切り上げています。
    size_t alignment = nw::g3d::TexPatternAnimObj::BUFFER_ALIGNMENT;
    size_t objSize = nw::g3d::Align(sizeof(*pAnimObj), alignment);
    size_t bufSize = nw::g3d::TexPatternAnimObj::CalcBufferSize(arg);
    void* ptr = AllocMem2(objSize + bufSize, alignment);
    NW_G3D_ASSERT_NOT_NULL(ptr);

    pAnimObj = new(ptr) nw::g3d::TexPatternAnimObj();
    ptr = nw::g3d::AddOffset(ptr, objSize);

    bool success = pAnimObj->Init(arg, ptr, bufSize);
    (void)success;
    NW_G3D_ASSERT(success);

    return pAnimObj;
}

nw::g3d::ShapeAnimObj* CreateShapeAnimObj(const ShapeAnimObj::InitArg& arg)
{
    nw::g3d::ShapeAnimObj* pAnimObj = NULL;

    // オブジェクトと初期化用バッファを同じバッファからとるため、
    // オブジェクトのサイズをバッファのアライメントで切り上げています。
    size_t alignment = nw::g3d::ShapeAnimObj::BUFFER_ALIGNMENT;
    size_t objSize = nw::g3d::Align(sizeof(*pAnimObj), alignment);
    size_t bufSize = nw::g3d::ShapeAnimObj::CalcBufferSize(arg);
    void* ptr = AllocMem2(objSize + bufSize, alignment);
    NW_G3D_ASSERT_NOT_NULL(ptr);

    pAnimObj = new(ptr) nw::g3d::ShapeAnimObj();
    ptr = nw::g3d::AddOffset(ptr, objSize);

    bool success = pAnimObj->Init(arg, ptr, bufSize);
    (void)success;
    NW_G3D_ASSERT(success);

    return pAnimObj;
}

void DestroyAnimObj(nw::g3d::AnimObj* pAnimObj)
{
    NW_G3D_ASSERT_NOT_NULL(pAnimObj);

    // BufferPtr は pAnimObj のバッファの一部として確保したので解放不要。
    // 別途確保していた場合は GetResultBuffer() で取得したバッファを解放します。
    FreeMem2(pAnimObj);
}

//--------------------------------------------------------------------------------------------------

nw::g3d::ShadingModelObj* CreateShadingModelObj(const ShadingModelObj::InitArg& arg)
{
    nw::g3d::ShadingModelObj* pShadingModelObj = NULL;

    {
        size_t alignment = nw::g3d::ShadingModelObj::BUFFER_ALIGNMENT;
        size_t bufSize = nw::g3d::ShadingModelObj::CalcBufferSize(arg);
        size_t objSize = nw::g3d::Align(sizeof(*pShadingModelObj), alignment);
        void* ptr = AllocMem2(objSize + bufSize, alignment);
        NW_G3D_ASSERT_NOT_NULL(ptr);

        pShadingModelObj = new(ptr) nw::g3d::ShadingModelObj();
        ptr = nw::g3d::AddOffset(ptr, objSize);

        bool success = pShadingModelObj->Init(arg, ptr, bufSize);
        (void)success;
        NW_G3D_ASSERT(success);
    }

    {
        // Uniform Block のバッファは所定のアライメントが必要です。
        size_t size = pShadingModelObj->CalcBlockBufferSize();
        void* ptr = AllocMem2(size, nw::g3d::ShadingModelObj::BLOCK_BUFFER_ALIGNMENT);
        NW_G3D_ASSERT_NOT_NULL(ptr);

        bool success = pShadingModelObj->SetupBlockBuffer(ptr, size);
        (void)success;
        NW_G3D_ASSERT(success);
    }

    return pShadingModelObj;
}

void DestroyShadingModelObj(nw::g3d::ShadingModelObj* pShadingModelObj)
{
    NW_G3D_ASSERT_NOT_NULL(pShadingModelObj);

    if (void* ptr = pShadingModelObj->GetBlockBufferPtr())
    {
        FreeMem2(ptr);
    }

    // BufferPtr は pShadingModelObj のバッファの一部として確保したので解放不要。
    FreeMem2(pShadingModelObj);
}

nw::g3d::ShaderSelector* CreateShaderSelector(const ShaderSelector::InitArg& arg)
{
    nw::g3d::ShaderSelector* pShaderSelector = NULL;

    {
        size_t alignment = nw::g3d::ShaderSelector::BUFFER_ALIGNMENT;
        size_t bufSize = nw::g3d::ShaderSelector::CalcBufferSize(arg);
        size_t objSize = nw::g3d::Align(sizeof(*pShaderSelector), alignment);
        void* ptr = AllocMem2(objSize + bufSize, alignment);
        NW_G3D_ASSERT_NOT_NULL(ptr);

        pShaderSelector = new(ptr) nw::g3d::ShaderSelector();
        ptr = nw::g3d::AddOffset(ptr, objSize);

        bool success = pShaderSelector->Init(arg, ptr, bufSize);
        (void)success;
        NW_G3D_ASSERT(success);
    }

    return pShaderSelector;
}

void DestroyShaderSelector(nw::g3d::ShaderSelector* pShaderSelector)
{
    NW_G3D_ASSERT_NOT_NULL(pShaderSelector);

    // BufferPtr は pShaderSelector のバッファの一部として確保したので解放不要。
    FreeMem2(pShaderSelector);
}

//--------------------------------------------------------------------------------------------------

void RegistAnim(ResourceHolder* pHolder)
{
    for (int idxFile = 0, numFile = pHolder->files.Size(); idxFile < numFile; ++idxFile)
    {
        nw::g3d::ResFile* pResFile = pHolder->files[idxFile];

        for (int idxAnim = 0, numAnim = pResFile->GetSkeletalAnimCount();
            idxAnim < numAnim; ++idxAnim)
        {
            nw::g3d::ResSkeletalAnim* pAnim = pResFile->GetSkeletalAnim(idxAnim);
            void* ptr = pAnim;
            pHolder->modelAnims.PushBack(ptr);
        }
        for (int idxAnim = 0, numAnim = pResFile->GetBoneVisAnimCount();
            idxAnim < numAnim; ++idxAnim)
        {
            nw::g3d::ResVisibilityAnim* pAnim = pResFile->GetBoneVisAnim(idxAnim);
            void* ptr = pAnim;
            pHolder->modelAnims.PushBack(ptr);
        }
        for (int idxAnim = 0, numAnim = pResFile->GetMatVisAnimCount();
            idxAnim < numAnim; ++idxAnim)
        {
            nw::g3d::ResVisibilityAnim* pAnim = pResFile->GetMatVisAnim(idxAnim);
            void* ptr = pAnim;
            pHolder->modelAnims.PushBack(ptr);
        }
        for (int idxAnim = 0, numAnim = pResFile->GetShaderParamAnimCount();
            idxAnim < numAnim; ++idxAnim)
        {
            nw::g3d::ResShaderParamAnim* pAnim = pResFile->GetShaderParamAnim(idxAnim);
            void* ptr = pAnim;
            pHolder->modelAnims.PushBack(ptr);
        }
        for (int idxAnim = 0, numAnim = pResFile->GetColorAnimCount();
            idxAnim < numAnim; ++idxAnim)
        {
            nw::g3d::ResShaderParamAnim* pAnim = pResFile->GetColorAnim(idxAnim);
            void* ptr = pAnim;
            pHolder->modelAnims.PushBack(ptr);
        }
        for (int idxAnim = 0, numAnim = pResFile->GetTexSrtAnimCount();
            idxAnim < numAnim; ++idxAnim)
        {
            nw::g3d::ResShaderParamAnim* pAnim = pResFile->GetTexSrtAnim(idxAnim);
            void* ptr = pAnim;
            pHolder->modelAnims.PushBack(ptr);
        }
        for (int idxAnim = 0, numAnim = pResFile->GetTexPatternAnimCount();
            idxAnim < numAnim; ++idxAnim)
        {
            nw::g3d::ResTexPatternAnim* pAnim = pResFile->GetTexPatternAnim(idxAnim);
            void* ptr = pAnim;
            pHolder->modelAnims.PushBack(ptr);
        }
        for (int idxAnim = 0, numAnim = pResFile->GetShapeAnimCount();
            idxAnim < numAnim; ++idxAnim)
        {
            nw::g3d::ResShapeAnim* pAnim = pResFile->GetShapeAnim(idxAnim);
            void* ptr = pAnim;
            pHolder->modelAnims.PushBack(ptr);
        }
    }
}

void DestroyAll(ResourceHolder* pHolder)
{
    for (int idxModel = 0, numModel = pHolder->models.Size();
        idxModel < numModel; ++idxModel)
    {
        UserModel* pUserModel = pHolder->models[idxModel];
        // EnvModel は後で削除するのでここでは消さない
        if (pHolder->pEnvModel == pUserModel)
        {
            continue;
        }

        nw::g3d::ModelObj* pModelObj = pUserModel->GetModelObj();
        pUserModel->DestroyAnimObj();
        DestroyUserModel(pModelObj);
        DestroyModelObj(pModelObj);
    }

    if (pHolder->pEnvModel)
    {
        UserModel* pUserModel = pHolder->pEnvModel;
        nw::g3d::ModelObj* pModelObj = pUserModel->GetModelObj();
        pUserModel->DestroyAnimObj();
        DestroyUserModel(pModelObj);
        DestroyModelObj(pModelObj);
        pHolder->pEnvModel = NULL;
    }

    for (int idxAssign = 0, numAssign = pHolder->assigns.Size();
        idxAssign < numAssign; ++idxAssign)
    {
        ModelAssign* pModelAssign = pHolder->assigns[idxAssign];
        nw::g3d::ResModel* pResModel = pModelAssign->GetResModel();
        pModelAssign->Cleanup();
        DestroyShaderAssign(pResModel);
    }

    for (int idxFile = 0, numFile = pHolder->files.Size();
        idxFile < numFile; ++idxFile)
    {
        nw::g3d::ResFile* pResFile = pHolder->files[idxFile];
        pResFile->Cleanup();
        FreeMem2(pResFile);
    }

    for (int idxShader = 0, numShader = pHolder->shaders.Size();
        idxShader < numShader; ++idxShader)
    {
        nw::g3d::ResShaderArchive* pShaderArchive = pHolder->shaders[idxShader];
        for (int idxShadingModel = 0, numShadingModel = pShaderArchive->GetShadingModelCount();
            idxShadingModel < numShadingModel; ++idxShadingModel)
        {
            DestroyUserShadingModel(pShaderArchive->GetShadingModel(idxShadingModel));
        }
        pShaderArchive->Cleanup();
        FreeMem2(pShaderArchive);
    }

    pHolder->Shutdown();
}

//--------------------------------------------------------------------------------------------------

void SetupMaterials(const SetupMaterialsArg& arg)
{
    nw::g3d::ModelObj* pModelObj = arg.pModelObj;
    for (int idxMat = 0, numMat = pModelObj->GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        nw::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(idxMat);
        nw::g3d::ResMaterial* pResMaterial = pMaterialObj->GetResource();
        MaterialAssign* pMatAssign = pResMaterial->GetUserPtr<MaterialAssign>();
        nw::g3d::ResShadingModel* pResShadingModel = pMatAssign->GetPass(PASS_DEFAULT).pShadingModel;
        nw::g3d::ShaderUtility::InitShaderParam(pMaterialObj, pResShadingModel);

        int idxParamEnv = pMaterialObj->GetShaderParamIndex("texsrt_env");
        if (idxParamEnv >= 0)
        {
            nw::g3d::TexSrtEx* pSrt = pMaterialObj->EditShaderParam<nw::g3d::TexSrtEx>(idxParamEnv);
            pSrt->pEffectMtx = arg.pEnvMtx;
        }
        int idxParamProj = pMaterialObj->GetShaderParamIndex("texsrt_proj");
        if (idxParamProj >= 0)
        {
            nw::g3d::TexSrtEx* pSrt = pMaterialObj->EditShaderParam<nw::g3d::TexSrtEx>(idxParamProj);
            pSrt->pEffectMtx = arg.pProjMtx;
        }
    }
}

}}} // namespace nw::g3d::demo
