﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <g3ddemo_MaterialPick.h>

#include <nw/g3d/fnd/g3d_GfxShader.h>
#include <nw/g3d/fnd/g3d_GX2Utility.h>
#include <nw/g3d/math/g3d_Vector2.h>
#include <g3ddemo_GfxUtility.h>
#include <g3ddemo_ModelUtility.h>

namespace nw { namespace g3d { namespace demo {

void
MaterialPicker::Setup(SetupArg& arg)
{
    NW_G3D_ASSERT_NOT_NULL(arg.pShadingModel);

    m_pResShadingModel = arg.pShadingModel;
    m_ScreenWidth = arg.width;
    m_ScreenHeight = arg.height;

    size_t size = arg.maxModel * sizeof(UserModel*);
    m_ModelList.SetBuffer(AllocMem2(size), size);
}

void
MaterialPicker::Cleanup()
{
    Clear();

    if (void* pBuffer = m_ModelList.GetBuffer())
    {
        FreeMem2(pBuffer);
    }
}

void
MaterialPicker::Entry(UserModel* pUserModel)
{
    NW_G3D_ASSERT_NOT_NULL(pUserModel);
    m_ModelList.PushBack(pUserModel);
}

void
MaterialPicker::UpdateBlock()
{
    for (int idxModel = 0, numModel = m_ModelList.Size(); idxModel < numModel; ++idxModel)
    {
        UserModel* pUserModel = m_ModelList[idxModel];

        nw::g3d::ModelObj* pModelObj = pUserModel->GetModelObj();
        NW_G3D_ASSERT_NOT_NULL(pModelObj);

        for (int idxShape = 0, numShape = pModelObj->GetShapeCount(); idxShape < numShape; ++idxShape )
        {
            nw::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(idxShape);

            nw::g3d::fnd::GfxBuffer& shpBlock = pShapeObj->GetShpBlock(0);
            nw::g3d::ShpBlock* pShpBuffer = static_cast<nw::g3d::ShpBlock*>(shpBlock.GetData());

            // マテリアル情報をシェイプブロックのユーザ領域に書き込みます。
            u32 userModelPtr = reinterpret_cast<u32>(pUserModel);
            u32 matId = pShapeObj->GetMaterialIndex();
            Copy32<false>(&pShpBuffer->userUInt[0], &userModelPtr, sizeof(u32) >> 2);
            Copy32<false>(&pShpBuffer->userUInt[1], &matId, sizeof(u32) >> 2);
            shpBlock.DCFlush();
        }
    }
}

void
MaterialPicker::Erase(UserModel* pUserModel)
{
    NW_G3D_ASSERT_NOT_NULL(pUserModel);

    for (int idxModel = 0, numModel = m_ModelList.Size(); idxModel < numModel; ++idxModel)
    {
        if (m_ModelList[idxModel] == pUserModel)
        {
            m_ModelList.Erase(idxModel);
            break;
        }
    }
}

void
MaterialPicker::Clear()
{
    m_ModelList.Clear();
}

int
MaterialPicker::Pick(
    PickInfo* pickInfo,
    int numMax,
    const nw::g3d::math::Vec2& pos1,
    const nw::g3d::math::Vec2& pos2,
    nw::g3d::GfxContext* pCtx
)
{
    NW_G3D_ASSERT_NOT_NULL(pickInfo);
    NW_G3D_ASSERT_NOT_NULL(pCtx);

    int numModel = m_ModelList.Size();
    int pickedMatNum = 0;

    if (numModel == 0)
    {
        return pickedMatNum;
    }

    nw::g3d::math::Vec2 posTopLeft;
    nw::g3d::math::Vec2 posBottomRight;
    posTopLeft.x = (pos2.x > pos1.x ? pos1.x : pos2.x);
    posTopLeft.y = (pos2.y > pos1.y ? pos1.y : pos2.y);
    posBottomRight.x = (pos2.x > pos1.x ? pos2.x : pos1.x);
    posBottomRight.y = (pos2.y > pos1.y ? pos2.y : pos1.y);

#ifdef _WIN32
    posTopLeft.y = ( m_ScreenHeight - posTopLeft.y );
    posBottomRight.y = ( m_ScreenHeight - posBottomRight.y );
#endif

    static const int MIN_SIZE = 4;
    int width = RoundUp(static_cast<int>(posBottomRight.x - posTopLeft.x), MIN_SIZE);
    int height = RoundUp(static_cast<int>(posBottomRight.y - posTopLeft.y), MIN_SIZE);
    if (width < MIN_SIZE)
    {
        width = MIN_SIZE;
    }
    if (height < MIN_SIZE)
    {
        height = MIN_SIZE;
    }

    FrameBuffer frameBuffer;
    {
        FrameBuffer::InitArg initArg(width, height);
        size_t bufferSize = FrameBuffer::CalcSize(initArg);
        frameBuffer.Init(initArg, AllocMem2(bufferSize), bufferSize);

        GX2Surface& surface = frameBuffer.GetColorBufferTexture(GX2_RENDER_TARGET_0)->texture.GetGX2Texture()->surface;
        surface.format = GX2_SURFACE_FORMAT_TC_R32_G32_UINT;
        surface.tileMode = GX2_TILE_MODE_LINEAR_ALIGNED;

        frameBuffer.Setup();
        frameBuffer.Alloc(AllocMem2);

        frameBuffer.GetViewport().SetViewport(
            -posTopLeft.x,
            -posTopLeft.y,
            static_cast<float>(m_ScreenWidth),
            static_cast<float>(m_ScreenHeight),
            0.01f,
            1.f
        );
    }

    ColorBufferTexture* colorBufTex = frameBuffer.GetColorBufferTexture(GX2_RENDER_TARGET_0);
    DepthBufferTexture* depthBufTex = frameBuffer.GetDepthBufferTexture();
    nw::g3d::GfxColorBuffer& colorBuffer = colorBufTex->renderBuffer;
    nw::g3d::GfxDepthBuffer& depthBuffer = depthBufTex->renderBuffer;
    nw::g3d::CPUCache::Invalidate(colorBufTex->texture.GetBasePtr(), colorBufTex->texture.GetBaseSize());
    nw::g3d::CPUCache::Invalidate(depthBufTex->texture.GetBasePtr(), depthBufTex->texture.GetBaseSize());

    // マテリアルピックを行うために、 UserModel とマテリアルの ID の情報を記録した画像を描画します。
    {
        nw::g3d::ClearBuffers(&colorBuffer, &depthBuffer, 0.0f, 0.0f, 0.0f, 1.0f, GX2_CLEAR_BOTH);

        pCtx->Activate();
        frameBuffer.Load();

        for (int idxModel = 0; idxModel < numModel; ++idxModel)
        {
            UserModel* pUserModel = m_ModelList[idxModel];
            pUserModel->DebugDraw(PASS_MATERIAL_PICK, 0, GX2_SHADER_MODE_UNIFORM_BLOCK);
        }

        nw::g3d::GfxManage::DrawDone();
    }

    colorBufTex->texture.DCRefresh();
    u32* pImage = static_cast<u32*>(colorBufTex->texture.GetImagePtr(0));

#ifdef _WIN32
    const u32 pitch = width;
#else
    const u32 pitch = colorBufTex->texture.GetPitch();
#endif

    // 矩形領域に含まれるマテリアルを検索します。
    for (int h = 0; h < height && pickedMatNum < numMax; ++h)
    {
        for (int w = 0; w < width && pickedMatNum < numMax; ++w)
        {
            UserModel* pUserModel = reinterpret_cast<UserModel*>( pImage[(pitch * h + w) * 2 + 0] );
            u32 matId = pImage[(pitch * h + w) * 2 + 1];

            if (pUserModel == NULL)
            {
                continue;
            }

            bool found = false;
            for (int idxPickedMat = 0; idxPickedMat < pickedMatNum; ++idxPickedMat)
            {
                if (pickInfo[idxPickedMat].userModelPtr == pUserModel &&
                    pickInfo[idxPickedMat].matId == matId)
                {
                    found = true;
                    break;
                }
            }

            if (!found)
            {
                pickInfo[pickedMatNum].userModelPtr = pUserModel;
                pickInfo[pickedMatNum].matId = matId;
                ++pickedMatNum;
            }
        }
    }

    frameBuffer.Free(FreeMem2);
    frameBuffer.Cleanup();
    FreeMem2(frameBuffer.GetBufferPtr());

    return pickedMatNum;
}

}}} // namespace nw::g3d::demo
