﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d.h>
#include <g3ddemo_DemoUtility.h>
#include <g3ddemo_GfxUtility.h>
#include <g3ddemo_UserModel.h>
#include <g3ddemo_ModelUtility.h>

namespace g3ddemo = nw::g3d::demo;

namespace {

enum
{
    // モデルに使用するテクスチャのアライメントの最大値を設定します。
    // リソースファイルごとのアライメントを事前に取得しておくことで、
    // 必要なメモリのアライメントを最小限に抑えてメモリを節約することも可能です。
    TEXTURE_ALIGNMENT   = 8192
};

#define SHADER_PATH "shader/"
#define MODEL_PATH  ""

const char* BFSHA_PATH[] = {
    SHADER_PATH "demo.bfsha",
    NULL
};
const char* BFRES_PATH[] = {
    MODEL_PATH "env.bfres",
    MODEL_PATH "human.bfres",
    NULL
};

nw::g3d::Vec3 s_CameraPosition;
nw::g3d::Vec3 s_CameraUp;
nw::g3d::Vec3 s_CameraTarget;

const int NUM_VIEW = 1;

g3ddemo::ResourceHolder s_Holder;
g3ddemo::ScreenInfo s_Screen;

int s_ModelIndex = 1;
int s_AnimIndex = 1;

g3ddemo::UserView s_View;
nw::g3d::Mtx34 s_TexMtxEnv;
nw::g3d::Mtx34 s_TexMtxProj;

g3ddemo::ModelAssign* SetupModelAssign(nw::g3d::ResModel* pResModel);
g3ddemo::UserModel* SetupUserModel(nw::g3d::ModelObj* pModelObj);

} // anonymous namespace

//--------------------------------------------------------------------------------------------------
int UserModelMain(int argc, const char* argv[])
{
    (void)argc;
    (void)argv;

    // 初期化処理。
    g3ddemo::Init();
    g3ddemo::InitDisplayArg initDisplayArg;
    initDisplayArg.modeDRC = GX2_DRC_NONE;
    g3ddemo::InitDisplay(initDisplayArg);

    g3ddemo::Pad pad;
    if (!pad.Reset())
    {
        g3ddemo::PostQuitMsg();
    }

    // グラフィックスリソースの初期化処理。
    nw::g3d::GfxContext::Prepare(); // 構築に GL コンテクストが必要。
    nw::g3d::GfxContext* pCtx = nw::g3d::demo::AllocMem2<nw::g3d::GfxContext>(
        sizeof(*pCtx), GX2_CONTEXT_STATE_ALIGNMENT);
    pCtx->Setup();
    pCtx->TempPrepare();

    g3ddemo::FrameBuffer frameBuffer;
    {
        g3ddemo::FrameBuffer::InitArg initArg(1280, 720);
        initArg.colorBufferFTV = true;
        size_t bufferSize = g3ddemo::FrameBuffer::CalcSize(initArg);
        frameBuffer.Init(initArg, g3ddemo::AllocMem2(bufferSize), bufferSize);
        frameBuffer.Setup();
        frameBuffer.Alloc(g3ddemo::AllocMem1);
    }

    nw::g3d::GfxColorBuffer& colorBuffer =
        frameBuffer.GetColorBufferTexture(GX2_RENDER_TARGET_0)->renderBuffer;
    nw::g3d::GfxDepthBuffer& depthBuffer = frameBuffer.GetDepthBufferTexture()->renderBuffer;

    s_Screen.Setup(1024);
    s_Screen.SetShapeColor(0x7F, 0x7F, 0x7F, 0x7F);
    s_Screen.SetFontColor(0xFF, 0xFF, 0xFF);
    s_Screen.SetFontSize(2.0f);

    // ビューの Uniform Block
    s_View.Setup();

    s_Holder.Init();

    // シェーダの初期化。
    u32 maxInputRingItem = 0;
    u32 maxOutputRingItem = 0;
    for (const char** path = BFSHA_PATH; *path != NULL; ++path)
    {
        void* pFile = g3ddemo::LoadFile(path[0], NULL, GX2_SHADER_ALIGNMENT);
        NW_G3D_ASSERT(nw::g3d::ResShaderArchive::IsValid(pFile));
        nw::g3d::ResShaderArchive* pShaderArchive = nw::g3d::ResShaderArchive::ResCast(pFile);
        pShaderArchive->Setup();

        for (int idxShadingModel = 0, numShadingModel = pShaderArchive->GetShadingModelCount();
            idxShadingModel < numShadingModel; ++idxShadingModel)
        {
            nw::g3d::ResShadingModel* pShadingModel =
                pShaderArchive->GetShadingModel(idxShadingModel);
            maxInputRingItem = std::max(maxInputRingItem, pShadingModel->GetMaxInputRingItem());
            maxOutputRingItem = std::max(maxOutputRingItem, pShadingModel->GetMaxOutputRingItem());
        }

        s_Holder.shaders.PushBack(pShaderArchive);
    }

    // ジオメトリシェーダ用にリングバッファの設定を行います。
    nw::g3d::GfxRingBuffer ringBuffer;
    {
        size_t inputSize = nw::g3d::GfxRingBuffer::CalcInputBufferSize(maxInputRingItem);
        if (inputSize > 0)
        {
            ringBuffer.SetInputBuffer(g3ddemo::AllocMem1(inputSize, GX2_SHADER_ALIGNMENT), inputSize);
        }
        size_t outputSize = nw::g3d::GfxRingBuffer::CalcOutputBufferSize(maxOutputRingItem);
        if (outputSize > 0)
        {
            ringBuffer.SetOutputBuffer(g3ddemo::AllocMem1(outputSize, GX2_SHADER_ALIGNMENT), outputSize);
        }
        ringBuffer.Load();
    }

    // リソースファイルの初期化。
    for (const char** path = BFRES_PATH; *path != NULL; ++path)
    {
        // ResFile のアライメントは内部に持っているテクスチャの最大アライメントに一致します。
        size_t size = 0;
        void* pFile = g3ddemo::LoadFile(path[0], &size, TEXTURE_ALIGNMENT);
        NW_G3D_ASSERT(nw::g3d::ResFile::IsValid(pFile));
        nw::g3d::ResFile* pResFile = nw::g3d::ResFile::ResCast(pFile);
        pResFile->Setup();
        // ResFile 内のテクスチャや頂点を CPU で書き込んだ場合は
        // CPU のキャッシュを吐き出す必要があります。
        s_Holder.files.PushBack(pResFile);
    }

    // アニメーションリソースの登録
    g3ddemo::RegistAnim(&s_Holder);

    // モデルリソースとシェーダの関連付け。
    for (int idxFile = 0, numFile = s_Holder.files.Size(); idxFile < numFile; ++idxFile)
    {
        nw::g3d::ResFile* pResFile = s_Holder.files[idxFile];
        for (int idxModel = 0, numModel = pResFile->GetModelCount();
            idxModel < numModel; ++idxModel)
        {
            nw::g3d::ResModel* pResModel = pResFile->GetModel(idxModel);

            // シェーダ割り当ての構築。
            g3ddemo::ModelAssign* pModelAssign = SetupModelAssign(pResModel);
            s_Holder.assigns.PushBack(pModelAssign);

            // モデルインスタンスの構築。
            g3ddemo::CreateModelObjArg arg(pResModel);
            arg.initArg.ViewCount(NUM_VIEW);
            nw::g3d::ModelObj* pModelObj = g3ddemo::CreateModelObj(arg);
            g3ddemo::UserModel* pUserModel = SetupUserModel(pModelObj);
            s_Holder.models.PushBack(pUserModel);

            if (0 == strcmp("env", pResModel->GetName()))
            {
                s_Holder.pEnvModel = pUserModel;
            }
        }
    }

    // モデルとアニメーションの関連付け。
    if (s_ModelIndex < s_Holder.models.Size() && s_AnimIndex < s_Holder.modelAnims.Size())
    {
        g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
        void* pResAnim = s_Holder.modelAnims[s_AnimIndex];
        pUserModel->AddAnim(pResAnim);
    }

    // カメラ初期化。
    {
        s_CameraPosition.Set(3.8f, 4.2f, 6.0f);
        s_CameraUp.Set(0.0f, 1.0f, 0.0f);
        s_CameraTarget.Set(0.0f, 2.5f, 0.0f);

        s_View.GetViewMtx(0).LookAt(
            s_CameraPosition,
            s_CameraUp,
            s_CameraTarget);
    }

    bool restart = false;

    // メインループ
    while (g3ddemo::ProcessMsg())
    {
        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            if (pad.IsHold(g3ddemo::Pad::TRIGGER_R))
            {
                restart = true;
            }
            g3ddemo::PostQuitMsg();
        }

        // 計算
        {
            pCtx->TempPrepare();

            // カメラコントロール。
            ControllCamera(&s_CameraPosition, &s_CameraUp, &s_CameraTarget, &s_View.GetViewMtx(0), &pad);
            s_View.GetViewMtx(0).LookAt(s_CameraPosition, s_CameraUp, s_CameraTarget);

            // 投影行列。
            s_View.GetProjMtx(0).Perspective(
                nw::g3d::Math::DegToRad(45.0f), 16.0f / 9.0f, 0.01f, 1000.0f);

            // 環境マップ用行列。
            s_TexMtxEnv.LookAt(
                nw::g3d::Vec3::Make(0.0f, 2.5f, 7.5f),
                nw::g3d::Vec3::Make(0.0f, 1.0f, 0.0f),
                nw::g3d::Vec3::Make(0.0f, 2.5f, 0.0f));
            s_TexMtxEnv.SetT(nw::g3d::Vec3::Make(0.0f, 0.0f, 0.0f));

            // 投影マップ用行列。
            s_TexMtxProj.TexProjPerspective(nw::g3d::Math::DegToRad(45.0f), 1.0f);
            s_TexMtxProj.Mul(s_TexMtxProj, s_View.GetViewMtx(0));

            if (s_ModelIndex < s_Holder.models.Size())
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];

                pUserModel->CalcAnim();
                pUserModel->CalcWorld();
                pUserModel->UpdateFrame();

                pUserModel->UpdateShader(); // 更新に GL コンテクストが必要。
            }
            if (s_Holder.pEnvModel)
            {
                s_Holder.pEnvModel->UpdateShader();
            }
        }

        // 残っているコマンドの完了を待ちます。
        //nw::g3d::GfxManage::DrawDone(); // 直列実行では不要

        // GPU 待ちが必要な計算
        {
            pCtx->TempPrepare(); // 更新に GL コンテクストが必要。

            if (s_ModelIndex < s_Holder.models.Size())
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
                pUserModel->CalcGPU();
            }

            // ビュー毎の更新。
            for (int idxView = 0; idxView < NUM_VIEW; ++idxView)
            {
                s_View.Calc(idxView);
                if (s_ModelIndex < s_Holder.models.Size())
                {
                    g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
                    pUserModel->CalcViewGPU(idxView);
                }
            }

            // 環境
            if (s_Holder.pEnvModel)
            {
                s_Holder.pEnvModel->CalcGPU();
            }
        }

        nw::g3d::CPUCache::Sync(); // CPU キャッシュ操作の完了を待ちます。
        // GPU のリードキャッシュはここで一括して Invalidate します。
        nw::g3d::GPUCache::InvalidateAll();

        // TV 描画。
        {
            nw::g3d::ClearBuffers(&colorBuffer, &depthBuffer,
                0.3f, 0.3f, 0.3f, 1.0f, GX2_CLEAR_BOTH);

            pCtx->Activate(); // Clear に変更されたコンテクストの復帰。
            frameBuffer.Load();

            GX2ShaderMode shaderMode = GX2_SHADER_MODE_UNIFORM_BLOCK;
            nw::g3d::SetShaderMode(shaderMode);

            if (s_ModelIndex < s_Holder.models.Size())
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
                pUserModel->DebugDraw(g3ddemo::PASS_DEFAULT, 0, shaderMode);
            }

            // 情報表示のためにシェーダモードを切り替えます。
            nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_BLOCK);
            g3ddemo::ScreenInfo::LoadState();
            s_Screen.Draw();

            g3ddemo::CopyOut(&colorBuffer, GX2_SCAN_TARGET_TV);
        }

        nw::g3d::GfxManage::DrawDone();

        g3ddemo::SwapScanBuffers();
    }

    // グラフィックスリソースの終了処理。
    pCtx->TempPrepare();

    if (void* ptr = ringBuffer.GetInputBuffer())
    {
        g3ddemo::FreeMem1(ptr);
    }
    if (void* ptr = ringBuffer.GetOutputBuffer())
    {
        g3ddemo::FreeMem1(ptr);
    }

    // エディットライブラリが所有しているモデルを破棄しないように、
    // エディットの終了処理後に残りのデータの破棄を行います。
    g3ddemo::DestroyAll(&s_Holder);

    s_View.Cleanup();

    //
    s_Screen.Cleanup();

    frameBuffer.Free(g3ddemo::FreeMem1);
    frameBuffer.Cleanup();
    g3ddemo::FreeMem2(frameBuffer.GetBufferPtr());

    pCtx->Cleanup();
    g3ddemo::FreeMem2(pCtx);
    pCtx = NULL;

    // 終了処理。
    g3ddemo::ShutdownDisplay();
    g3ddemo::Shutdown();

    return restart ? -1 : EXIT_SUCCESS;
}

namespace {

g3ddemo::ModelAssign* SetupModelAssign(nw::g3d::ResModel* pResModel)
{
    g3ddemo::CreateShaderAssign(pResModel);
    g3ddemo::ModelAssign* pModelAssign = pResModel->GetUserPtr<g3ddemo::ModelAssign>();

    // モデルに記録されたデフォルトシェーダの関連付け。
    nw::g3d::BindResult result = nw::g3d::BindResult::NotBound();
    for (int idxShader = 0, numShader = s_Holder.shaders.Size();
        !result.IsComplete() && idxShader < numShader; ++idxShader)
    {
        nw::g3d::ResShaderArchive* pShaderArchive = s_Holder.shaders[idxShader];
        result = pModelAssign->BindShader(g3ddemo::PASS_DEFAULT, pShaderArchive);
    }
    //NW_G3D_ASSERT(result.IsComplete());

    pModelAssign->Setup();
    return pModelAssign;
}

g3ddemo::UserModel* SetupUserModel(nw::g3d::ModelObj* pModelObj)
{
    g3ddemo::CreateUserModel(pModelObj);
    g3ddemo::UserModel* pUserModel = pModelObj->GetUserPtr<g3ddemo::UserModel>();
    pUserModel->SetView(&s_View);
    if (s_Holder.pEnvModel)
    {
        pUserModel->SetEnv(s_Holder.pEnvModel->GetModelObj());
    }

    for (int idxMat = 0, numMat = pModelObj->GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        nw::g3d::MaterialObj* pMaterialObj = pModelObj->GetMaterial(idxMat);
        int idxParamEnv = pMaterialObj->GetShaderParamIndex("texsrt_env");
        if (idxParamEnv >= 0)
        {
            nw::g3d::TexSrtEx* pSrt = pMaterialObj->EditShaderParam<nw::g3d::TexSrtEx>(idxParamEnv);
            pSrt->pEffectMtx = &s_TexMtxEnv;
        }
        int idxParamProj = pMaterialObj->GetShaderParamIndex("texsrt_proj");
        if (idxParamProj >= 0)
        {
            nw::g3d::TexSrtEx* pSrt = pMaterialObj->EditShaderParam<nw::g3d::TexSrtEx>(idxParamProj);
            pSrt->pEffectMtx = &s_TexMtxProj;
        }
    }

    // アニメーションインスタンスの構築。
    g3ddemo::UserModel::CreateAnimObjArg createAnimObjArg;
    if (!s_Holder.modelAnims.Empty())
    {
        createAnimObjArg.Reserve(&s_Holder.modelAnims[0], s_Holder.modelAnims.Size());
    }
    pUserModel->CreateAnimObj(createAnimObjArg);

    return pUserModel;
}

} // anonymous namespace
