﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d.h>
#include <g3ddemo_DemoUtility.h>
#include <g3ddemo_GfxUtility.h>
#include <g3ddemo_UserModel.h>
#include <g3ddemo_ModelUtility.h>
#include <g3ddemo_GPUMetric.h>

namespace g3ddemo = nw::g3d::demo;

namespace {

enum
{
    // モデルに使用するテクスチャのアライメントの最大値を設定します。
    // リソースファイルごとのアライメントを事前に取得しておくことで、
    // 必要なメモリのアライメントを最小限に抑えてメモリを節約することも可能です。
    TEXTURE_ALIGNMENT   = 8192
};

#define SHADER_PATH "shader/"
#define MODEL_PATH  ""

const char* BFSHA_PATH[] = {
    SHADER_PATH "demo.bfsha",
    SHADER_PATH "streamout.bfsha",
    NULL
};
const char* BFRES_PATH[] = {
    MODEL_PATH "env.bfres",
    MODEL_PATH "boy.bfres",
    NULL
};
const int NUM_VIEW = 1;
const int NUM_COPY_MODEL = 1000;
const float GRID_DISTANCE = 200.0f;
const float LOD_THRESHOLD = 0.1f;
const int s_EnvIndex = 0;
const int s_DrawIndex = 1;
const int s_AnimIndex = 0;
const int MAX_MODEL_LOD = 2;
const float MODEL_LOD_THRESHOLDS[MAX_MODEL_LOD] = { 1.0f, 0.2f };
const nw::g3d::Vec3 CAMERA_POSITION = nw::g3d::Vec3::Make(0.0f, 100.0f, 3100.0f);
const nw::g3d::Vec3 CAMERA_UP = nw::g3d::Vec3::Make(0.0f, 1.0f, 0.0f);
const nw::g3d::Vec3 CAMERA_TARGET = nw::g3d::Vec3::Make(0.0f, 0.0f, 0.0f);

// 再実行される場合を考えてここで初期化しません。
bool s_UseCamera;
nw::g3d::Vec3 s_CameraPosition;
nw::g3d::Vec3 s_CameraUp;
nw::g3d::Vec3 s_CameraTarget;
float s_DoubleZnear;
u32 s_Counter;

nw::g3d::Vec3 s_LayoutTranslate[NUM_COPY_MODEL];

g3ddemo::ResourceHolder s_Holder;
g3ddemo::ScreenInfo s_Screen;
g3ddemo::ProcessMeter s_Meter;

g3ddemo::UserView s_View;
nw::g3d::Mtx34 s_TexMtxEnv;
nw::g3d::Mtx34 s_TexMtxProj;

g3ddemo::ModelAssign* SetupModelAssign(nw::g3d::ResModel* pResModel);
g3ddemo::UserModel* SetupUserModel(nw::g3d::ModelObj* pModelObj);
void CalcMenu(const g3ddemo::Pad &pad);
void MakeUserModelDL();
void ChangeStreamOut();
void ChangeLockedCache();
void ChangeBakedAnimation();

enum UseFlag
{
    USE_NOTHING                 = 0,
    USE_DISPLAY_LIST            = 0x1 << 0,
    USE_MULTI_CORE              = 0x1 << 1,
    USE_STREAMOUT               = 0x1 << 2,
    USE_LOCKED_CACHE            = 0x1 << 3,
    USE_GPU_THREAD_OPTIMIZATION = 0x1 << 4,
    USE_ANIMATION_LOD           = 0x1 << 5,
    USE_BAKE_ANIMATION          = 0x1 << 6,
    USE_NOT_DRAW_PIXEL          = 0x1 << 7,
    USE_MODEL_LOD               = 0x1 << 8,

#ifdef _WIN32
    INITIAL_FLAG                = USE_NOTHING
#else
    INITIAL_FLAG                = USE_DISPLAY_LIST            |
                                  USE_MULTI_CORE              |
                                  USE_LOCKED_CACHE            |
                                  USE_GPU_THREAD_OPTIMIZATION |
                                  USE_BAKE_ANIMATION

#endif
};

bit32 s_UseFlag;
bool s_ChangeDrawCommand;
bool s_PerfEnabled;

enum
{
    SUB_DL_1,
    MAIN_DL,
    SUB_DL_2,
    NUM_DL,
};

enum
{
    READY_DL,
    FLUSH_DL,
    INFO_DL,
    NUM_SETUP_DL
};

// 各コア用のディスプレイリストです。
g3ddemo::DisplayList s_SetupDL[NUM_SETUP_DL];
g3ddemo::DisplayList s_StreamOutDL[NUM_DL];
g3ddemo::DisplayList s_DrawDL[NUM_DL];

// UserModel 用のディスプレイリストです。
g3ddemo::DisplayList s_UserModelDL[NUM_COPY_MODEL][MAX_MODEL_LOD];

const u32 s_Sub1CoreId = 0;
const u32 s_Sub2CoreId = 2;

GX2ShaderMode s_ShaderMode;
bool s_IsExit;
bool s_IsDrawTask;
bool s_IsStreamOutTask;

#ifndef _WIN32

// マルチコアで処理分散を行うための実装
// 出来る限り複数のコアに分散して CPU 処理を行います。

MPTaskQ s_TaskQueue;
MPTaskPtr s_TaskQueueArray[NUM_COPY_MODEL];
MPTask    s_TaskArray[NUM_COPY_MODEL];
OSRendezvous s_Rendezvous;
OSRendezvous s_Rendezvous2;
OSRendezvous s_Rendezvous3;

static void ExcuteDrawTask(int core)
{
    // コアごとにディスプレイリストへコマンドを積みこみます。
    s_DrawDL[core].Begin();

    // execute the task
    MPRunTasksFromTaskQ(&s_TaskQueue, 1);

    s_DrawDL[core].End();
}

static void ExcuteStreamOutTask(int core)
{
    // パスが分かれている都合上、
    // ストリームアウト用のディスプレイリストは別にしておきます。
    s_StreamOutDL[core].Begin();

    nw::g3d::SetStreamOutEnable(GX2_TRUE);
    nw::g3d::SetRasterizerClipControl(GX2_FALSE, GX2_TRUE);

    // execute the task
    MPRunTasksFromTaskQ(&s_TaskQueue, 1);

    nw::g3d::SetStreamOutEnable(GX2_FALSE);
    nw::g3d::SetRasterizerClipControl(GX2_TRUE, GX2_TRUE);

    s_StreamOutDL[core].End();
}

static void ExcuteCalcTask(int core)
{
    // execute the task
    MPRunTasksFromTaskQ(&s_TaskQueue, 1);
}

static void ExecuteTask()
{
    if (OSIsMainCore())
    {
        OSInitRendezvous(&s_Rendezvous2);
    }
    OSWaitRendezvous(&s_Rendezvous, OS_WAIT_CORE_ALL);

    // wait till the state of task queue is MP_TASKQ_RUN
    MPWaitTaskQ(&s_TaskQueue, MP_WAIT_TASKQ_RUN);

    if (OSIsMainCore())
    {
        OSInitRendezvous(&s_Rendezvous3);
    }
    OSWaitRendezvous(&s_Rendezvous2, OS_WAIT_CORE_ALL);

    int core = OSGetCoreId();

    int coreId = static_cast<int>(g3ddemo::ProcessMeter::GROUP_CPU0) + core;
    int idxCore = s_Meter.BeginTimeSpan(static_cast<g3ddemo::ProcessMeter::Group>(coreId), 0, 0x88, 0x88);

    // タスクの種類によって実行する関数を変えます。
    if (s_IsDrawTask)
    {
        ExcuteDrawTask(core);
    }
    else if (s_IsStreamOutTask)
    {
        ExcuteStreamOutTask(core);
    }
    else
    {
        ExcuteCalcTask(core);
    }

    s_Meter.EndTimeSpan(idxCore);

    MPWaitTaskQ(&s_TaskQueue, MP_WAIT_TASKQ_DONE);

    if (OSIsMainCore())
    {
        OSInitRendezvous(&s_Rendezvous);
    }
    OSWaitRendezvous(&s_Rendezvous3, OS_WAIT_CORE_ALL);
}

static int SubMain(int intArg, void *ptrArg)
{
    while (!s_IsExit)
    {
        ExecuteTask();
    }

    return 0;
}
#endif

static float CalcProjectionRatio(const nw::g3d::Sphere* sphere)
{
    nw::g3d::Mtx34& view = s_View.GetViewMtx(0);
    float viewZ = view.m20 * sphere->center.x + view.m21 * sphere->center.y + view.m22 * sphere->center.z + view.m23;

    // バウンディングスフィアをカメラ投影したときの割合を計算します。
    return (viewZ != 0.0f) ? nw::g3d::math::Math::Abs(sphere->radius * s_DoubleZnear / viewZ) : 1.0f;
}

static void CalcUserModel(void *data, u32 index)
{
    g3ddemo::UserModel* pUserModel = static_cast<g3ddemo::UserModel*>(data);

    // カメラから遠いアニメーションの計算は半分のフレームで計算します。
    bool halfFrame = false;
    float ratio = -1.0f;
    const nw::g3d::Sphere* sphere = pUserModel->GetModelObj()->GetBounding();
    pUserModel->SetLODLevel(0, 0);

    if (sphere)
    {
        if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_ANIMATION_LOD) && (s_Counter % 2) == (index % 2))
        {
            if (( ratio = CalcProjectionRatio(sphere) ) < LOD_THRESHOLD)
            {
                halfFrame = true;
            }
        }

        if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_MODEL_LOD))
        {
            if (ratio < 0.0f)
            {
                ratio = CalcProjectionRatio(sphere);
            }
            for (int idxLod = 1; idxLod < MAX_MODEL_LOD; ++idxLod)
            {
                if (ratio < MODEL_LOD_THRESHOLDS[idxLod])
                {
                    pUserModel->SetLODLevel(0, idxLod);
                }
            }
        }
    }

    if (!halfFrame)
    {
        nw::g3d::Mtx34& layoutMtx = pUserModel->GetLayoutMtx();
        layoutMtx.SetT(s_LayoutTranslate[index]);
        pUserModel->CalcAnim();
        pUserModel->CalcWorld();
    }
    pUserModel->UpdateFrame();
    pUserModel->UpdateShader();
}

static void CalcGPUUserModel(void *data, u32 index)
{
    g3ddemo::UserModel* pUserModel = static_cast<g3ddemo::UserModel*>(data);

    const nw::g3d::Sphere* sphere = pUserModel->GetModelObj()->GetBounding();
    if (sphere)
    {
        if (!s_View.GetViewVolume(0).TestIntersection(*sphere))
        {
            return;
        }

        if (((s_Counter % 2) == (index % 2)) && (CalcProjectionRatio(sphere) < LOD_THRESHOLD))
        {
            return;
        }
    }

    pUserModel->CalcGPU();
    // ビュー用毎の更新。
    for (int idxView = 0; idxView < NUM_VIEW; ++idxView)
    {
        pUserModel->CalcViewGPU(idxView);
    }
}

static void DrawUserModel(void *data, u32 index)
{
    g3ddemo::UserModel* pUserModel = static_cast<g3ddemo::UserModel*>(data);
    const nw::g3d::Sphere* sphere = pUserModel->GetModelObj()->GetBounding();
    if (sphere && !nw::g3d::ut::CheckFlag(s_UseFlag, USE_NOT_DRAW_PIXEL))
    {
        // モデル単位でフラスタムカリングのテストを行います。
        // テストに失敗した場合は描画しません。
        // 必要に応じてシェイプや地形ではサブメッシュ単位で判定することもできます。
        if (!s_View.GetViewVolume(0).TestIntersection(*sphere))
        {
            return;
        }
    }

    if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST))
    {
        s_UserModelDL[index][pUserModel->GetLODLevel(0)].Call();
    }
    else
    {
        pUserModel->DebugDraw(g3ddemo::PASS_DEFAULT, 0, s_ShaderMode);
    }
}

static void StreamOutUserModel(void *data, u32 index)
{
    (void)index;

    g3ddemo::UserModel* pUserModel = static_cast<g3ddemo::UserModel*>(data);
    pUserModel->StreamOut(s_ShaderMode);
}

void RunTaskQueue(void (*func)(void*, u32))
{
#ifndef _WIN32
    // 毎フレームタスクの作成を行います。
    // 決まったタスクを毎フレームを行うのであれば MPResetTaskQ を用いて処理負荷を軽減できます。
    MPInitTaskQ(&s_TaskQueue, s_TaskQueueArray, NUM_COPY_MODEL);
    for (int idxModel = 0 ; idxModel < NUM_COPY_MODEL; ++idxModel)
    {
        MPTask *t = &s_TaskArray[idxModel];
        MPInitTask(t, reinterpret_cast<void*>(func),
            static_cast<void*>(s_Holder.models[idxModel]), idxModel);
        MPEnqueTask(&s_TaskQueue, t);
    }

    // TaskQ を実行状態(RUN)にします。
    MPStartTaskQ(&s_TaskQueue);

    ExecuteTask();

    MPTermTaskQ(&s_TaskQueue);
#else
    (void)func;
#endif
}

} // anonymous namespace

//--------------------------------------------------------------------------------------------------
int ParallelMain(int argc, const char* argv[])
{
    (void)argc;
    (void)argv;

    // 初期化処理。
    g3ddemo::Init();
    g3ddemo::InitDisplayArg initDisplayArg;
    initDisplayArg.modeDRC = GX2_DRC_NONE;
    g3ddemo::InitDisplay(initDisplayArg);

    g3ddemo::Pad pad;
    if (!pad.Reset())
    {
        g3ddemo::PostQuitMsg();
    }

    // グラフィックスリソースの初期化処理。
    nw::g3d::GfxContext::Prepare(); // 構築に GL コンテクストが必要。
    nw::g3d::GfxContext* pCtx = nw::g3d::demo::AllocMem2<nw::g3d::GfxContext>(
        sizeof(*pCtx), GX2_CONTEXT_STATE_ALIGNMENT);
    pCtx->Setup();
    pCtx->TempPrepare();

    g3ddemo::FrameBuffer frameBuffer;
    {
        g3ddemo::FrameBuffer::InitArg initArg(1280, 720);
        initArg.colorBufferFTV = true;
        size_t bufferSize = g3ddemo::FrameBuffer::CalcSize(initArg);
        frameBuffer.Init(initArg, g3ddemo::AllocMem2(bufferSize), bufferSize);
        frameBuffer.Setup();
        frameBuffer.Alloc(g3ddemo::AllocMem1);
    }

    nw::g3d::GfxColorBuffer& colorBuffer =
        frameBuffer.GetColorBufferTexture(GX2_RENDER_TARGET_0)->renderBuffer;
    nw::g3d::GfxDepthBuffer& depthBuffer = frameBuffer.GetDepthBufferTexture()->renderBuffer;

    s_Screen.Setup(1024);
    s_Screen.SetShapeColor(0x7F, 0x7F, 0x7F, 0x7F);
    s_Screen.SetFontColor(0xFF, 0xFF, 0xFF);
    s_Screen.SetFontSize(2.0f);

    // ビューの Uniform Block
    s_View.Setup();

    s_Holder.Init();

    // シェーダの初期化。
    u32 optimizeGPRs[3] = { 0 };
    u32 optimizeEntries[3] = { 0 };
    u32 maxInputRingItem = 0;
    u32 maxOutputRingItem = 0;
    {
        u32 maxGPRs[3] = { 0 };
        u32 maxEntries[3] = { 0 };
        for (const char** path = BFSHA_PATH; *path != NULL; ++path)
        {
            void* pFile = g3ddemo::LoadFile(path[0], NULL, GX2_SHADER_ALIGNMENT);
            NW_G3D_ASSERT(nw::g3d::ResShaderArchive::IsValid(pFile));
            nw::g3d::ResShaderArchive* pShaderArchive = nw::g3d::ResShaderArchive::ResCast(pFile);
            pShaderArchive->Setup();

            for (int idxShadingModel = 0, numShadingModel = pShaderArchive->GetShadingModelCount();
                idxShadingModel < numShadingModel; ++idxShadingModel)
            {
                nw::g3d::ResShadingModel* pShadingModel =
                    pShaderArchive->GetShadingModel(idxShadingModel);
                maxInputRingItem = std::max(maxInputRingItem, pShadingModel->GetMaxInputRingItem());
                maxOutputRingItem = std::max(maxOutputRingItem, pShadingModel->GetMaxOutputRingItem());

                // GPRs と StackEntiries の最大値を収集します。
                for (int stage = 0; stage <= nw::g3d::ResShaderProgram::STAGE_FRAGMENT; ++stage)
                {
                    maxGPRs[stage] = std::max(maxGPRs[stage], pShadingModel->GetMaxGPRs(static_cast<nw::g3d::ResShaderProgram::Stage>(stage)));
                    maxEntries[stage] = std::max(maxEntries[stage], pShadingModel->GetMaxStackEntries(static_cast<nw::g3d::ResShaderProgram::Stage>(stage)));
                }
            }

            s_Holder.shaders.PushBack(pShaderArchive);

            if (s_Holder.pStreamoutArchive == NULL &&
                0 == strcmp("streamout", pShaderArchive->GetName()))
            {
                s_Holder.pStreamoutArchive = pShaderArchive;
            }
        }
        // 最低限必要な数以外の余剰数を求めます。
        u32 sumGPRs = 0;
        u32 sumEntries = 0;
        for (int stage = 0; stage <= nw::g3d::ResShaderProgram::STAGE_FRAGMENT; ++stage)
        {
            sumGPRs += maxGPRs[stage];
            sumEntries += maxEntries[stage];
        }
        u32 allowanceGPRs = GX2_TOTAL_GPRS_GS_DISABLED - sumGPRs;
        u32 allowanceEntries = GX2_TOTAL_STACK_ENTRIES - sumEntries;

        // それぞれの stage の比率によって GPRs と Entries の数を計算します。
        optimizeGPRs[nw::g3d::ResShaderProgram::STAGE_VERTEX] = maxGPRs[nw::g3d::ResShaderProgram::STAGE_VERTEX] +
            static_cast<u32>((static_cast<float>(maxGPRs[nw::g3d::ResShaderProgram::STAGE_VERTEX]) / static_cast<float>(sumGPRs)) * allowanceGPRs);
        optimizeEntries[nw::g3d::ResShaderProgram::STAGE_VERTEX] =
            static_cast<u32>((static_cast<float>(maxEntries[nw::g3d::ResShaderProgram::STAGE_VERTEX]) / static_cast<float>(sumEntries)) * allowanceEntries);
        optimizeGPRs[nw::g3d::ResShaderProgram::STAGE_GEOMETRY] = maxEntries[nw::g3d::ResShaderProgram::STAGE_VERTEX] +
            static_cast<u32>((static_cast<float>(maxGPRs[nw::g3d::ResShaderProgram::STAGE_GEOMETRY]) / static_cast<float>(sumGPRs)) * allowanceGPRs);
        optimizeEntries[nw::g3d::ResShaderProgram::STAGE_GEOMETRY] = maxGPRs[nw::g3d::ResShaderProgram::STAGE_GEOMETRY] +
            static_cast<u32>((static_cast<float>(maxEntries[nw::g3d::ResShaderProgram::STAGE_GEOMETRY]) / static_cast<float>(sumEntries)) * allowanceEntries);
        optimizeGPRs[nw::g3d::ResShaderProgram::STAGE_FRAGMENT] = maxEntries[nw::g3d::ResShaderProgram::STAGE_GEOMETRY] +
            GX2_TOTAL_GPRS_GS_DISABLED - optimizeGPRs[nw::g3d::ResShaderProgram::STAGE_VERTEX] - optimizeGPRs[nw::g3d::ResShaderProgram::STAGE_GEOMETRY];
        optimizeEntries[nw::g3d::ResShaderProgram::STAGE_FRAGMENT] =
            GX2_TOTAL_STACK_ENTRIES - optimizeEntries[nw::g3d::ResShaderProgram::STAGE_VERTEX] - optimizeEntries[nw::g3d::ResShaderProgram::STAGE_GEOMETRY];
    }

    // ジオメトリシェーダ用にリングバッファの設定を行います。
    nw::g3d::GfxRingBuffer ringBuffer;
    {
        size_t inputSize = nw::g3d::GfxRingBuffer::CalcInputBufferSize(maxInputRingItem);
        if (inputSize > 0)
        {
            ringBuffer.SetInputBuffer(g3ddemo::AllocMem1(inputSize, GX2_SHADER_ALIGNMENT), inputSize);
        }
        size_t outputSize = nw::g3d::GfxRingBuffer::CalcOutputBufferSize(maxOutputRingItem);
        if (outputSize > 0)
        {
            ringBuffer.SetOutputBuffer(g3ddemo::AllocMem1(outputSize, GX2_SHADER_ALIGNMENT), outputSize);
        }
        ringBuffer.Load();
    }

    // リソースファイルの初期化。
    for (const char** path = BFRES_PATH; *path != NULL; ++path)
    {
        // ResFile のアライメントは内部に持っているテクスチャの最大アライメントに一致します。
        size_t size = 0;
        void* pFile = g3ddemo::LoadFile(path[0], &size, TEXTURE_ALIGNMENT);
        NW_G3D_ASSERT(nw::g3d::ResFile::IsValid(pFile));
        nw::g3d::ResFile* pResFile = nw::g3d::ResFile::ResCast(pFile);
        pResFile->Setup();
        // ResFile 内のテクスチャや頂点を CPU で書き込んだ場合は
        // CPU のキャッシュを吐き出す必要があります。
        s_Holder.files.PushBack(pResFile);
    }

    // アニメーションリソースの登録
    g3ddemo::RegistAnim(&s_Holder);

    // モデルリソースとシェーダの関連付け。
    for (int idxFile = 0, numFile = s_Holder.files.Size(); idxFile < numFile; ++idxFile)
    {
        nw::g3d::ResFile* pResFile = s_Holder.files[idxFile];
        for (int idxModel = 0, numModel = pResFile->GetModelCount();
            idxModel < numModel; ++idxModel)
        {
            nw::g3d::ResModel* pResModel = pResFile->GetModel(idxModel);

            // シェーダ割り当ての構築。
            g3ddemo::ModelAssign* pModelAssign = SetupModelAssign(pResModel);
            s_Holder.assigns.PushBack(pModelAssign);
        }
    }

    // env モデル構築。
    if (s_EnvIndex < s_Holder.assigns.Size())
    {
        g3ddemo::ModelAssign* pModelAssign = s_Holder.assigns[s_EnvIndex];
        nw::g3d::ResModel* pResModel = pModelAssign->GetResModel();

        // モデルインスタンスの構築。
        g3ddemo::CreateModelObjArg arg(pResModel);
        arg.initArg.ViewCount(NUM_VIEW);
        nw::g3d::ModelObj* pModelObj = g3ddemo::CreateModelObj(arg);
        g3ddemo::UserModel* pUserModel = SetupUserModel(pModelObj);

        s_Holder.pEnvModel = pUserModel;
    }

    // 描画モデル構築
    if (s_DrawIndex < s_Holder.assigns.Size())
    {
        g3ddemo::ModelAssign* pModelAssign = s_Holder.assigns[s_DrawIndex];
        nw::g3d::ResModel* pResModel = pModelAssign->GetResModel();

        for (int idxCopy = 0; idxCopy < NUM_COPY_MODEL; ++idxCopy)
        {
            // モデルインスタンスの構築。
            g3ddemo::CreateModelObjArg arg(pResModel);
            arg.initArg.ViewCount(NUM_VIEW);
            arg.initArg.EnableBounding();
            nw::g3d::ModelObj* pModelObj = g3ddemo::CreateModelObj(arg);
            g3ddemo::UserModel* pUserModel = SetupUserModel(pModelObj);
            s_Holder.models.PushBack(pUserModel);
        }
    }

    // モデルとアニメーションの関連付け。
    for (int idxModel = 0, numModel = s_Holder.models.Size();
        idxModel < numModel; ++idxModel)
    {
        if (s_AnimIndex < s_Holder.modelAnims.Size())
        {
            g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
            void* pResAnim = s_Holder.modelAnims[s_AnimIndex];
            pUserModel->AddAnim(pResAnim);

            // スケルタルアニメーションに限定
            nw::g3d::ResSkeletalAnim* pResSkeletalAnim = static_cast<nw::g3d::ResSkeletalAnim*>(pResAnim);
            float frame = pResSkeletalAnim->GetFrameCount() * (static_cast<float>(rand()) / static_cast<float>(RAND_MAX));
            pUserModel->SetFrame(frame);
        }
    }

    // レイアウト計算。
    int matrixSize = static_cast<int>(nw::g3d::math::Math::Sqrt(static_cast<float>(s_Holder.models.Size())));
    float center = (GRID_DISTANCE * matrixSize) / 2;
    int xGrid = 0, zGrid = 0;
    for (int idxModel = 0, numModel = s_Holder.models.Size();
        idxModel != numModel;
        ++idxModel)
    {
        float randomX = (static_cast<float>(rand()) / static_cast<float>(RAND_MAX)) - 0.5f;
        float xStir = randomX * 0.5f * GRID_DISTANCE;

        float randomZ = (static_cast<float>(rand()) / static_cast<float>(RAND_MAX)) - 0.5f;
        float zStir = randomZ * 0.5f * GRID_DISTANCE;

        s_LayoutTranslate[idxModel].Set(xGrid * GRID_DISTANCE - center + xStir, 0.0f, zGrid *GRID_DISTANCE - center + zStir);
        ++xGrid;

        if (xGrid == matrixSize)
        {
            xGrid = 0;
            ++zGrid;
        }
    }

    // カメラ初期化。
    s_UseCamera = false;
    s_CameraPosition.Set(CAMERA_POSITION);
    s_CameraUp.Set(CAMERA_UP);
    s_CameraTarget.Set(CAMERA_TARGET);

    s_View.GetViewMtx(0).LookAt(
        s_CameraPosition,
        s_CameraUp,
        s_CameraTarget);

    bool restart = false;

    s_ChangeDrawCommand = true;
    s_PerfEnabled = false;
    s_UseFlag = INITIAL_FLAG;

    // ディスプレイリストの初期化を行います。
    for (int i = 0; i < NUM_SETUP_DL; ++i)
    {
        s_SetupDL[i].Setup(2, 0x4000);
    }
    for (int i = 0; i < NUM_DL; ++i)
    {
        s_StreamOutDL[i].Setup(2);
    }
    for (int i = 0; i < NUM_DL; ++i)
    {
        s_DrawDL[i].Setup(2);
    }
    for (int i = 0; i < NUM_COPY_MODEL; ++i)
    {
        for (int j = 0; j < MAX_MODEL_LOD; ++j)
        {
            s_UserModelDL[i][j].Setup(2, 0x8000);
        }
    }

    s_Meter.Setup();

#ifndef _WIN32
    // サブコア起動。
    {
        s_IsExit = false;
        s_IsDrawTask = false;
        s_IsStreamOutTask = false;

        OSInitRendezvous(&s_Rendezvous);
        OSInitRendezvous(&s_Rendezvous2);
        OSInitRendezvous(&s_Rendezvous3);

        // サブコアは無限ループでタスクを処理し続けます。
        OSThread* defaultSub1 = OSGetDefaultThread(s_Sub1CoreId);
        OSRunThread(defaultSub1,
            SubMain,
            0,
            NULL);

        OSThread* defaultSub2 = OSGetDefaultThread(s_Sub2CoreId);
        OSRunThread(defaultSub2,
            SubMain,
            0,
            NULL);
    }
#endif
    g3ddemo::GPUMetric gpuMetric;
    gpuMetric.Setup();
    // フラグに応じて UserModel の状態を変更します。
    if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_STREAMOUT))
    {
        ChangeStreamOut();
    }
    if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_LOCKED_CACHE))
    {
        ChangeLockedCache();
    }
    if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_BAKE_ANIMATION))
    {
        ChangeBakedAnimation();
    }

    // このデモでは CopyOut 後に Clear を行っているので最初に一度 Clear します。
    nw::g3d::ClearBuffers(&colorBuffer, &depthBuffer,
        0.3f, 0.3f, 0.3f, 1.0f, GX2_CLEAR_BOTH);

    // メインループ
    s_Counter = 0;
    while (g3ddemo::ProcessMsg())
    {
        pCtx->TempPrepare(); // ProcessMeter の登録に GL コンテクストが必要。
        int idxFrame = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_FRAME);

        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            if (pad.IsHold(g3ddemo::Pad::TRIGGER_R))
            {
                restart = true;
            }
            g3ddemo::PostQuitMsg();
        }

        CalcMenu(pad);

        if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST))
        {
            // ディスプレイリストの描画キックを行います。
            // フレームの先頭を呼び出すことで GPU をフルに用いて実行させることができます。
            s_SetupDL[READY_DL].DirectCall();
            for (int core = 0; core < NUM_DL; ++core)
            {
                s_StreamOutDL[core].DirectCall();
            }
            s_SetupDL[FLUSH_DL].DirectCall();
            for (int core = 0; core < NUM_DL; ++core)
            {
                s_DrawDL[core].DirectCall();
            }
            s_SetupDL[INFO_DL].DirectCall();
        }

        // 計算
        {
            int idxCalcCPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU);

            float zNear = 10.0;
            float zFar = 20000.0f;
            if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_NOT_DRAW_PIXEL))
            {
                zNear = zFar - 0.1f; // デバッグ用。
            }

            // カメラコントロール
            if (s_UseCamera)
            {
                ControllCamera(&s_CameraPosition, &s_CameraUp, &s_CameraTarget, &s_View.GetViewMtx(0), &pad);
                s_View.GetViewMtx(0).LookAt(s_CameraPosition, s_CameraUp, s_CameraTarget);
            }

            // 投影行列。
            s_View.GetProjMtx(0).Perspective(
                nw::g3d::Math::DegToRad(45.0f), 16.0f / 9.0f, zNear, zFar);

            // フラスタムカリング用の ViewVolume 計算。
            nw::g3d::Mtx34 inverse;
            float det;
            inverse.Inverse(&det, s_View.GetViewMtx(0));
            s_View.GetViewVolume(0).SetPerspective(
                nw::g3d::Math::DegToRad(45.0f), 16.0f / 9.0f, zNear, zFar, inverse);

            // 環境マップ用行列。
            s_TexMtxEnv.LookAt(
                nw::g3d::Vec3::Make(0.0f, 2.5f, 7.5f),
                nw::g3d::Vec3::Make(0.0f, 1.0f, 0.0f),
                nw::g3d::Vec3::Make(0.0f, 2.5f, 0.0f));
            s_TexMtxEnv.SetT(nw::g3d::Vec3::Make(0.0f, 0.0f, 0.0f));

            // 投影マップ用行列。
            s_TexMtxProj.TexProjPerspective(nw::g3d::Math::DegToRad(45.0f), 1.0f);
            s_TexMtxProj.Mul(s_TexMtxProj, s_View.GetViewMtx(0));

            s_DoubleZnear = 1.0f / nw::g3d::math::Math::Tan(nw::g3d::Math::DegToRad(45.0f) * 0.5f);

            if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_MULTI_CORE))
            {
                // UserModel ごとの計算処理をタスクにしてマルチコアに分散させます。
                RunTaskQueue(CalcUserModel);
            }
            else
            {
                for (int idxModel = 0, numModel = s_Holder.models.Size();
                    idxModel != numModel;
                    ++idxModel)
                {
                    g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
                    CalcUserModel(pUserModel, idxModel);
                }
            }

            if (s_Holder.pEnvModel)
            {
                s_Holder.pEnvModel->UpdateShader();
            }

            s_Meter.EndTimeSpan(idxCalcCPU);
        }

        // TV 描画。
        {
            if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST))
            {
                // メイングラフィクスコア用のディスプレイリストの作成を開始します。
                s_SetupDL[READY_DL].Begin();
            }

            gpuMetric.Begin();

            // GPU のリードキャッシュはここで一括して Invalidate します。
            nw::g3d::GPUCache::InvalidateAll();

            pCtx->Activate(); // Clear に変更されたコンテクストの復帰。
            frameBuffer.Load();

            s_ShaderMode = GX2_SHADER_MODE_UNIFORM_BLOCK;

            if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_GPU_THREAD_OPTIMIZATION))
            {
                nw::g3d::SetShaderModeEx(
                    s_ShaderMode,
                    optimizeGPRs[nw::g3d::ResShaderProgram::STAGE_VERTEX],
                    optimizeEntries[nw::g3d::ResShaderProgram::STAGE_VERTEX],
                    optimizeGPRs[nw::g3d::ResShaderProgram::STAGE_GEOMETRY],
                    optimizeEntries[nw::g3d::ResShaderProgram::STAGE_GEOMETRY],
                    optimizeGPRs[nw::g3d::ResShaderProgram::STAGE_FRAGMENT],
                    optimizeEntries[nw::g3d::ResShaderProgram::STAGE_FRAGMENT]);
            }
            else
            {
                nw::g3d::SetShaderMode(s_ShaderMode);
            }

            int idxGPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_GPU, nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST));

            if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST))
            {
                s_SetupDL[READY_DL].End();
            }

            int idxMakeDL = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU, 0xFF, 0xFF, 0);

            // ストリームアウトで頂点を出力する方法は
            // 頂点の再利用率次第では描画の GPU 処理が増える可能性があります。
            // ただし、マルチパスや画面分割で同じモデルを描画する場合
            // 多くの場合が処理の最適化になります。
            if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_STREAMOUT))
            {
                if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST | USE_MULTI_CORE))
                {
                    s_IsStreamOutTask = true;
                    RunTaskQueue(StreamOutUserModel);
                    s_IsStreamOutTask = false;
                }
                else
                {
                    if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST))
                    {
                        s_StreamOutDL[MAIN_DL].Begin();
                    }
                    nw::g3d::SetStreamOutEnable(GX2_TRUE);
                    nw::g3d::SetRasterizerClipControl(GX2_FALSE, GX2_TRUE);

                    for (int idxModel = 0, numModel = s_Holder.models.Size();
                        idxModel != numModel;
                        ++idxModel)
                    {
                        g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
                        StreamOutUserModel(pUserModel, idxModel);
                    }

                    nw::g3d::SetStreamOutEnable(GX2_FALSE);
                    nw::g3d::SetRasterizerClipControl(GX2_TRUE, GX2_TRUE);

                    if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST))
                    {
                        s_StreamOutDL[MAIN_DL].End();
                    }
                }

                if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST))
                {
                    s_SetupDL[FLUSH_DL].Begin();
                }

                // ストリームアウトされた頂点のキャッシュをケアします。
                nw::g3d::GPUCache::FlushAll();
                nw::g3d::GPUCache::InvalidateAll();

                if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST))
                {
                    s_SetupDL[FLUSH_DL].End();
                }
            }

            if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST) && s_ChangeDrawCommand)
            {
                // ストリームアウトを使うなど、
                // UserModel の描画に変化が加えられた場合はディスプレイリストを作り直します。
                MakeUserModelDL();
                s_ChangeDrawCommand = false;
            }

            // UserModel の描画を行います。
            // UserModel はフレームごとのディスプレイリスト（s_DrawDL）と
            // 静的なディスプレイリスト（s_UserModelDL）を組み合わせて描画します。
            // コアごとに作られた s_DrawDL から s_UserModelDL を CallDisplayList で参照します。
            // 描画コマンドをマルチコアで作成するにはディスプレイリストが必須になります。
            if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST | USE_MULTI_CORE))
            {
                s_IsDrawTask = true;
                RunTaskQueue(DrawUserModel);
                s_IsDrawTask = false;
            }
            else
            {
                if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST))
                {
                    s_DrawDL[MAIN_DL].Begin();
                }
                for (int idxModel = 0, numModel = s_Holder.models.Size();
                    idxModel != numModel;
                    ++idxModel)
                {
                    g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
                    DrawUserModel(pUserModel, idxModel);
                }
                if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST))
                {
                    s_DrawDL[MAIN_DL].End();
                }
            }

            s_Meter.EndTimeSpan(idxMakeDL);

            if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST))
            {
                s_SetupDL[INFO_DL].Begin();
            }

            gpuMetric.End();

            s_Meter.EndTimeSpan(idxGPU);

            // 情報表示のためにシェーダモードを切り替えます。
            nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_BLOCK);
            g3ddemo::ScreenInfo::LoadState();
            s_Screen.Draw();
            s_Meter.Draw();

            if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST))
            {
                s_SetupDL[INFO_DL].End();
            }
        }

        // 残っているコマンドの完了を待ちます。
        nw::g3d::GfxManage::DrawDone();

        int idxCopyOut = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_SYS_GPU);
        // コピーアウトを実行します。
        // GPU がコピーアウトとクリアしている間に GPU 用のリソースを書き換えます。
        g3ddemo::CopyOut(&colorBuffer, GX2_SCAN_TARGET_TV);
        nw::g3d::ClearBuffers(&colorBuffer, &depthBuffer,
            0.3f, 0.3f, 0.3f, 1.0f, GX2_CLEAR_BOTH);

        pCtx->TempPrepare();
        s_Meter.EndTimeSpan(idxCopyOut);
        nw::g3d::fnd::GfxManage::FlushCommands();

        // メトリックを集計します。
        gpuMetric.Calc(nw::g3d::ut::CheckFlag(s_UseFlag, USE_DISPLAY_LIST));
        if(gpuMetric.IsCalcComplete())
        {
            if (s_PerfEnabled)
            {
                gpuMetric.Print();
                s_PerfEnabled = false;
            }
        }

        {
            pCtx->TempPrepare();
            int idxCalcGPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU, 0, 0xFF, 0);

            // ビュー毎の更新。
            for (int idxView = 0; idxView < NUM_VIEW; ++idxView)
            {
                s_View.Calc(idxView);
            }

            if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_MULTI_CORE))
            {
                RunTaskQueue(CalcGPUUserModel);
            }
            else
            {
                for (int idxModel = 0, numModel = s_Holder.models.Size();
                    idxModel != numModel;
                    ++idxModel)
                {
                    g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];

                    CalcGPUUserModel(pUserModel, idxModel);
                }
            }

            // 環境
            if (s_Holder.pEnvModel)
            {
                s_Holder.pEnvModel->CalcGPU();
            }

            s_Meter.EndTimeSpan(idxCalcGPU);
        }
        nw::g3d::CPUCache::Sync(); // CPU キャッシュ操作の完了を待ちます。

        // コピーアウトの完了を待ちます。
        nw::g3d::GfxManage::DrawDone();
        s_Meter.Calc();

        pCtx->TempPrepare(); // ProcessMeter の登録に GL コンテクストが必要。
        s_Meter.EndTimeSpan(idxFrame);

        g3ddemo::SwapScanBuffers();

        ++s_Counter;
    }

    for (int i = 0; i < NUM_SETUP_DL; ++i)
    {
        s_SetupDL[i].Cleanup();
    }
    for (int i = 0; i < NUM_DL; ++i)
    {
        s_StreamOutDL[i].Cleanup();
    }
    for (int i = 0; i < NUM_DL; ++i)
    {
        s_DrawDL[i].Cleanup();
    }
    for (int i = 0; i < NUM_COPY_MODEL; ++i)
    {
        for (int j = 0; j < MAX_MODEL_LOD; ++j)
        {
            s_UserModelDL[i][j].Cleanup();
        }
    }

    if (nw::g3d::ut::CheckFlag(s_UseFlag, USE_BAKE_ANIMATION))
    {
        ChangeBakedAnimation();
    }

    gpuMetric.Cleanup();
#ifndef _WIN32
    // サブコア終了。
    {
        s_IsExit = true;

        // サブコアを確実に終了するために
        // MPWaitTaskQ で MP_WAIT_TASKQ_RUN で待機しているサブコアを空で実行します。
        MPResetTaskQ(&s_TaskQueue);
        MPStartTaskQ(&s_TaskQueue);

        ExecuteTask();

        MPWaitTaskQ(&s_TaskQueue, MP_WAIT_TASKQ_DONE);
    }
#endif

    // グラフィックスリソースの終了処理。
    pCtx->TempPrepare();

    if (void* ptr = ringBuffer.GetInputBuffer())
    {
        g3ddemo::FreeMem1(ptr);
    }
    if (void* ptr = ringBuffer.GetOutputBuffer())
    {
        g3ddemo::FreeMem1(ptr);
    }

    // エディットライブラリが所有しているモデルを破棄しないように、
    // エディットの終了処理後に残りのデータの破棄を行います。
    g3ddemo::DestroyAll(&s_Holder);

    s_View.Cleanup();

    s_Screen.Cleanup();
    s_Meter.Cleanup();

    frameBuffer.Free(g3ddemo::FreeMem1);
    frameBuffer.Cleanup();
    g3ddemo::FreeMem2(frameBuffer.GetBufferPtr());

    pCtx->Cleanup();
    g3ddemo::FreeMem2(pCtx);
    pCtx = NULL;

    // 終了処理。
    g3ddemo::ShutdownDisplay();
    g3ddemo::Shutdown();

    return restart ? -1 : EXIT_SUCCESS;
}

namespace {

g3ddemo::ModelAssign* SetupModelAssign(nw::g3d::ResModel* pResModel)
{
    g3ddemo::CreateShaderAssign(pResModel);
    g3ddemo::ModelAssign* pModelAssign = pResModel->GetUserPtr<g3ddemo::ModelAssign>();

    // モデルに記録されたデフォルトシェーダの関連付け。
    nw::g3d::BindResult result = nw::g3d::BindResult::NotBound();
    for (int idxShader = 0, numShader = s_Holder.shaders.Size();
        !result.IsComplete() && idxShader < numShader; ++idxShader)
    {
        nw::g3d::ResShaderArchive* pShaderArchive = s_Holder.shaders[idxShader];
        result = pModelAssign->BindShader(g3ddemo::PASS_DEFAULT, pShaderArchive);
    }
    //NW_G3D_ASSERT(result.IsComplete());

    // ストリームアウトシェーダの関連付け。
    if (s_Holder.pStreamoutArchive)
    {
        pModelAssign->BindStreamOutShader(s_Holder.pStreamoutArchive);
    }
    pModelAssign->Setup();
    return pModelAssign;
}

g3ddemo::UserModel* SetupUserModel(nw::g3d::ModelObj* pModelObj)
{
    g3ddemo::CreateUserModel(pModelObj);
    g3ddemo::UserModel* pUserModel = pModelObj->GetUserPtr<g3ddemo::UserModel>();
    pUserModel->SetView(&s_View);
    if (s_Holder.pEnvModel)
    {
        pUserModel->SetEnv(s_Holder.pEnvModel->GetModelObj());
    }

    g3ddemo::SetupMaterialsArg argSetup(pModelObj);
    argSetup.pEnvMtx = &s_TexMtxEnv;
    argSetup.pProjMtx = &s_TexMtxProj;
    g3ddemo::SetupMaterials(argSetup);

    // アニメーションインスタンスの構築。
    g3ddemo::UserModel::CreateAnimObjArg createAnimObjArg;
    if (!s_Holder.modelAnims.Empty())
    {
        createAnimObjArg.Reserve(&s_Holder.modelAnims[0], s_Holder.modelAnims.Size());
    }
    pUserModel->CreateAnimObj(createAnimObjArg);

    return pUserModel;
}

void ResetDL()
{
    // 状態が変化した場合などに構築済みのディスプレイリストを破棄します。
    for (int i = 0; i < NUM_SETUP_DL; ++i)
    {
        s_SetupDL[i].Reset();
    }
    for (int i = 0; i < NUM_DL; ++i)
    {
        s_StreamOutDL[i].Reset();
    }
    for (int i = 0; i < NUM_DL; ++i)
    {
        s_DrawDL[i].Reset();
    }
}

void MakeUserModelDL()
{
#ifndef _WIN32
    // UserModel ごとに静的なディスプレイリストを構築します。
    // このデモでは Model を一つのディスプレイリストにします。
    // 必要に応じてシェーダや描画設定を適切な粒度でディスプレイリストにして連続で呼び出すことで
    // 描画負荷を軽減と取り回しの柔軟性とのトレードオフができます。
    for (int idxModel = 0, numModel = s_Holder.models.Size();
        idxModel != numModel;
        ++idxModel)
    {
        g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
        for (int idxLod = 0; idxLod < MAX_MODEL_LOD; ++idxLod)
        {
            pUserModel->SetLODLevel(0, idxLod);
            s_UserModelDL[idxModel][idxLod].Begin();
            pUserModel->DebugDraw(g3ddemo::PASS_DEFAULT, 0, GX2_SHADER_MODE_UNIFORM_BLOCK);
            s_UserModelDL[idxModel][idxLod].End();
        }
    }
#endif
}

void ChangeStreamOut()
{
    for (int idxModel = 0, numModel = s_Holder.models.Size();
        idxModel != numModel;
        ++idxModel)
    {
        g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
        if (pUserModel->IsStreamOutEnabled())
        {
            pUserModel->DisableStreamOut();
        }
        else
        {
            pUserModel->EnableStreamOut();
        }
    }
    s_ChangeDrawCommand = true;
}

void ChangeLockedCache()
{
    for (int idxModel = 0, numModel = s_Holder.models.Size();
        idxModel != numModel;
        ++idxModel)
    {
        g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
        if (pUserModel->IsLockedCacheEnabled())
        {
            pUserModel->DisableLockedCache();
        }
        else
        {
            pUserModel->EnableLockedCache();
        }
    }
}

void ChangeBakedAnimation()
{
    for (int idxAnim = 0, numAnim = s_Holder.modelAnims.Size();
        idxAnim != numAnim;
        ++idxAnim)
    {
        void* pResAnim = s_Holder.modelAnims[idxAnim];

        // スケルタルアニメーションに限定
        nw::g3d::ResSkeletalAnim* pResSkeletalAnim = static_cast<nw::g3d::ResSkeletalAnim*>(pResAnim);
        if (pResSkeletalAnim->IsCurveBaked())
        {
            void* bakeBuffer = pResSkeletalAnim->ResetCurve();
            g3ddemo::FreeMem2(bakeBuffer);
        }
        else
        {
            void* bakeBuffer = g3ddemo::AllocMem2(pResSkeletalAnim->GetBakedSize());
            pResSkeletalAnim->BakeCurve(bakeBuffer, pResSkeletalAnim->GetBakedSize());
        }
    }
}

void CalcMenu(const g3ddemo::Pad &pad)
{
    (void)pad;
#ifndef _WIN32
    if (pad.IsHold(g3ddemo::Pad::TRIGGER_L) && pad.IsTriggered(g3ddemo::Pad::TRIGGER_R))
    {
        s_UseCamera = !s_UseCamera;
    }

    if (!s_UseCamera)
    {
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_A))
        {
            s_UseFlag = nw::g3d::ut::InvertFlag(s_UseFlag, USE_DISPLAY_LIST);
            ResetDL();
        }
        else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_B))
        {
            s_UseFlag = nw::g3d::ut::InvertFlag(s_UseFlag, USE_MULTI_CORE);
        }
        else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_X))
        {
            s_UseFlag = nw::g3d::ut::InvertFlag(s_UseFlag, USE_STREAMOUT);
            ChangeStreamOut();
            ResetDL();
        }
        else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_Y))
        {
            s_UseFlag = nw::g3d::ut::InvertFlag(s_UseFlag, USE_LOCKED_CACHE);
            ChangeLockedCache();
        }
        else if (pad.IsTriggered(g3ddemo::Pad::TRIGGER_R))
        {
            s_PerfEnabled = true;
        }
        else if (pad.IsTriggered(g3ddemo::Pad::TRIGGER_Z))
        {
            s_UseFlag = nw::g3d::ut::InvertFlag(s_UseFlag, USE_NOT_DRAW_PIXEL);
        }
        else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_UP))
        {
            s_UseFlag = nw::g3d::ut::InvertFlag(s_UseFlag, USE_GPU_THREAD_OPTIMIZATION);
        }
        else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_DOWN))
        {
            s_UseFlag = nw::g3d::ut::InvertFlag(s_UseFlag, USE_ANIMATION_LOD);
        }
        else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
        {
            s_UseFlag = nw::g3d::ut::InvertFlag(s_UseFlag, USE_BAKE_ANIMATION);
            ChangeBakedAnimation();
        }
        else if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
        {
            s_UseFlag = nw::g3d::ut::InvertFlag(s_UseFlag, USE_MODEL_LOD);
        }
    }

    // スクリーン情報
    {
        float fontSize = 2.0f;
        s_Screen.Reset();
        float ofsX = fontSize;
        float ofsY = fontSize;

        s_Screen.PutStringFmt(ofsX, ofsY, "Num Model: %d", NUM_COPY_MODEL);

        // 実機版のみ利用可能
        if (s_UseCamera)
        {
            ofsY += fontSize;
            s_Screen.PutString(ofsX, ofsY, "L+R:     Disable Camera");
            ofsY += fontSize;
            s_Screen.PutString(ofsX, ofsY, "L Stick: Rotate");
            ofsY += fontSize;
            s_Screen.PutString(ofsX, ofsY, "R Stick: Move XZ");
            ofsY += fontSize;
            s_Screen.PutString(ofsX, ofsY, "A:       Move Y+");
            ofsY += fontSize;
            s_Screen.PutString(ofsX, ofsY, "B:       Move Y-");
            ofsY += fontSize;
            s_Screen.PutString(ofsX, ofsY, "X:       Zoom In");
            ofsY += fontSize;
            s_Screen.PutString(ofsX, ofsY, "Y:       Zoom Out");
        }
        else
        {
            ofsY += fontSize;
            s_Screen.PutString(ofsX, ofsY, "L+R:  Enable Camera");
            ofsY += fontSize;
            s_Screen.PutStringFmt(ofsX, ofsY, "A:     Display Lists           [%s]", nw::g3d::CheckFlag(s_UseFlag, USE_DISPLAY_LIST) ? "Enabled" : "Disabled");
            ofsY += fontSize;
            s_Screen.PutStringFmt(ofsX, ofsY, "B:     Multi-Core              [%s]", nw::g3d::CheckFlag(s_UseFlag, USE_MULTI_CORE) ? "Enabled" : "Disabled");
            ofsY += fontSize;
            s_Screen.PutStringFmt(ofsX, ofsY, "X:     Stream Out              [%s]", nw::g3d::CheckFlag(s_UseFlag, USE_STREAMOUT) ? "Enabled" : "Disabled");
            ofsY += fontSize;
            s_Screen.PutStringFmt(ofsX, ofsY, "Y:     Locked Cache            [%s]", nw::g3d::CheckFlag(s_UseFlag, USE_LOCKED_CACHE) ? "Enabled" : "Disabled");
            ofsY += fontSize;
            s_Screen.PutString(ofsX, ofsY, "R:     Print GPU Metric");
            ofsY += fontSize;
            s_Screen.PutStringFmt(ofsX, ofsY, "Up:    GPU Thread Optimization [%s]", nw::g3d::CheckFlag(s_UseFlag, USE_GPU_THREAD_OPTIMIZATION) ? "Enabled" : "Disabled");
            ofsY += fontSize;
            s_Screen.PutStringFmt(ofsX, ofsY, "Down:  Animation LOD           [%s]", nw::g3d::CheckFlag(s_UseFlag, USE_ANIMATION_LOD) ? "Enabled" : "Disabled");
            ofsY += fontSize;
            s_Screen.PutStringFmt(ofsX, ofsY, "Left:  Baked Animation         [%s]", nw::g3d::CheckFlag(s_UseFlag, USE_BAKE_ANIMATION) ? "Enabled" : "Disabled");
            ofsY += fontSize;
            s_Screen.PutStringFmt(ofsX, ofsY, "Right: Model LOD               [%s]", nw::g3d::CheckFlag(s_UseFlag, USE_MODEL_LOD) ? "Enabled" : "Disabled");
        }

        s_Screen.DCFlush();
    }
#endif
}

} // anonymous namespace
