﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d.h>
#include <nw/g3d/g3d_edit.h>
#include <g3ddemo_DemoUtility.h>
#include <g3ddemo_GfxUtility.h>
#include <g3ddemo_UserModel.h>
#include <g3ddemo_ModelUtility.h>

namespace g3ddemo = nw::g3d::demo;

namespace {

enum
{
    // モデルに使用するテクスチャのアライメントの最大値を設定します。
    // リソースファイルごとのアライメントを事前に取得しておくことで、
    // 必要なメモリのアライメントを最小限に抑えてメモリを節約することも可能です。
    TEXTURE_ALIGNMENT   = 8192
};

#define SHADER_PATH "shader/"
#define MODEL_PATH  ""

const char* BFSHA_PATH[] = {
    SHADER_PATH "demo.bfsha",
    SHADER_PATH "debug.bfsha",
    SHADER_PATH "streamout.bfsha",
    SHADER_PATH "demo_branch.bfsha", // アタッチ用の静的分岐シェーダ
    NULL
};
const char* BFRES_PATH[] = {
    MODEL_PATH "env.bfres",
    MODEL_PATH "human.bfres",
    NULL
};

bool s_UseCamera;
nw::g3d::Vec3 s_CameraPosition;
nw::g3d::Vec3 s_CameraUp;
nw::g3d::Vec3 s_CameraTarget;

const int NUM_VIEW = 1;
#if NW_G3D_CONFIG_USE_HOSTIO
// シェイプで使用するユーザエリアのサイズです。
const int SHAPE_USER_AREA_SIZE = sizeof(float) * 4 * 2;
#endif

g3ddemo::ResourceHolder s_Holder;
g3ddemo::ScreenInfo s_Screen;
g3ddemo::ProcessMeter s_Meter;

enum
{
    ITEM_MODEL,
    ITEM_ANIM,
    ITEM_SHADER,
    NUM_ITEM
};

int s_ItemIndex = ITEM_MODEL;
int s_ModelIndex = 1;
int s_AnimIndex = 0;
int s_ShaderIndex = 0;

nw::g3d::ResShadingModel* s_pDebugShader = NULL;
nw::g3d::ResShaderOption* s_pOutputOption = NULL;
int s_OutputOptionIndex = -1;

g3ddemo::UserView s_View;
nw::g3d::Mtx34 s_TexMtxEnv;
nw::g3d::Mtx34 s_TexMtxProj;

g3ddemo::ModelAssign* SetupModelAssign(nw::g3d::ResModel* pResModel,
    nw::g3d::ResShaderArchive** pShaderArchiveArray = NULL, int ShaderArchiveCount = 0,
    nw::g3d::ResShaderArchive* pAttachShader = NULL);
g3ddemo::UserModel* SetupUserModel(nw::g3d::ModelObj* pModelObj);
void CalcMenuCPU(const g3ddemo::Pad &pad);
void CalcMenuGPU();

#if NW_G3D_CONFIG_USE_HOSTIO

nw::g3d::ResShaderArchive* s_pBranchShader = NULL;
bool s_IsBranchShaderAttached = false;
void ChangeShader(nw::g3d::ResShaderArchive* pShaderArchive);

g3ddemo::EditAllocator s_Allocator;

class Editor : public nw::g3d::edit::EditCallback
{
public:
    virtual nw::g3d::ResFile* LoadFile(const nw::g3d::edit::LoadFileArg& arg)
    {
        // 3DEditor が開いたモデルがリソースとして送られてきます。
        // リソースファイルはアプリケーションが管理する必要あるため、コピーします。
        void* pFile = g3ddemo::AllocMem2(arg.fileSize, arg.align);
        memcpy(pFile, arg.resFile, arg.fileSize);
        // CPU がコピーしたので、GPU が参照できるようキャッシュをフラッシュします。
        nw::g3d::CPUCache::Flush(pFile, arg.fileSize);

        NW_G3D_ASSERT(nw::g3d::ResFile::IsValid(pFile));
        nw::g3d::ResFile* pResFile = nw::g3d::ResFile::ResCast(pFile);
        pResFile->Setup();

        s_Holder.files.PushBack(pResFile);

        // モデルインスタンス構築処理は、アプリケーションで使用している構築処理を
        // そのまま使用することができます。
        // このデモではコールバック内でモデルを構築、登録していますが、
        // コールバック外で遅延処理しても構いません。
        for (int idxModel = 0, numModel = pResFile->GetModelCount();
            idxModel < numModel; ++idxModel)
        {
            nw::g3d::ResModel* pResModel = pResFile->GetModel(idxModel);

            // シェーダ割り当ての構築。
            g3ddemo::ModelAssign* pModelAssign = SetupModelAssign(pResModel);
            s_Holder.assigns.PushBack(pModelAssign);

            // キーシェイプをもつ場合には、シェイプアニメのため StreamOut を有効にします。
            bool hasKeyShape = false;
            for (int idxShape = 0, numShape = pResModel->GetShapeCount();
                 idxShape < numShape; ++idxShape)
            {
                nw::g3d::res::ResShape* pResShape = pResModel->GetShape(idxShape);
                if (pResShape->GetKeyShapeCount() > 0)
                {
                    hasKeyShape = true;
                    break;
                }
            }

            // モデルインスタンスの構築。
            g3ddemo::CreateModelObjArg createModelObjArg(pResModel);
            createModelObjArg.initArg.ViewCount(NUM_VIEW);
            createModelObjArg.initArg.EnableBounding();
            if (hasKeyShape)
            {
                createModelObjArg.initArg.ShapeUserAreaSize(SHAPE_USER_AREA_SIZE);
            }
            nw::g3d::ModelObj* pModelObj = g3ddemo::CreateModelObj(createModelObjArg);
            g3ddemo::UserModel* pUserModel = SetupUserModel(pModelObj);
            pUserModel->EnableEditCalcSkeletalAnimations();
            if (hasKeyShape)
            {
                pUserModel->EnableStreamOut();
            }
            s_Holder.models.PushBack(pUserModel);

            // エディットへの登録。
            // エディットライブラリのために専用のクラスなどを作成する必要がありません。
            pUserModel->AttachToEdit();
        }

        return pResFile;
    }

    virtual void UnloadFile(const nw::g3d::edit::UnloadFileArg& arg)
    {
        // 3DEditor でファイルが閉じられたので、関連するデータを破棄します。
        // GPU が参照中のデータを破棄してしまわないよう、アプリケーションの
        // フレームワークに応じて描画完了待ちを行ったり、描画をスキップするなど、
        // 必要な対策を講じてください。
        // このデモでは GPU による描画が完了してから呼ばれているため、
        // 特に対策は行っていません。
        nw::g3d::ResFile* pResFile = arg.resFile;

        for (int idxResMdl = 0, numResMdl = pResFile->GetModelCount();
            idxResMdl < numResMdl; ++idxResMdl)
        {
            nw::g3d::ResModel* pResModel = pResFile->GetModel(idxResMdl);

            // モデルインスタンスの破棄。
            for (int idxModel = 0, numModel = s_Holder.models.Size();
                idxModel < numModel; ++idxModel)
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[idxModel];
                nw::g3d::ModelObj* pModelObj = pUserModel->GetModelObj();
                if (pModelObj->GetResource() == pResModel)
                {
                    pUserModel->DestroyAnimObj();
                    g3ddemo::DestroyUserModel(pModelObj);
                    g3ddemo::DestroyModelObj(pModelObj);
                    s_Holder.models.Erase(idxModel);
                    break;
                }
            }

            // シェーダ割り当ての破棄。
            for (int idxAssign = 0, numAssign = s_Holder.assigns.Size();
                idxAssign < numAssign; ++idxAssign)
            {
                g3ddemo::ModelAssign* pModelAssign = s_Holder.assigns[idxAssign];
                if (pModelAssign->GetResModel() == pResModel)
                {
                    pModelAssign->Cleanup();
                    g3ddemo::DestroyShaderAssign(pResModel);
                    s_Holder.assigns.Erase(idxAssign);
                    break;
                }
            }

            // リソースファイルの破棄。
            for (int idxFile = 0, numFile = s_Holder.files.Size();
                idxFile < numFile; ++idxFile)
            {
                if (s_Holder.files[idxFile] == pResFile)
                {
                    pResFile->Cleanup();
                    g3ddemo::FreeMem2(pResFile);
                    s_Holder.files.Erase(idxFile);
                    break;
                }
            }

        }

        // モデルのインデックスがずれるので範囲外のインデックスを 0 に戻します。
        if (s_ModelIndex >= s_Holder.models.Size())
        {
            s_ModelIndex = 0;
        }
    }

    virtual void UpdateMaterials(const nw::g3d::edit::UpdateMaterialsArg& arg)
    {
        // UpdateShaderAssign で行われた変更に伴って ModelObj に変更が必要な場合は
        // このコールバックで処理を行います。
        // 編集のために使用するヒープを区別したい場合は arg.state で状態を判定します。
        // UpdateShaderAssign() で初期化したリソースに従って MaterialObj や ShapeObj が
        // エディットライブラリによって作り直されているので、必要な初期化を行います。

        nw::g3d::ModelObj* pModelObj = arg.modelObj;

        // シェーダが変更された場合のためにシェーダインスタンスを再構築します。
        for (int idxShape = 0, numShape = pModelObj->GetShapeCount();
            idxShape < numShape; ++idxShape)
        {
            nw::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(idxShape);
            g3ddemo::UserShape* pUserShape = pShapeObj->GetUserPtr<g3ddemo::UserShape>();
            pUserShape->Cleanup();
            pUserShape->Setup();
        }

        // マテリアルを更新します。
        g3ddemo::SetupMaterialsArg argSetup(pModelObj);
        argSetup.pEnvMtx = &s_TexMtxEnv;
        argSetup.pProjMtx = &s_TexMtxProj;
        g3ddemo::SetupMaterials(argSetup);
    }

    virtual void UpdateShaderAssign(const nw::g3d::edit::UpdateShaderAssignArg& arg)
    {
        // モデルリソースの構造変更を伴う編集が 3DEditor で行われたため、
        // 古いモデルリソースの破棄と新しいモデルリソースの構築を行います。
        // モデルインスタンスが新しいリソースを参照するように、
        // エディットライブラリが参照関係を変更します。
        switch(arg.state)
        {
        case nw::g3d::edit::UpdateShaderAssignArg::STATE_MODEL_UPDATE:
        case nw::g3d::edit::UpdateShaderAssignArg::STATE_MODEL_END:
            {
                // モデルリソースにユーザ領域を関連付けて使用していた場合は破棄します。
                g3ddemo::ModelAssign* pModelAssign = arg.oldResModel->GetUserPtr<g3ddemo::ModelAssign>();
                NW_G3D_ASSERT_NOT_NULL(pModelAssign);
                pModelAssign->Cleanup();
                g3ddemo::DestroyShaderAssign(arg.oldResModel);
            }
            break;
        default:
            break;
        }
        switch(arg.state)
        {
        case nw::g3d::edit::UpdateShaderAssignArg::STATE_MODEL_BEGIN:
        case nw::g3d::edit::UpdateShaderAssignArg::STATE_MODEL_UPDATE:
            {
                // モデルリソースにユーザ領域を関連付けて管理します。
                // アプリケーションで使用している構築処理をそのまま使用することができます。
                SetupModelAssign(arg.newResModel, arg.shaderArchivePtrs, arg.numShaderArchive,
                    s_IsBranchShaderAttached ? s_pBranchShader : NULL);
            }
            break;
        default:
            break;
        }
    }

    virtual void UpdateBoneBind(const nw::g3d::edit::UpdateBoneBindArg& arg)
    {
        // 3DEditor でプレビュー配置のバインド設定が行われた際にモデルに通知を行います。
        // 配置自体はここで行わずにアプリケーションのバインド機能によってバインドします。
        g3ddemo::UserModel* pUserModel = arg.modelObj->GetUserPtr<g3ddemo::UserModel>();
        pUserModel->BindTo(arg.parentModelObj, arg.parentBoneIndex);
    }

    virtual void UpdateModelLayout(const nw::g3d::edit::UpdateModelLayoutArg& arg)
    {
        g3ddemo::UserModel* pUserModel = arg.modelObj->GetUserPtr<g3ddemo::UserModel>();
        pUserModel->GetScale().Set(*nw::g3d::Vec3::Cast(arg.scale.a));
        pUserModel->GetLayoutMtx().SetR(*nw::g3d::Vec3::Cast(arg.rotate.a));
        pUserModel->GetLayoutMtx().SetT(*nw::g3d::Vec3::Cast(arg.translate.a));
    }

    virtual bool SendModelLayout(nw::g3d::edit::SendModelLayoutData* outputData,
        const nw::g3d::edit::SendModelLayoutArg& arg)
    {
        const g3ddemo::UserModel* pUserModel = arg.modelObj->GetUserPtr<g3ddemo::UserModel>();
        *nw::g3d::Vec3::Cast(outputData->scale.a) = pUserModel->GetScale();
        const nw::g3d::Mtx34& layoutMtx = pUserModel->GetLayoutMtx();
        float sy = -layoutMtx.m20;
        // Y 軸周りの回転は ±90°の範囲として計算します。
        outputData->rotate.y = nw::g3d::Math::Asin(sy);
        if (nw::g3d::Math::Abs(sy) > 0.9999f)
        {
            // Y 軸周りの回転が ±90°の場合は角度が一意に決まらないため、
            // X 軸周りの回転は 0°として扱います。
            outputData->rotate.x = 0.0f;
            // Z 軸周りの回転は ±180°の範囲として計算します。
            outputData->rotate.z = nw::g3d::Math::Atan2(layoutMtx.m12, layoutMtx.m02);
        }
        else
        {
            // X 軸および Z 軸周りの回転は ±180°の範囲として計算します。
            outputData->rotate.x = nw::g3d::Math::Atan2(layoutMtx.m21, layoutMtx.m22);
            outputData->rotate.z = nw::g3d::Math::Atan2(layoutMtx.m10, layoutMtx.m00);
        }
        outputData->translate.x = layoutMtx.m03;
        outputData->translate.y = layoutMtx.m13;
        outputData->translate.z = layoutMtx.m23;
        return true;
    }

    virtual void AttachShader(const nw::g3d::edit::AttachShaderArg& arg)
    {
        // リアルタイム編集しやすくするために、静的分岐のシェーダに切り替えます。
        if (arg.shaderArchive == s_pBranchShader)
        {
            s_IsBranchShaderAttached = true;
            ChangeShader(arg.shaderArchive);
        }
    }

    virtual void DetachShader(const nw::g3d::edit::DetachShaderArg& arg)
    {
        // リアルタイム編集が終了したので、バリエーションのシェーダに切り替えます。
        if (arg.shaderArchive == s_pBranchShader)
        {
            s_IsBranchShaderAttached = false;
            ChangeShader(NULL);
        }
    }

    virtual void BindSceneAnim(const nw::g3d::edit::BindSceneAnimArg& arg)
    {
        (void)arg;
    }

    virtual void UnbindSceneAnim(const nw::g3d::edit::UnbindSceneAnimArg& arg)
    {
        (void)arg;
    }

    virtual void ApplySceneAnim(const nw::g3d::edit::ApplySceneAnimArg& arg)
    {
        for (int viewIdx = 0; viewIdx < NUM_VIEW && viewIdx < arg.numCameraAnimObj; ++viewIdx)
        {
            const nw::g3d::CameraAnimObj* pCameraAnimObj = arg.cameraAnimObjPtrs[viewIdx];
            const nw::g3d::CameraAnimResult* result = pCameraAnimObj->GetResult();
            nw::g3d::Mtx34 viewMtx;
            viewMtx.LookAt(
                nw::g3d::Vec3::Make(result->view.pos[0], result->view.pos[1], result->view.pos[2]),
                nw::g3d::Vec3::Make(0.0f, 1.0f, 0.0f),
                nw::g3d::Vec3::Make(result->view.aim[0], result->view.aim[1], result->view.aim[2]));
            nw::g3d::Mtx44 projMtx;
            projMtx.Perspective(
                result->proj.fovy, result->proj.aspect, result->proj.nearZ, result->proj.farZ);
            memcpy(&s_View.GetViewMtx(viewIdx), &viewMtx, sizeof(s_View.GetViewMtx(viewIdx)));
            memcpy(&s_View.GetProjMtx(viewIdx), &projMtx, sizeof(s_View.GetProjMtx(viewIdx)));
        }
    }
} s_Editor;

#endif

} // anonymous namespace

int EditMain(int argc, const char* argv[])
{
    (void)argc;
    (void)argv;

    // 初期化処理。
    g3ddemo::Init();
    g3ddemo::InitDisplayArg initDisplayArg;
    g3ddemo::InitDisplay(initDisplayArg);

    g3ddemo::Pad pad;
    if (!pad.Reset())
    {
        g3ddemo::PostQuitMsg();
    }

    // グラフィックスリソースの初期化処理。
    nw::g3d::GfxContext::Prepare(); // 構築に GL コンテクストが必要。
    nw::g3d::GfxContext* pCtx = nw::g3d::demo::AllocMem2<nw::g3d::GfxContext>(
        sizeof(*pCtx), GX2_CONTEXT_STATE_ALIGNMENT);
    pCtx->Setup();
    pCtx->TempPrepare();

    g3ddemo::FrameBuffer frameBuffer;
    {
        g3ddemo::FrameBuffer::InitArg initArg(1280, 720);
        initArg.colorBufferFTV = true;
        size_t bufferSize = g3ddemo::FrameBuffer::CalcSize(initArg);
        frameBuffer.Init(initArg, g3ddemo::AllocMem2(bufferSize), bufferSize);
        frameBuffer.Setup();
        frameBuffer.Alloc(g3ddemo::AllocMem1);
    }

    g3ddemo::FrameBuffer frameBufferDRC;
    {
        g3ddemo::FrameBuffer::InitArg initArg(854, 480);
        size_t bufferSize = g3ddemo::FrameBuffer::CalcSize(initArg);
        frameBufferDRC.Init(initArg, g3ddemo::AllocMem2(bufferSize), bufferSize);
        frameBufferDRC.Setup();
        frameBufferDRC.Alloc(g3ddemo::AllocMem1);
    }

    nw::g3d::GfxColorBuffer& colorBuffer =
        frameBuffer.GetColorBufferTexture(GX2_RENDER_TARGET_0)->renderBuffer;
    nw::g3d::GfxDepthBuffer& depthBuffer = frameBuffer.GetDepthBufferTexture()->renderBuffer;

    nw::g3d::GfxColorBuffer& colorBufferDRC =
        frameBufferDRC.GetColorBufferTexture(GX2_RENDER_TARGET_0)->renderBuffer;
    nw::g3d::GfxDepthBuffer& depthBufferDRC = frameBufferDRC.GetDepthBufferTexture()->renderBuffer;

    s_Screen.Setup(1024);
    s_Screen.SetShapeColor(0x7F, 0x7F, 0x7F, 0x7F);
    s_Screen.SetFontColor(0xFF, 0xFF, 0xFF);
    s_Screen.SetFontSize(2.0f);
    s_Meter.Setup();

#if NW_G3D_CONFIG_USE_HOSTIO
    // edit library
    {
        nw::g3d::edit::EditServer::CreateArg arg;
        arg.allocator = &s_Allocator;
        arg.editCallback = &s_Editor;

        bool createResult = nw::g3d::edit::EditServer::CreateInstance( arg );
        NW_G3D_ASSERT(createResult);

        bool openConnectionResult = nw::g3d::edit::EditServer::Instance().Open();
        NW_G3D_ASSERT(openConnectionResult);
    }
#endif

    // ビューの Uniform Block
    s_View.Setup();

    s_Holder.Init();
    s_ItemIndex = ITEM_MODEL;
    s_ModelIndex = 1;
    s_AnimIndex = 0;
    s_ShaderIndex = 0;

    // シェーダの初期化。
    u32 maxInputRingItem = 0;
    u32 maxOutputRingItem = 0;
    for (const char** path = BFSHA_PATH; *path != NULL; ++path)
    {
        void* pFile = g3ddemo::LoadFile(path[0], NULL, GX2_SHADER_ALIGNMENT);
        NW_G3D_ASSERT(nw::g3d::ResShaderArchive::IsValid(pFile));
        nw::g3d::ResShaderArchive* pShaderArchive = nw::g3d::ResShaderArchive::ResCast(pFile);
        pShaderArchive->Setup();

        for (int idxShadingModel = 0, numShadingModel = pShaderArchive->GetShadingModelCount();
            idxShadingModel < numShadingModel; ++idxShadingModel)
        {
            nw::g3d::ResShadingModel* pShadingModel =
                pShaderArchive->GetShadingModel(idxShadingModel);
            maxInputRingItem = std::max(maxInputRingItem, pShadingModel->GetMaxInputRingItem());
            maxOutputRingItem = std::max(maxOutputRingItem, pShadingModel->GetMaxOutputRingItem());
        }

        s_Holder.shaders.PushBack(pShaderArchive);

        if (s_Holder.pStreamoutArchive == NULL &&
            0 == strcmp("streamout", pShaderArchive->GetName()))
        {
            s_Holder.pStreamoutArchive = pShaderArchive;
        }
#if NW_G3D_CONFIG_USE_HOSTIO
        if (strstr(*path, "demo_branch"))
        {
            s_pBranchShader = pShaderArchive;
        }
#endif
    }
    for (int idxShader = 0, numShader = s_Holder.shaders.Size();
        idxShader < numShader; ++idxShader)
    {
        nw::g3d::ResShaderArchive* pShaderArchive = s_Holder.shaders[idxShader];
        s_pDebugShader = pShaderArchive->GetShadingModel("debug");
        if (s_pDebugShader)
        {
            s_OutputOptionIndex = s_pDebugShader->GetDynamicOptionIndex("output");
            s_pOutputOption = s_pDebugShader->GetDynamicOption("output");
            NW_G3D_ASSERT_NOT_NULL(s_pOutputOption);
            s_ShaderIndex = s_pOutputOption->GetDefaultIndex();
            break;
        }
    }
    NW_G3D_ASSERT_NOT_NULL(s_pDebugShader);
    NW_G3D_ASSERT_NOT_NULL(s_pBranchShader);

    // ジオメトリシェーダ用にリングバッファの設定を行います。
    nw::g3d::GfxRingBuffer ringBuffer;
    {
        size_t inputSize = nw::g3d::GfxRingBuffer::CalcInputBufferSize(maxInputRingItem);
        if (inputSize > 0)
        {
            ringBuffer.SetInputBuffer(g3ddemo::AllocMem1(inputSize, GX2_SHADER_ALIGNMENT), inputSize);
        }
        size_t outputSize = nw::g3d::GfxRingBuffer::CalcOutputBufferSize(maxOutputRingItem);
        if (outputSize > 0)
        {
            ringBuffer.SetOutputBuffer(g3ddemo::AllocMem1(outputSize, GX2_SHADER_ALIGNMENT), outputSize);
        }
        ringBuffer.Load();
    }

    // リソースファイルの初期化。
    for (const char** path = BFRES_PATH; *path != NULL; ++path)
    {
        // ResFile のアライメントは内部に持っているテクスチャの最大アライメントに一致します。
        size_t size = 0;
        void* pFile = g3ddemo::LoadFile(path[0], &size, TEXTURE_ALIGNMENT);
        NW_G3D_ASSERT(nw::g3d::ResFile::IsValid(pFile));
        nw::g3d::ResFile* pResFile = nw::g3d::ResFile::ResCast(pFile);
        pResFile->Setup();
        // ResFile 内のテクスチャや頂点を CPU で書き込んだ場合は
        // CPU のキャッシュを吐き出す必要があります。
        s_Holder.files.PushBack(pResFile);
    }

    // アニメーションリソースの登録
    g3ddemo::RegistAnim(&s_Holder);

    // モデルリソースとシェーダの関連付け。
    for (int idxFile = 0, numFile = s_Holder.files.Size(); idxFile < numFile; ++idxFile)
    {
        nw::g3d::ResFile* pResFile = s_Holder.files[idxFile];
        for (int idxModel = 0, numModel = pResFile->GetModelCount();
            idxModel < numModel; ++idxModel)
        {
            nw::g3d::ResModel* pResModel = pResFile->GetModel(idxModel);

            // シェーダ割り当ての構築。
            g3ddemo::ModelAssign* pModelAssign = SetupModelAssign(pResModel);
            s_Holder.assigns.PushBack(pModelAssign);

            // モデルインスタンスの構築。
            g3ddemo::CreateModelObjArg arg(pResModel);
            arg.initArg.ViewCount(NUM_VIEW);
            arg.initArg.EnableBounding();
            nw::g3d::ModelObj* pModelObj = g3ddemo::CreateModelObj(arg);
            g3ddemo::UserModel* pUserModel = SetupUserModel(pModelObj);
#if NW_G3D_CONFIG_USE_HOSTIO
            pUserModel->EnableEditCalcSkeletalAnimations();
#endif
            s_Holder.models.PushBack(pUserModel);

            if (0 == strcmp("env", pResModel->GetName()))
            {
                s_Holder.pEnvModel = pUserModel;
            }
        }
    }

    // カメラ初期化。
    {
        s_UseCamera = false;
        s_CameraPosition.Set(0.0f, 2.5f, 7.5f);
        s_CameraUp.Set(0.0f, 1.0f, 0.0f);
        s_CameraTarget.Set(0.0f, 2.5f, 0.0f);

        s_View.GetViewMtx(0).LookAt(
            s_CameraPosition,
            s_CameraUp,
            s_CameraTarget);
    }

    bool restart = false;

    // メインループ
    while (g3ddemo::ProcessMsg())
    {
        pCtx->TempPrepare(); // 更新に GL コンテクストが必要。
        int idxFrame = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_FRAME);
        if (!pad.Read() || pad.IsTriggered(g3ddemo::Pad::BUTTON_START))
        {
            if (pad.IsHold(g3ddemo::Pad::TRIGGER_R))
            {
                restart = true;
            }
            g3ddemo::PostQuitMsg();
        }

        CalcMenuCPU(pad);


        // 計算
        {
            int idxCalcCPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU);
            pCtx->TempPrepare();

            // カメラコントロール
            if (s_UseCamera)
            {
                ControllCamera(&s_CameraPosition, &s_CameraUp, &s_CameraTarget, &s_View.GetViewMtx(0), &pad);
                if (pad.IsHold(g3ddemo::Pad::TRIGGER_Z))
                {
                    g3ddemo::LookAtModel(&s_CameraPosition, &s_CameraTarget,
                        s_Holder.models[s_ModelIndex]->GetModelObj());
                }
                s_View.GetViewMtx(0).LookAt(s_CameraPosition, s_CameraUp, s_CameraTarget);
            }

            // 投影行列。
            s_View.GetProjMtx(0).Perspective(
                nw::g3d::Math::DegToRad(45.0f), 16.0f / 9.0f, 0.01f, 1000.0f);

            // 環境マップ用行列。
            s_TexMtxEnv.LookAt(
                nw::g3d::Vec3::Make(0.0f, 2.5f, 7.5f),
                nw::g3d::Vec3::Make(0.0f, 1.0f, 0.0f),
                nw::g3d::Vec3::Make(0.0f, 2.5f, 0.0f));
            s_TexMtxEnv.SetT(nw::g3d::Vec3::Make(0.0f, 0.0f, 0.0f));

            // 投影マップ用行列。
            s_TexMtxProj.TexProjPerspective(nw::g3d::Math::DegToRad(45.0f), 1.0f);
            s_TexMtxProj.Mul(s_TexMtxProj, s_View.GetViewMtx(0));

            if (!s_Holder.models.Empty())
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
                pUserModel->CalcWorld();
                pUserModel->CalcAnim();
                pUserModel->UpdateFrame();
#if NW_G3D_CONFIG_USE_HOSTIO
                nw::g3d::edit::EditServer::Instance().CalcAnimations();
#endif

                pUserModel->UpdateShader(); // 更新に GL コンテクストが必要。
            }
            if (s_Holder.pEnvModel)
            {
                s_Holder.pEnvModel->UpdateShader();
            }
            s_Meter.EndTimeSpan(idxCalcCPU);
        }

        // 残っているコマンドの完了を待ちます。
        //nw::g3d::GfxManage::DrawDone(); // 直列実行では不要

        // GPU 待ちが必要な計算
        {
            int idxCalcGPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU);
            pCtx->TempPrepare(); // 更新に GL コンテクストが必要。

            if (!s_Holder.models.Empty())
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
                pUserModel->CalcGPU();

                nw::g3d::ModelObj* pModelObj = pUserModel->GetModelObj();
                for (int idxShape = 0, numShape = pModelObj->GetShapeCount();
                    idxShape < numShape; ++idxShape)
                {
                    nw::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(idxShape);
                    g3ddemo::UserShape* pUserShape = pShapeObj->GetUserPtr<g3ddemo::UserShape>();
                    pUserShape->GetShaderSelector(g3ddemo::PASS_DEBUG)->WriteDynamicKey(
                        s_OutputOptionIndex, s_ShaderIndex);
                }
            }

            // ビュー毎の更新。
            for (int idxView = 0; idxView < NUM_VIEW; ++idxView)
            {
                s_View.Calc(idxView);
                if (!s_Holder.models.Empty())
                {
                    g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
                    pUserModel->CalcViewGPU(idxView);
                }
            }

            // 環境
            if (s_Holder.pEnvModel)
            {
                s_Holder.pEnvModel->CalcGPU();
            }

            s_Meter.EndTimeSpan(idxCalcGPU);
            // 情報の表示。
            CalcMenuGPU();
        }

        nw::g3d::CPUCache::Sync(); // CPU キャッシュ操作の完了を待ちます。
        // GPU のリードキャッシュはここで一括して Invalidate します。
        nw::g3d::GPUCache::InvalidateAll();

        // DRC 描画。
        {
            int idxClear = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_SYS_GPU);
            nw::g3d::ClearBuffers(&colorBufferDRC, &depthBufferDRC,
                0.3f, 0.3f, 0.3f, 1.0f, GX2_CLEAR_BOTH);
            pCtx->TempPrepare();
            s_Meter.EndTimeSpan(idxClear);

            pCtx->Activate(); // Clear に変更されたコンテクストの復帰。
            int idxDraw = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU, 0, 0xFF, 0);
            int idxGPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_GPU);
            frameBufferDRC.Load();

            GX2ShaderMode shaderMode = GX2_SHADER_MODE_UNIFORM_BLOCK;
            nw::g3d::SetShaderMode(shaderMode);

            if (!s_Holder.models.Empty())
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
                pUserModel->DebugDraw(g3ddemo::PASS_DEBUG, 0, shaderMode);
            }
            s_Meter.EndTimeSpan(idxGPU);
            s_Meter.EndTimeSpan(idxDraw);

            int idxCopyOut = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_SYS_GPU);
            g3ddemo::CopyOut(&colorBufferDRC, GX2_SCAN_TARGET_DRC_FIRST);
            pCtx->TempPrepare();
            s_Meter.EndTimeSpan(idxCopyOut);
        }

        // TV 描画。
        {
            int idxClear = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_SYS_GPU);
            nw::g3d::ClearBuffers(&colorBuffer, &depthBuffer,
                0.5f, 0.5f, 0.5f, 1.0f, GX2_CLEAR_BOTH);
            pCtx->TempPrepare();
            s_Meter.EndTimeSpan(idxClear);

            pCtx->Activate(); // Clear に変更されたコンテクストの復帰。
            int idxDraw = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_CPU, 0, 0xFF, 0);
            int idxGPU = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_GPU);
            frameBuffer.Load();

            GX2ShaderMode shaderMode = GX2_SHADER_MODE_UNIFORM_BLOCK;
            nw::g3d::SetShaderMode(shaderMode);

            if (!s_Holder.models.Empty())
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
                if (pUserModel->IsStreamOutEnabled())
                {
                    // ストリームアウトによって座標変換を行います。
                    nw::g3d::SetStreamOutEnable(GX2_TRUE);
                    nw::g3d::SetRasterizerClipControl(GX2_FALSE, GX2_TRUE);
                    pUserModel->StreamOut(shaderMode);
                    nw::g3d::SetStreamOutEnable(GX2_FALSE);
                    nw::g3d::SetRasterizerClipControl(GX2_TRUE, GX2_TRUE);
                    // ストリームアウトされた頂点のキャッシュをケアします。
                    nw::g3d::GPUCache::FlushAll();
                    nw::g3d::GPUCache::InvalidateAll();
                }
                pUserModel->DebugDraw(g3ddemo::PASS_DEFAULT, 0, shaderMode);
            }
            s_Meter.EndTimeSpan(idxGPU);
            s_Meter.EndTimeSpan(idxDraw);

            // 情報表示のためにシェーダモードを切り替えます。
            nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_BLOCK);
            g3ddemo::ScreenInfo::LoadState();
            s_Screen.Draw();
            s_Meter.Draw();

            int idxCopyOut = s_Meter.BeginTimeSpan(g3ddemo::ProcessMeter::GROUP_SYS_GPU);
            g3ddemo::CopyOut(&colorBuffer, GX2_SCAN_TARGET_TV);
            pCtx->TempPrepare();
            s_Meter.EndTimeSpan(idxCopyOut);
        }

        nw::g3d::GfxManage::DrawDone();
        s_Meter.EndTimeSpan(idxFrame);

        g3ddemo::SwapScanBuffers();

#if NW_G3D_CONFIG_USE_HOSTIO
        // 描画中のリソースを破棄してしまわないよう、描画完了後に呼び出します。
        // UnloadFile 内で描画リソースを破棄しないような遅延処理による対策も可。
        pCtx->TempPrepare();
        nw::g3d::edit::EditServer::Instance().Poll();
#endif
    }

    // グラフィックスリソースの終了処理。
    pCtx->TempPrepare();

#if NW_G3D_CONFIG_USE_HOSTIO
    {
        nw::g3d::edit::EditServer::Instance().Clear(true);
        nw::g3d::edit::EditServer::Instance().Close();
        nw::g3d::edit::EditServer::DeleteInstance();
    }
#endif

    if (void* ptr = ringBuffer.GetInputBuffer())
    {
        g3ddemo::FreeMem1(ptr);
    }
    if (void* ptr = ringBuffer.GetOutputBuffer())
    {
        g3ddemo::FreeMem1(ptr);
    }

    // エディットライブラリが所有しているモデルを破棄しないように、
    // エディットの終了処理後に残りのデータの破棄を行います。
    g3ddemo::DestroyAll(&s_Holder);

    s_View.Cleanup();

    //
    s_Meter.Cleanup();
    s_Screen.Cleanup();

    frameBuffer.Free(g3ddemo::FreeMem1);
    frameBuffer.Cleanup();
    g3ddemo::FreeMem2(frameBuffer.GetBufferPtr());

    frameBufferDRC.Free(g3ddemo::FreeMem1);
    frameBufferDRC.Cleanup();
    g3ddemo::FreeMem2(frameBufferDRC.GetBufferPtr());

    pCtx->Cleanup();
    g3ddemo::FreeMem2(pCtx);
    pCtx = NULL;

    // 終了処理。
    g3ddemo::ShutdownDisplay();
    g3ddemo::Shutdown();

    return restart ? -1 : EXIT_SUCCESS;
}

namespace {

g3ddemo::ModelAssign* SetupModelAssign(nw::g3d::ResModel* pResModel,
    nw::g3d::ResShaderArchive** pShaderArchiveArray, int ShaderArchiveCount,
    nw::g3d::ResShaderArchive* pAttachShader)
{
    g3ddemo::CreateShaderAssign(pResModel);
    g3ddemo::ModelAssign* pModelAssign = pResModel->GetUserPtr<g3ddemo::ModelAssign>();

    // モデルに記録されたデフォルトシェーダの関連付け。
    nw::g3d::BindResult result = nw::g3d::BindResult::NotBound();
    // ロードシェーダ
    if (pShaderArchiveArray)
    {
        result = pModelAssign->BindShader(
            g3ddemo::PASS_DEFAULT, pShaderArchiveArray, ShaderArchiveCount);
    }
    // アタッチシェーダ
    if (pAttachShader)
    {
        result = pModelAssign->BindShader(g3ddemo::PASS_DEFAULT, pAttachShader);
    }
    // グローバルシェーダ
    for (int idxShader = 0, numShader = s_Holder.shaders.Size();
        !result.IsComplete() && idxShader < numShader; ++idxShader)
    {
        nw::g3d::ResShaderArchive* pShaderArchive = s_Holder.shaders[idxShader];
        result = pModelAssign->BindShader(g3ddemo::PASS_DEFAULT, pShaderArchive);
    }
    //NW_G3D_ASSERT(result.IsComplete());

    // ストリームアウトシェーダの関連付け。
    if (s_Holder.pStreamoutArchive)
    {
        pModelAssign->BindStreamOutShader(s_Holder.pStreamoutArchive);
    }
    // デバッグシェーダの関連付け。
    if (s_pDebugShader)
    {
        pModelAssign->BindDebugShader(s_pDebugShader);
    }
    pModelAssign->Setup();
    return pModelAssign;
}

g3ddemo::UserModel* SetupUserModel(nw::g3d::ModelObj* pModelObj)
{
    g3ddemo::CreateUserModel(pModelObj);
    g3ddemo::UserModel* pUserModel = pModelObj->GetUserPtr<g3ddemo::UserModel>();
    pUserModel->SetView(&s_View);
    if (s_Holder.pEnvModel)
    {
        pUserModel->SetEnv(s_Holder.pEnvModel->GetModelObj());
    }

    g3ddemo::SetupMaterialsArg argSetup(pModelObj);
    argSetup.pEnvMtx = &s_TexMtxEnv;
    argSetup.pProjMtx = &s_TexMtxProj;
    g3ddemo::SetupMaterials(argSetup);

    // アニメーションインスタンスの構築。
    g3ddemo::UserModel::CreateAnimObjArg createAnimObjArg;
    if (!s_Holder.modelAnims.Empty())
    {
        createAnimObjArg.Reserve(&s_Holder.modelAnims[0], s_Holder.modelAnims.Size());
    }
    pUserModel->CreateAnimObj(createAnimObjArg);

    return pUserModel;
}

#if NW_G3D_CONFIG_USE_HOSTIO
void ChangeShader(nw::g3d::ResShaderArchive* pShader)
{
    for (int idxModel = 0, numModel = s_Holder.models.Size();
        idxModel < numModel; ++idxModel)
    {
        nw::g3d::ModelObj* pModelObj = s_Holder.models[idxModel]->GetModelObj();
        nw::g3d::ResModel* pResModel = pModelObj->GetResource();
        g3ddemo::ModelAssign* pModelAssign = pResModel->GetUserPtr<g3ddemo::ModelAssign>();
        NW_G3D_ASSERT_NOT_NULL(pModelAssign);
        pModelAssign->Cleanup();
        g3ddemo::DestroyShaderAssign(pResModel);
        for (int idxAssign = 0, numAssign = s_Holder.assigns.Size();
            idxAssign < numAssign; ++idxAssign)
        {
            if (s_Holder.assigns[idxAssign] == pModelAssign)
            {
                s_Holder.assigns.Erase(idxAssign);
                break;
            }
        }
        pModelAssign = SetupModelAssign(pResModel, NULL, 0, pShader);
        s_Holder.assigns.PushBack(pModelAssign);

        for (int idxShape = 0, numShape = pModelObj->GetShapeCount();
            idxShape < numShape; ++idxShape)
        {
            nw::g3d::ShapeObj* pShapeObj = pModelObj->GetShape(idxShape);
            g3ddemo::UserShape* pUserShape = pShapeObj->GetUserPtr<g3ddemo::UserShape>();
            pUserShape->Cleanup();
            pUserShape->Setup();
        }
    }
}
#endif

void CalcMenuCPU(const g3ddemo::Pad &pad)
{
    if (pad.IsHold(g3ddemo::Pad::TRIGGER_L) && pad.IsTriggered(g3ddemo::Pad::TRIGGER_R))
    {
        s_UseCamera = !s_UseCamera;
    }

    if (!s_UseCamera)
    {
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_DOWN))
        {
            s_ItemIndex = (s_ItemIndex + 1) % NUM_ITEM;
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_UP))
        {
            s_ItemIndex = (s_ItemIndex + NUM_ITEM - 1) % NUM_ITEM;
        }
#if NW_G3D_CONFIG_USE_HOSTIO
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_X))
        {
            nw::g3d::ResShaderArchive* pArchive = s_Holder.shaders[1];
            if (!nw::g3d::edit::EditServer::Instance().HasShaderArchive(pArchive))
            {
                nw::g3d::edit::EditServer::AttachShaderArchiveArg arg;
                arg.resShaderArchive = pArchive;
                nw::g3d::edit::EditServer::Instance().AttachShaderArchive(arg);
            }
        }
        if (pad.IsTriggered(g3ddemo::Pad::BUTTON_Y))
        {
            nw::g3d::ResShaderArchive* pArchive = s_pBranchShader;
            if (!nw::g3d::edit::EditServer::Instance().HasShaderArchive(pArchive))
            {
                nw::g3d::edit::EditServer::AttachShaderArchiveArg arg;
                arg.resShaderArchive = pArchive;
                nw::g3d::edit::EditServer::Instance().AttachShaderArchive(arg);
            }
        }
#endif
        if (s_ItemIndex == ITEM_MODEL)
        {
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_A))
            {
                // 3DEditor の編集対象として ModelObj を登録します。
                g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
                pUserModel->AttachToEdit();
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_B))
            {
                // 3DEditor の編集対象になっている ModelObj を取り外します。
                g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
                pUserModel->DetachFromEdit();
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
            {
                s_ModelIndex = (s_ModelIndex + 1) % s_Holder.models.Size();
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
            {
                int numModel = s_Holder.models.Size();
                s_ModelIndex = (s_ModelIndex + numModel - 1) % numModel;
            }
        }
        if (s_ItemIndex == ITEM_ANIM)
        {
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_A))
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
                void* pResAnim = s_Holder.modelAnims[s_AnimIndex];
                pUserModel->AddAnim(pResAnim);
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_B))
            {
                g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
                void* pResAnim = s_Holder.modelAnims[s_AnimIndex];
                if (bit32 key = pUserModel->FindAnim(pResAnim))
                {
                    pUserModel->RemoveAnim(key);
                }
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
            {
                s_AnimIndex = (s_AnimIndex + 1) % s_Holder.modelAnims.Size();
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
            {
                int numAnim = s_Holder.modelAnims.Size();
                s_AnimIndex = (s_AnimIndex + numAnim - 1) % numAnim;
            }
        }
        if (s_ItemIndex == ITEM_SHADER)
        {
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_RIGHT))
            {
                s_ShaderIndex = (s_ShaderIndex + 1) % s_pOutputOption->GetChoiceCount();
            }
            if (pad.IsTriggered(g3ddemo::Pad::BUTTON_LEFT))
            {
                int numChoice = s_pOutputOption->GetChoiceCount();
                s_ShaderIndex = (s_ShaderIndex + numChoice - 1) % numChoice;
            }
        }
    }
}

void CalcMenuGPU()
{
    float ofsX = 1.5f;
    float ofsY = 2.0f;
    float line = s_Screen.GetFontHeight();
    s_Meter.Calc();
    s_Screen.Reset();
    s_Screen.SetFontColor(0xFF, 0xFF, 0xFF);

    if (s_UseCamera)
    {
        s_Screen.PutRect(ofsX, ofsY, 20.0f, line * 8);
        s_Screen.PutString(ofsX, ofsY, "L+R:     Disable Camera");
        ofsY += line;
        s_Screen.PutString(ofsX, ofsY, "L Stick: Rotate");
        ofsY += line;
        s_Screen.PutString(ofsX, ofsY, "R Stick: Move XZ");
        ofsY += line;
        s_Screen.PutString(ofsX, ofsY, "A:       Move Y+");
        ofsY += line;
        s_Screen.PutString(ofsX, ofsY, "B:       Move Y-");
        ofsY += line;
        s_Screen.PutString(ofsX, ofsY, "X:       Zoom In");
        ofsY += line;
        s_Screen.PutString(ofsX, ofsY, "Y:       Zoom Out");
        ofsY += line;
        s_Screen.PutString(ofsX, ofsY, "ZR:      Look Model");
        ofsY += line;
    }
    else
    {
        s_Screen.PutRect(ofsX, ofsY, 20.0f, line * (NUM_ITEM + 3));
        s_Screen.PutString(ofsX, ofsY, "L+R: Enable Camera");
        ofsY += line;
        s_Screen.PutString(ofsX, ofsY, "X:   Attach Debug Shader");
        ofsY += line;
        s_Screen.PutString(ofsX, ofsY, "Y:   Attach Demo Shader");
        ofsY += line;
        s_Screen.PutString(ofsX, ofsY + s_ItemIndex * line, ">");
        if (!s_Holder.models.Empty())
        {
            g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
            nw::g3d::ModelObj* pModelObj = pUserModel->GetModelObj();
            nw::g3d::ResModel* pResModel = pModelObj->GetResource();
            s_Screen.PutStringFmt(ofsX, ofsY, "  [%d/%d] %c %s.fmdb",
                s_ModelIndex + 1, s_Holder.models.Size(), pUserModel->IsAttachedToEdit() ? '+' : '-',
                pResModel->GetName());
        }
        else
        {
            s_Screen.PutStringFmt(ofsX, ofsY, "  [0/0] No model");
        }
        ofsY += line;
        if (!s_Holder.modelAnims.Empty())
        {
            void* ptr = s_Holder.modelAnims[s_AnimIndex];
            nw::g3d::BinaryBlockHeader* pHeader = static_cast<nw::g3d::BinaryBlockHeader*>(ptr);
            const char* name = "";
            const char* ext = "";
            switch (pHeader->sigWord)
            {
            case nw::g3d::ResSkeletalAnim::SIGNATURE:
                {
                    nw::g3d::ResSkeletalAnim* pAnim = static_cast<nw::g3d::ResSkeletalAnim*>(ptr);
                    name = pAnim->GetName();
                    ext = "fskb";
                }
                break;
            case nw::g3d::ResVisibilityAnim::SIGNATURE:
                {
                    nw::g3d::ResVisibilityAnim* pAnim = static_cast<nw::g3d::ResVisibilityAnim*>(ptr);
                    name = pAnim->GetName();
                    ext = pAnim->GetVisibilityType() == nw::g3d::ResVisibilityAnim::BONE_VISIBILITY ?
                        "fvbb" : "fvmb";
                }
                break;
            case nw::g3d::ResShaderParamAnim::SIGNATURE:
                {
                    nw::g3d::ResShaderParamAnim* pAnim = static_cast<nw::g3d::ResShaderParamAnim*>(ptr);
                    name = pAnim->GetName();
                    ext = "fspb";
                }
                break;
            case nw::g3d::ResTexPatternAnim::SIGNATURE:
                {
                    nw::g3d::ResTexPatternAnim* pAnim = static_cast<nw::g3d::ResTexPatternAnim*>(ptr);
                    name = pAnim->GetName();
                    ext = "ftpb";
                }
                break;
            case nw::g3d::ResShapeAnim::SIGNATURE:
                {
                    nw::g3d::ResShapeAnim* pAnim = static_cast<nw::g3d::ResShapeAnim*>(ptr);
                    name = pAnim->GetName();
                    ext = "fshb";
                }
                break;
            }
            char enable = s_Holder.models[s_ModelIndex]->FindAnim(ptr) == 0 ? '-' : '+';
            s_Screen.PutStringFmt(ofsX, ofsY, "  [%d/%d] %c %s.%s",
                s_AnimIndex + 1, s_Holder.modelAnims.Size(), enable, name, ext);
        }
        else
        {
            s_Screen.PutStringFmt(ofsX, ofsY, "  [0/0] No animation");
        }
        ofsY += line;
        s_Screen.PutStringFmt(ofsX, ofsY, "  [%d/%d] %s", s_ShaderIndex + 1,
            s_pOutputOption->GetChoiceCount(), s_pOutputOption->GetChoiceName(s_ShaderIndex));
        ofsY += line;
    }

    if (!s_Holder.models.Empty())
    {
        g3ddemo::UserModel* pUserModel = s_Holder.models[s_ModelIndex];
        if (pUserModel->IsStreamOutEnabled())
        {
            s_Screen.SetFontColor(0xFF, 0, 0);
            s_Screen.PutString(ofsX, ofsY, "<StreamOut in use>");
            ofsY += line;
        }
    }

    s_Screen.DCFlush();
}

} // anonymous namespace
