﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_DEMO_USERMODEL_H_
#define NW_G3D_DEMO_USERMODEL_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GfxObject.h>
#include <nw/g3d/math/g3d_Matrix34.h>
#include <nw/g3d/math/g3d_Matrix44.h>
#include <nw/g3d/res/g3d_Binding.h>
#include <nw/g3d/g3d_ShapeObj.h>
#include <nw/g3d/g3d_SkeletalAnimObj.h>
#include <nw/g3d/g3d_VisibilityAnimObj.h>
#include <nw/g3d/g3d_ShaderParamAnimObj.h>
#include <nw/g3d/g3d_TexPatternAnimObj.h>
#include <nw/g3d/g3d_ShapeAnimObj.h>
#include <nw/g3d/g3d_ShaderObj.h>
#include <nw/g3d/g3d_edit.h>
#include <g3ddemo_DemoUtility.h>

namespace nw { namespace g3d { namespace res {

class ResShaderArchive;
class ResShadingModel;
class ResShaderProgram;
class ResModel;
class ResMaterial;
class ResSampler;
class ResShaderAssign;
class ResVertex;
class ResVtxAttrib;
class ResBuffer;

}}} // namespace nw::g3d::res

namespace nw { namespace g3d {

class ModelObj;
class MaterialObj;
class ShapeObj;

}} // namespace nw::g3d

namespace nw { namespace g3d { namespace demo {

enum PassType
{
    PASS_DEFAULT,
    PASS_DEBUG,
    PASS_MATERIAL_PICK,
    PASS_OUTLINE,
    PASS_PREZ,
    PASS_SHADOW,
    PASS_OPAQUE,
    PASS_TRANSLUCENT,
    PASS_GEOMETRY,
    PASS_WATER,
    NUM_PASS
};

enum StreamType
{
    STREAM_P,
    STREAM_PN,
    STREAM_PNT,
    NUM_STREAM
};

enum BlockType
{
    BLOCK_VIEW,
    BLOCK_ENV,
    BLOCK_OUTLINE,
    BLOCK_CONTEXT,
    NUM_BLOCK
};

enum
{
    NUM_VIEW                = 6,
    NUM_SKELETAL_ANIM       = 4,
    NUM_VISIBILITY_ANIM     = 4,
    NUM_SHADERPARAM_ANIM    = 4,
    NUM_TEXPATTERN_ANIM     = 4,
    NUM_SHAPE_ANIM          = 4
};

//--------------------------------------------------------------------------------------------------

class MaterialAssign
{
public:
    enum { MAX_SAMPLER = 16, MAX_UNIFORM_BLOCK = 16 };

    struct Pass
    {
        nw::g3d::ShadingModelObj* pShadingModelObj;
        nw::g3d::ResShadingModel* pShadingModel;
        const nw::g3d::ResSampler* pSamplerArray[MAX_SAMPLER];
        s8 blockIndexArray[NUM_BLOCK];
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    explicit MaterialAssign(nw::g3d::ResMaterial* pResMaterial);

    nw::g3d::BindResult BindShader(PassType type, nw::g3d::ResShadingModel* pShadingModel);
    void BindDebugShader(nw::g3d::ResShadingModel* pShadingModel);

    void Setup();
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    nw::g3d::ResMaterial* GetResMaterial() { return m_pResMaterial; }
    const nw::g3d::ResMaterial* GetResMaterial() const { return m_pResMaterial; }

    Pass& GetPass(PassType type)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(type, NUM_PASS);
        return m_PassArray[type];
    }

    const Pass& GetPass(PassType type) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(type, NUM_PASS);
        return m_PassArray[type];
    }

    //@}

private:
    nw::g3d::ResMaterial* m_pResMaterial;
    Pass m_PassArray[NUM_PASS];

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(MaterialAssign);
};

//--------------------------------------------------------------------------------------------------

class ShapeAssign
{
public:
    enum { MAX_ATTRIB = 16, MAX_BUFFER = 16 };

    struct Pass
    {
        nw::g3d::ShadingModelObj* pShadingModelObj;
        nw::g3d::ResShadingModel* pShadingModel;
        const nw::g3d::ResVtxAttrib* pVtxAttribArray[MAX_ATTRIB];
        const nw::g3d::ResBuffer* pVtxBufferArray[MAX_BUFFER];
        int vtxBufferCount;
        nw::g3d::GfxFetchShader fetchShader;
        nw::g3d::GfxFetchShader fetchShaderSO; // デモ用に切り替えたいので無駄を承知で二重持ちする
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    ShapeAssign(nw::g3d::ResShape* pResShape, nw::g3d::ResMaterial* pResMaterial);

    void BindShader(PassType type, nw::g3d::ResShadingModel* pShadingModel);
    void BindStreamOutShader(nw::g3d::ResShaderArchive* pShaderArchive);
    void BindDebugShader(nw::g3d::ResShadingModel* pShadingModel);

    void Setup();
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    nw::g3d::ResShape* GetResShape() { return m_pResShape; }
    const nw::g3d::ResShape* GetResShape() const { return m_pResShape; }

    Pass& GetPass(PassType type)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(type, NUM_PASS);
        return m_PassArray[type];
    }

    const Pass& GetPass(PassType type) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(type, NUM_PASS);
        return m_PassArray[type];
    }

    StreamType GetStreamType() const { return m_StreamType; }

    nw::g3d::ResShadingModel* GetStreamOutShader() { return m_pStreamOutShader; }
    const nw::g3d::ResShadingModel* GetStreamOutShader() const { return m_pStreamOutShader; }

    //@}

private:
    nw::g3d::ResShape* m_pResShape;
    nw::g3d::ResMaterial* m_pResMaterial;
    Pass m_PassArray[NUM_PASS];
    StreamType m_StreamType;
    nw::g3d::ResShadingModel* m_pStreamOutShader;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(ShapeAssign);
};

//--------------------------------------------------------------------------------------------------

class ModelAssign
{
public:

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    ModelAssign(nw::g3d::ResModel* pResModel);

    nw::g3d::BindResult BindShader(PassType type, nw::g3d::ResShaderArchive* pShaderArchive);
    nw::g3d::BindResult BindShader(PassType type, nw::g3d::ResShadingModel* pShadingModel);
    nw::g3d::BindResult BindShader(PassType type,
        nw::g3d::ResShaderArchive** ppShaderArchives, int ShaderArchiveCount);
    void BindStreamOutShader(nw::g3d::ResShaderArchive* pShaderArchive);
    void BindDebugShader(nw::g3d::ResShadingModel* pShadingModel);

    void Setup();
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    nw::g3d::ResModel* GetResModel() { return m_pResModel; }
    const nw::g3d::ResModel* GetResModel() const { return m_pResModel; }

    nw::g3d::ResShaderArchive* GetModelShader() { return m_pModelShader; }
    const nw::g3d::ResShaderArchive* GetModelShader() const { return m_pModelShader; }
    void SetModelShader(nw::g3d::ResShaderArchive* pModelShader) { m_pModelShader = pModelShader; }

    //@}

private:
    nw::g3d::ResModel* m_pResModel;
    nw::g3d::ResShaderArchive* m_pModelShader;

    void BindShaderShape(PassType type);

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(ModelAssign);
};

//--------------------------------------------------------------------------------------------------

class UserView
{
public:

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    UserView();

    void Setup();
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    nw::g3d::Mtx44& GetProjMtx(int viewIndex) { return m_View[viewIndex].projMtx; }
    const nw::g3d::Mtx44& GetProjMtx(int viewIndex) const { return m_View[viewIndex].projMtx; }
    nw::g3d::Mtx34& GetViewMtx(int viewIndex) { return m_View[viewIndex].viewMtx; }
    const nw::g3d::Mtx34& GetViewMtx(int viewIndex) const { return m_View[viewIndex].viewMtx; }
    nw::g3d::ViewVolume& GetViewVolume(int viewIndex) { return m_ViewVolume[viewIndex]; }
    const nw::g3d::ViewVolume& GetViewVolume(int viewIndex) const { return m_ViewVolume[viewIndex]; }

    const nw::g3d::GfxBuffer& GetViewBlock(int viewIndex) const { return m_ViewBlock[viewIndex]; }

    nw::g3d::Mtx34& GetInvViewMtx(int viewIndex) { return m_View[viewIndex].invViewMtx; }
    const nw::g3d::Mtx34& GetInvViewMtx(int viewIndex) const { return m_View[viewIndex].invViewMtx; }

    //@}

    void Calc(int viewIndex);

private:
    nw::g3d::GfxBuffer m_ViewBlock[NUM_VIEW];
    struct View
    {
        nw::g3d::Mtx44 projMtx;
        nw::g3d::Mtx34 viewMtx;

        nw::g3d::Mtx34 invViewMtx;
    } m_View[NUM_VIEW];
    nw::g3d::ViewVolume m_ViewVolume[NUM_VIEW];

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(UserView);
};

//--------------------------------------------------------------------------------------------------

class UserMaterial
{
public:

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    UserMaterial(MaterialObj* pMaterialObj);

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    MaterialObj* GetMaterialObj() { return m_pMaterialObj; }
    const MaterialObj* GetMaterialObj() const { return m_pMaterialObj; }

    bool GetRenderShadow() const { return m_RenderShadow; }
    void SetRenderShadow(bool value) { m_RenderShadow = value; }

    //@}

private:
    MaterialObj* m_pMaterialObj;

    bool m_RenderShadow;

    // TODO: RenderInfo etc

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(UserMaterial);
};

//--------------------------------------------------------------------------------------------------

class UserShape
{
public:
    enum { MAX_ATTRIB = 16, MAX_BUFFER = 16, MAX_ACTIVE_KEY_SHAPE = 8 };

    struct StreamOutPass
    {
        nw::g3d::ShaderSelector* pShaderSelector;
        nw::g3d::GfxBuffer streamOutBuffer;
        nw::g3d::GfxStreamOut streamOut;
        nw::g3d::GfxFetchShader fetchShader;
        u8 vtxAttribCount;
        u8 vtxBufferCount;
        int validBlendCount;
        const nw::g3d::ResVtxAttrib* pVtxAttribArray[MAX_ATTRIB];
        const nw::g3d::ResBuffer* pVtxBufferArray[MAX_BUFFER];
        s8 bufferSlotArray[MAX_BUFFER];
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    UserShape(ShapeObj* pShapeObj);
    ~UserShape();

    void Setup();
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    void UpdateShader();

    void UpdateFetchShader();

    void UpdateBlendWeight(int viewIndex);

    void UpdateShapeAnim();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    nw::g3d::ShapeObj* GetShapeObj() { return m_pShapeObj; }
    const nw::g3d::ShapeObj* GetShapeObj() const { return m_pShapeObj; }

    nw::g3d::ShaderSelector* GetShaderSelector(PassType type)
    {
        return m_pShaderSelectorArray[type];
    }
    const nw::g3d::ShaderSelector* GetShaderSelector(PassType type) const
    {
        return m_pShaderSelectorArray[type];
    }

    StreamOutPass& GetStreamOutPass() { return m_StreamOutPass; }
    const StreamOutPass& GetStreamOutPass() const { return m_StreamOutPass; }

    //@}

protected:

    nw::g3d::ShaderSelector* SetupShaderSelector(nw::g3d::ShadingModelObj* pShadingModelObj);

private:
    nw::g3d::ShapeObj* m_pShapeObj;
    nw::g3d::ShaderSelector* m_pShaderSelectorArray[NUM_PASS];

    StreamOutPass m_StreamOutPass;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(UserShape);
};

//--------------------------------------------------------------------------------------------------

class IUserContext
{
public:
    //----------------------------------------
    //! @name 取得/設定
    //@{
    virtual const nw::g3d::GfxBuffer& GetContextBlock() const = 0;
    //@}
};

//--------------------------------------------------------------------------------------------------

class UserModel
{
public:
    enum AnimKey
    {
        ANIM_TYPE_SKELETAL      = 0x00010000,
        ANIM_TYPE_VISIBILITY    = 0x00020000,
        ANIM_TYPE_SHADERPARAM   = 0x00030000,
        ANIM_TYPE_TEXPATTERN    = 0x00040000,
        ANIM_TYPE_SHAPE         = 0x00050000,
        MASK_ANIM_TYPE          = 0xFFFF0000,
        MASK_ANIM_SLOT          = 0x0000FFFF
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    UserModel(ModelObj* pModelObj);

    struct CreateAnimObjArg
    {
        CreateAnimObjArg() : initArgSkeletal() {}
        void Reserve(const void* const * pResAnim, int count);

        nw::g3d::SkeletalAnimObj::InitArg initArgSkeletal;
        nw::g3d::VisibilityAnimObj::InitArg initArgVisibility;
        nw::g3d::ShaderParamAnimObj::InitArg initArgShaderParam;
        nw::g3d::TexPatternAnimObj::InitArg initArgTexPattern;
        nw::g3d::ShapeAnimObj::InitArg initArgShape;
    };

    void CreateAnimObj(const CreateAnimObjArg& arg);
    void DestroyAnimObj();

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    ModelObj* GetModelObj() { return m_pModelObj; }
    const ModelObj* GetModelObj() const { return m_pModelObj; }

    void SetView(const UserView* pView) { m_pView = pView; }

    void SetEnv(const nw::g3d::ModelObj* pEnvModel) { m_pEnvModel = pEnvModel; }

    void SetOutline(const nw::g3d::ModelObj* pOutlineModel) { m_pOutlineModel = pOutlineModel; }

    void BindTo(const nw::g3d::ModelObj* pModelObj, int boneIndex)
    {
        m_pBindModel = pModelObj;
        m_BindBoneIndex = boneIndex;
    }

    nw::g3d::Vec3& GetScale() { return m_Scale; }
    const nw::g3d::Vec3& GetScale() const { return m_Scale; }

    nw::g3d::Mtx34& GetLayoutMtx() { return m_LayoutMtx; }
    const nw::g3d::Mtx34& GetLayoutMtx() const { return m_LayoutMtx; }

    const nw::g3d::Mtx34& GetBaseMtx() const { return m_BaseMtx; }

    void EnableStreamOut();
    void DisableStreamOut();
    bool IsStreamOutEnabled() const { return m_StreamOutEnable; }

    void EnableLockedCache() { m_LockedCacheEnable = true; }
    void DisableLockedCache() { m_LockedCacheEnable = false; }
    bool IsLockedCacheEnabled() const { return m_LockedCacheEnable; }

    void AttachToEdit();
    void DetachFromEdit();
    bool IsAttachedToEdit() const;

    void EnableFrustumCulling() { m_FrustumCullingEnable = true; }
    void DisableFrustumCulling() { m_FrustumCullingEnable = false; }
    bool IsFrustumCullingEnabled() const { return m_FrustumCullingEnable; }

    void EnableDoubleBuffer() { m_DoubleBufferEnable = true; }
    void DisableDoubleBuffer() { m_DoubleBufferEnable = false; }
    bool IsDoubleBuffer() const { return m_DoubleBufferEnable; }

    int GetLODLevel(int viewIndex) const { return m_LODLevel[viewIndex]; }
    void SetLODLevel(int viewIndex, int lodLevel) { m_LODLevel[viewIndex] = lodLevel; }

    void SetCalcLodLevel(int idxView, nw::g3d::ICalcLodLevelFunctor* calcLod) { m_pCalcLodLevel[idxView] = calcLod; }

    void SetContext(const IUserContext* pContext) { m_pContext = pContext; }

#if NW_G3D_CONFIG_USE_HOSTIO
    void EnableEditCalcSkeletalAnimations() { m_IsEditCalcSkeletalAnimationEnabled = true; }
#endif

    //@}

    //----------------------------------------
    //! @name アニメーション
    //@{

    static bit32 MakeAnimKey(int animType, int animIndex)
    {
        return static_cast<bit32>(animType) | static_cast<bit32>(animIndex);
    }
    static bool IsAnimKeyValid(bit32 key);
    static int GetAnimType(bit32 key) { return key & static_cast<bit32>(MASK_ANIM_TYPE); }
    static int GetAnimIndex(bit32 key) { return key & static_cast<bit32>(MASK_ANIM_SLOT); }

    bit32 FindAnim(void* pResAnim) const;
    bit32 AddAnim(void* pResAnim);
    void RemoveAnim(bit32 key);

    void SetWeight(bit32 key, float weight);
    float GetWeight(bit32 key) const;

    nw::g3d::ModelAnimObj* GetAnimObj(bit32 key);
    nw::g3d::SkeletalAnimBlender& GetBlender() { return m_SkelBlender; }

    //@}

    //----------------------------------------
    //! @name 計算
    //@{

    void UpdateShader();
    void UpdateBlendWeight();

    void UpdateFrame();
    void SetFrame(float frame);
    void CalcAnim();
    void CalcWorld();
    void CalcGPU();
    void CalcViewGPU(int viewIndex);

    //@}

    //----------------------------------------
    //! @name 描画
    //@{

    void DebugDraw(PassType passType, int viewIndex, GX2ShaderMode shaderMode);

    void StreamOut(GX2ShaderMode shaderMode);

    //@}

private:
    ModelObj* m_pModelObj;

    const UserView* m_pView;
    const nw::g3d::ModelObj* m_pEnvModel;
    const nw::g3d::ModelObj* m_pOutlineModel;
    const nw::g3d::ModelObj* m_pBindModel;
    int m_BindBoneIndex;
    nw::g3d::Vec3 m_Scale;
    nw::g3d::Mtx34 m_LayoutMtx;
    nw::g3d::Mtx34 m_BaseMtx;
    int m_CurBuffer;

    int m_LODLevel[NUM_VIEW];

    nw::g3d::SkeletalAnimBlender m_SkelBlender;

    float m_Weights[NUM_SKELETAL_ANIM];
    nw::g3d::SkeletalAnimObj* m_pSkeletalAnims[NUM_SKELETAL_ANIM];
    nw::g3d::VisibilityAnimObj* m_pVisibilityAnims[NUM_VISIBILITY_ANIM];
    nw::g3d::ShaderParamAnimObj* m_pShaderParamAnims[NUM_SHADERPARAM_ANIM];
    nw::g3d::TexPatternAnimObj* m_pTexPatternAnims[NUM_TEXPATTERN_ANIM];
    nw::g3d::ShapeAnimObj* m_pShapeAnims[NUM_SHAPE_ANIM];

    const IUserContext* m_pContext;

    bool m_StreamOutEnable;
    bool m_LockedCacheEnable;
    bool m_FrustumCullingEnable;
    bool m_DoubleBufferEnable;

#if NW_G3D_CONFIG_USE_HOSTIO
    bool m_IsEditCalcSkeletalAnimationEnabled;
#endif

    nw::g3d::ICalcLodLevelFunctor* m_pCalcLodLevel[NUM_VIEW];

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(UserModel);
};

//--------------------------------------------------------------------------------------------------

class UserShadingModel
{
public:
    enum { MAX_SAMPLER = MaterialAssign::MAX_SAMPLER };

    UserShadingModel(ResShadingModel* pShadingModel, const nw::g3d::ModelObj* pEnvModel);

    ResShadingModel* GetShadingModel() { return m_pResShadingModel; }
    const ResShadingModel* GetShadingModel() const { return m_pResShadingModel; }

    int* GetEnvSamplerIndices() { return m_EnvSamplerIndices; }
    const int* GetEnvSamplerIndices() const { return m_EnvSamplerIndices; }

private:
    ResShadingModel* m_pResShadingModel;

    int m_EnvSamplerIndices[MAX_SAMPLER];
};

}}} // namespace nw::g3d::demo

#endif // NW_G3D_DEMO_USERMODEL_H_
