﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_DEMO_MODELUTILITY_H_
#define NW_G3D_DEMO_MODELUTILITY_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/g3d_ModelObj.h>
#include <nw/g3d/g3d_SkeletalAnimObj.h>
#include <nw/g3d/g3d_VisibilityAnimObj.h>
#include <nw/g3d/g3d_ShaderParamAnimObj.h>
#include <nw/g3d/g3d_TexPatternAnimObj.h>
#include <nw/g3d/g3d_ShapeAnimObj.h>
#include <g3ddemo_DemoUtility.h>
#include <g3ddemo_UserModel.h>

namespace nw { namespace g3d { namespace res {

class ResShaderArchive;

}}} // namespace nw::g3d::res

namespace nw { namespace g3d { namespace demo {

//--------------------------------------------------------------------------------------------------

//! @brief リソースに対するユーザ領域である ShaderAssign を構築します。
void CreateShaderAssign(nw::g3d::ResModel* pResModel);

//! @brief リソースに対するユーザ領域である ShaderAssign を破棄します。
void DestroyShaderAssign(nw::g3d::ResModel* pResModel);

//! @brief インスタンスに対するユーザ領域である UserModel を構築します。
void CreateUserModel(nw::g3d::ModelObj* pModelObj);

//! @brief インスタンスに対するユーザ領域である UserModel を破棄します。
void DestroyUserModel(nw::g3d::ModelObj* pModelObj);

//! @brief リソースに対するユーザ領域である UserShadingModel を構築します。
void CreateUserShadingModel(nw::g3d::ResShadingModel* pResShadingModel, nw::g3d::ModelObj* pEnvObj);

//! @brief リソースに対するユーザ領域である UserShadingModel を破棄します。
void DestroyUserShadingModel(nw::g3d::ResShadingModel* pResShadingModel);

//--------------------------------------------------------------------------------------------------

struct CreateModelObjArg
{
    CreateModelObjArg(nw::g3d::ResModel* pResModel) : initArg(pResModel) {}

    nw::g3d::ModelObj::InitArg initArg;
};

//! @brief 引数に基づいてモデルを初期化します。
nw::g3d::ModelObj* CreateModelObj(const CreateModelObjArg& arg);

//! @brief モデルインスタンスを破棄します。
void DestroyModelObj(nw::g3d::ModelObj* pModelObj);

//! @brief 引数に基づいてスケルタルアニメを初期化します。
nw::g3d::SkeletalAnimObj* CreateSkeletalAnimObj(const SkeletalAnimObj::InitArg& arg);

//! @brief 引数に基づいてビジビリティアニメを初期化します。
nw::g3d::VisibilityAnimObj* CreateVisibilityAnimObj(const VisibilityAnimObj::InitArg& arg);

//! @brief 引数に基づいてシェーダパラメータアニメを初期化します。
nw::g3d::ShaderParamAnimObj* CreateShaderParamAnimObj(const ShaderParamAnimObj::InitArg& arg);

//! @brief 引数に基づいてテクスチャパターンアニメを初期化します。
nw::g3d::TexPatternAnimObj* CreateTexPatternAnimObj(const TexPatternAnimObj::InitArg& arg);

//! @brief 引数に基づいてシェイプアニメを初期化します。
nw::g3d::ShapeAnimObj* CreateShapeAnimObj(const ShapeAnimObj::InitArg& arg);

//! @brief アニメインスタンスを破棄します。
void DestroyAnimObj(nw::g3d::AnimObj* pAnimObj);

//! @brief 引数に基づいてシェーディングモデルを初期化します。
nw::g3d::ShadingModelObj* CreateShadingModelObj(const ShadingModelObj::InitArg& arg);

//! @brief シェーディングモデルインスタンスを破棄します。
void DestroyShadingModelObj(nw::g3d::ShadingModelObj* pShadingModelObj);

//! @brief 引数に基づいてシェーダセレクタを初期化します。
nw::g3d::ShaderSelector* CreateShaderSelector(const ShaderSelector::InitArg& arg);

//! @brief シェーダセレクタをを破棄します。
void DestroyShaderSelector(nw::g3d::ShaderSelector* pShaderSelector);

//--------------------------------------------------------------------------------------------------

//! @brief デモで使用するリソースを保持するためのユーティリティです。
class ResourceHolder
{
public:
    Vector<nw::g3d::ResShaderArchive*> shaders;
    Vector<nw::g3d::ResFile*> files;
    Vector<ModelAssign*> assigns;
    Vector<UserModel*> models;
    Vector<void*> modelAnims;

    nw::g3d::ResShaderArchive* pStreamoutArchive;
    UserModel* pEnvModel;
    UserModel* pOutlineModel;

    struct InitArg
    {
        InitArg()
            : maxShaders(128)
            , maxFiles(128)
            , maxAssigns(256)
            , maxModels(1024)
            , maxModelAnims(2048)
        {
        }

        int maxShaders;
        int maxFiles;
        int maxAssigns;
        int maxModels;
        int maxModelAnims;
    };

    void Init(const InitArg& arg = InitArg())
    {
        size_t shaderBufSize = sizeof(nw::g3d::ResShaderArchive*) * arg.maxShaders;
        shaders.SetBuffer(AllocMem2(shaderBufSize), shaderBufSize);
        size_t fileBufSize = sizeof(nw::g3d::ResFile*) * arg.maxFiles;
        files.SetBuffer(AllocMem2(fileBufSize), fileBufSize);
        size_t assignBufSize = sizeof(ModelAssign*) * arg.maxAssigns;
        assigns.SetBuffer(AllocMem2(assignBufSize), assignBufSize);
        size_t modelBufSize = sizeof(UserModel*) * arg.maxModels;
        models.SetBuffer(AllocMem2(modelBufSize), modelBufSize);
        size_t modelAnimBufSize = sizeof(void*) * arg.maxModelAnims;
        modelAnims.SetBuffer(AllocMem2(modelAnimBufSize), modelAnimBufSize);

        pStreamoutArchive = NULL;
        pEnvModel = NULL;
        pOutlineModel = NULL;
    }

    void Shutdown()
    {
        if (void* pBuffer = shaders.GetBuffer())
        {
            FreeMem2(pBuffer);
        }
        if (void* pBuffer = files.GetBuffer())
        {
            FreeMem2(pBuffer);
        }
        if (void* pBuffer = assigns.GetBuffer())
        {
            FreeMem2(pBuffer);
        }
        if (void* pBuffer = models.GetBuffer())
        {
            FreeMem2(pBuffer);
        }
        if (void* pBuffer = modelAnims.GetBuffer())
        {
            FreeMem2(pBuffer);
        }

        pStreamoutArchive = NULL;
        pEnvModel = NULL;
    }
};

//! @brief リソースホルダにアニメーションを登録するユーティリティです。
//!
//! リソースホルダに登録されているリソースファイルを走査し、
//! 各リソースファイルに存在するすべてのアニメーションをリソースホルダに登録します。
//!
void RegistAnim(ResourceHolder* pHolder);

//! @brief リソースホルダに登録されているすべてのリソースを解放します。
void DestroyAll(ResourceHolder* pHolder);

//--------------------------------------------------------------------------------------------------

struct SetupMaterialsArg
{
    explicit SetupMaterialsArg(nw::g3d::ModelObj* pModelObj)
        : pModelObj(pModelObj)
        , pEnvMtx(NULL)
        , pProjMtx(NULL)
    {
    }

    nw::g3d::ModelObj* pModelObj;
    nw::g3d::Mtx34* pEnvMtx;
    nw::g3d::Mtx34* pProjMtx;
};

//! @brief マテリアルの初期化を行うユーティリティです。
void SetupMaterials(const SetupMaterialsArg& arg);

}}} // namespace nw::g3d::demo

#endif // NW_G3D_DEMO_MODELUTILITY_H_
